package pt.digitalis.dif.presentation.entities.system.difsso;

import java.io.UnsupportedEncodingException;
import java.net.URLEncoder;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;

import org.json.JSONException;
import org.json.JSONObject;

import com.google.inject.Inject;
import com.mashape.unirest.http.HttpResponse;
import com.mashape.unirest.http.JsonNode;
import com.mashape.unirest.http.Unirest;
import com.mashape.unirest.http.exceptions.UnirestException;
import com.mashape.unirest.request.HttpRequestWithBody;

import pt.digitalis.dif.controller.http.HTTPConstants;
import pt.digitalis.dif.controller.http.HTTPControllerConfiguration;
import pt.digitalis.dif.controller.interfaces.IDIFContext;
import pt.digitalis.dif.controller.interfaces.INavigationHistory;
import pt.digitalis.dif.controller.objects.Breadcrumb;
import pt.digitalis.dif.dem.annotations.entities.StageDefinition;
import pt.digitalis.dif.dem.annotations.parameter.Parameter;
import pt.digitalis.dif.dem.annotations.presentation.OnAJAX;
import pt.digitalis.dif.dem.annotations.stage.Context;
import pt.digitalis.dif.dem.annotations.stage.InjectMessages;
import pt.digitalis.dif.dem.annotations.stage.View;
import pt.digitalis.dif.dem.interfaces.IStage;
import pt.digitalis.dif.dem.managers.IDEMManager;
import pt.digitalis.dif.exception.BusinessException;
import pt.digitalis.dif.ioc.DIFIoCRegistry;
import pt.digitalis.dif.utils.http.HttpUtils;
import pt.digitalis.log.LogLevel;
import pt.digitalis.utils.common.CollectionUtils;
import pt.digitalis.utils.common.StringUtils;

/**
 * Receives an SSO request from another DIF application and returns the generated token ID
 * 
 * @author Pedro Viegas <a href="mailto:pviegas@digitalis.pt">pviegas@digitalis.pt</a><br/>
 * @created 02/09/2015
 */
@StageDefinition(name = "SSO Requester: Used to make an SSO request to another DIF application", service = "ssoservice")
@View(target = "internal/ssoRequestRedirect.jsp")
public class SSORequester {

    /**  */
    @Context
    protected IDIFContext context;

    /**  */
    @Inject
    protected IDEMManager demManager;

    /**  */
    @InjectMessages
    protected Map<String, String> messages;

    /** the server to ridirect to (with SSO) */
    @Parameter
    protected String serverURL;

    /** the stage to access */
    @Parameter
    protected String stageID;

    /** the stage parameters */
    @Parameter
    protected String stageParameters;

    /** the token ID from the remote server */
    @Parameter
    protected String tokenID;

    /**
     * @return the URL encoded parameter list
     * @throws UnsupportedEncodingException
     */
    public String getDestinationStageParametersEncoded() throws UnsupportedEncodingException
    {
        StringBuffer buffer = new StringBuffer();

        if (StringUtils.isNotBlank(stageParameters))
        {
            for (Entry<String, String> parameterPair: CollectionUtils.stringToKeyValueMap(stageParameters).entrySet())
            {
                buffer.append("&");
                buffer.append(URLEncoder.encode(parameterPair.getKey(), "UTF8"));
                buffer.append("=");
                buffer.append(URLEncoder.encode(parameterPair.getValue(), "UTF8"));
            }
        }

        return buffer.toString();
    }

    /**
     * @return the back url if the process fails
     */
    public String getPreviousStageID()
    {
        List<Breadcrumb> history = DIFIoCRegistry.getRegistry().getImplementation(INavigationHistory.class)
                .getHistoryLastAccessed();

        if (history == null || history.isEmpty())
            return HTTPControllerConfiguration.getInstance().getHomeStageID();
        else
        {
            Breadcrumb crumb = history.get(0);

            return crumb.getStageID();
        }
    }

    /**
     * @return the back url if the process fails
     */
    public String getPreviousStageName()
    {
        IStage stage = demManager.getStage(this.getPreviousStageID());

        String title = stage.getMessageForLanguage(context.getLanguage(), "title");
        if (StringUtils.isBlank(title))
            title = stage.getName();

        return title;
    }

    /**
     * @return the back url if the process fails
     */
    public String getPreviousStageURL()
    {
        List<Breadcrumb> history = DIFIoCRegistry.getRegistry().getImplementation(INavigationHistory.class)
                .getHistoryLastAccessed();

        if (history == null || history.isEmpty())
            return HttpUtils.getStageLink(HTTPControllerConfiguration.getInstance().getHomeStageID());
        else
        {
            Breadcrumb crumb = history.get(0);

            return HttpUtils.getStageLinkWithParameters(crumb.getStageID(), crumb.getParameterPassed());
        }
    }

    /**
     * The SSO logic
     * 
     * @return the SSO response
     */
    @OnAJAX("requestSSO")
    public SSOResponse requestSSO()
    {
        if (!context.getSession().isLogged())
        {
            return new SSOResponse(false, null, messages.get("SSOOnlyWhenUserLogged"));
        }
        else
        {
            HttpRequestWithBody jsonRequest = null;
            HttpResponse<JsonNode> jsonResponse = null;

            try
            {
                if (!SSOSecurityConfiguration.getInstance().getActive())
                    return new SSOResponse(false, null, messages.get("SSODisabled"));
                else
                {
                    // Build final URL
                    if (!serverURL.endsWith("/"))
                        serverURL += "/";
                    if (!serverURL.startsWith("http://") && !serverURL.startsWith("https://"))
                        serverURL = "http://" + serverURL;

                    // Build parameter map
                    Map<String, Object> parameters = new HashMap<String, Object>();
                    parameters.put("userID", context.getSession().getUser().getID());
                    parameters.put("destinationStageID", stageID);
                    parameters.put(HTTPConstants.AJAX_MODE_PARAMETER, true);
                    parameters.put(HTTPConstants.STAGE_PARAMETER, SSORequestReceiver.class.getSimpleName());
                    parameters.put(HTTPConstants.EVENT_ID, SSORequestReceiver.SSO_REQUEST_AJAX_HANDLER_ID);

                    // Create request
                    jsonRequest = Unirest.post(serverURL + "ajax").header("accept", "application/json")
                            .queryString(parameters);
                    // Do the request and get the response
                    jsonResponse = jsonRequest.asJson();

                    if (jsonResponse.getStatus() == 200)
                    {
                        // HTTP:200 OK: carry on with success and analyze response
                        JSONObject result = jsonResponse.getBody().getObject().getJSONObject("result");
                        tokenID = result.getString("tokenID");

                        if (StringUtils.isNotBlank(tokenID))
                            return new SSOResponse(true, tokenID).setRedirectURL(serverURL + "page?"
                                    + HTTPConstants.STAGE_PARAMETER + "=" + SSORequestReceiver.class.getSimpleName());
                        else
                            return new SSOResponse(false, null,
                                    messages.get("processFailed") + ": " + messages.get("tokenNotReceived"));
                    }
                    else
                    {
                        // Not HTTP:200 OK: Process as error!
                        String errorMessage = messages.get("processFailed") + ": " + messages.get("errorCommunicating")
                                + " \"" + serverURL + "\"<br/>\n" + messages.get("result") + ": HTTP "
                                + jsonResponse.getStatus() + " - " + jsonResponse.getStatusText();

                        return new SSOResponse(false, null, errorMessage);
                    }
                }
            }
            catch (Exception e)
            {
                String errorMessage = messages.get("processFailed") + ": " + messages.get("errorCommunicating") + " \""
                        + serverURL;
                String errorMessageTech = messages.get("result") + ": " + e.getMessage();

                if (e instanceof JSONException || e instanceof UnirestException)
                    errorMessageTech = messages.get("result") + ": " + messages.get("noResponseOrBadFormat") + "<br/>\n"
                            + messages.get("Exception") + ": " + e.getMessage();

                new BusinessException(e).addToExceptionContext(context)
                        .addToExceptionContext("jsonRequest", jsonRequest)
                        .addToExceptionContext("jsonResponse", jsonResponse).log(LogLevel.ERROR);

                return new SSOResponse(false, null, errorMessage).setErrorMessageTechDetails(errorMessageTech);
            }

        }
    }
}
