/**
 * 2007, Digitalis Informatica. All rights reserved. Distribuicao e Gestao de Informatica, Lda. Estrada de Paco de Arcos
 * num.9 - Piso -1 2780-666 Paco de Arcos Telefone: (351) 21 4408990 Fax: (351) 21 4408999 http://www.digitalis.pt
 */
package pt.digitalis.dif.utils.http;

import java.net.URL;
import java.util.jar.Attributes;
import java.util.jar.Manifest;

import javax.servlet.ServletRequest;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpSession;

import pt.digitalis.dif.controller.http.HTTPConstants;
import pt.digitalis.dif.controller.http.HTTPControllerConfiguration;
import pt.digitalis.dif.controller.interfaces.IDIFContext;
import pt.digitalis.dif.controller.objects.DIFRequest;
import pt.digitalis.dif.dem.interfaces.IStage;
import pt.digitalis.dif.dem.managers.IDEMManager;
import pt.digitalis.dif.exception.InternalFrameworkException;
import pt.digitalis.dif.ioc.DIFIoCRegistry;
import pt.digitalis.utils.common.StringUtils;
import pt.digitalis.utils.config.ConfigurationsPreferencesImpl;

/**
 * General utilities for HTTP requests
 * 
 * @author Pedro Viegas <a href="mailto:pviegas@digitalis.pt">pviegas@digitalis.pt</a>
 * @created Dec 19, 2007
 */
public class HttpUtils {

    /** The alternative name to retrieve from the manifest */
    static private final String ALTERNATIVE_NAME_TAG = "AlternativeName";

    /** The cache for the application name and version */
    static private String appTagCache = null;

    /** The Configuration preferences to get the prefix */
    private static final ConfigurationsPreferencesImpl config = new ConfigurationsPreferencesImpl();

    /** The important dependencies to retrieve from the manifest */
    static private final String IMPORTANT_DEPENDENCIES_TAG = "ImportantDependencies";

    /** The cache for version only */
    static private String version = null;

    /**
     * Get the session id name, from the HttpSession
     * 
     * @param session
     *            the http session object
     * @return session id
     */
    public static String buildSessionId(HttpSession session)
    {
        return HTTPConstants.HTTP_CHANNEL_ID + session.getId();
    }

    /**
     * @return the current application bundle tag with it's name and version
     */
    static public String getApplicationTag()
    {
        if (appTagCache == null)
        {
            IDEMManager demManager = DIFIoCRegistry.getRegistry().getImplementation(IDEMManager.class);
            IStage mainStage = null;
            Manifest manifest = getManifest();

            try
            {
                mainStage = demManager.getStage(HTTPControllerConfiguration.getInstance().getHomeStageID());
            }
            catch (Exception e)
            {
            }

            if (manifest == null)
            {
                if (mainStage == null)
                    appTagCache = "";
                else
                    appTagCache = mainStage.getService().getApplication().getName();
            }
            else
            {
                String importantDependencies = "";

                if (manifest.getAttributes(IMPORTANT_DEPENDENCIES_TAG) != null
                        && manifest.getAttributes(IMPORTANT_DEPENDENCIES_TAG).getValue("id") != null)
                {
                    importantDependencies = " (" + manifest.getAttributes(IMPORTANT_DEPENDENCIES_TAG).getValue("id")
                            + ")";
                }

                String title = "";

                if (manifest.getAttributes(ALTERNATIVE_NAME_TAG) != null)
                {
                    title = manifest.getAttributes(ALTERNATIVE_NAME_TAG).getValue("id");
                }
                else
                {
                    title = manifest.getMainAttributes().getValue(Attributes.Name.IMPLEMENTATION_TITLE);
                }

                appTagCache = "<abbr title=\"v"
                        + manifest.getMainAttributes().getValue(Attributes.Name.IMPLEMENTATION_VERSION)
                        + importantDependencies + "\">" + title + "</abbr>";
            }
        }

        return appTagCache;
    }

    /**
     * Get the server base URL. This method requires that at least one HTTP request has been made.
     * 
     * @return the server base URL
     * @throws InternalFrameworkException
     */
    static public String getBaseURL() throws InternalFrameworkException
    {
        String url = HTTPControllerConfiguration.getInstance().getServerBaseURL();

        if (url == null || "".equals(url))
            throw new InternalFrameworkException("Server Base URL configuration must be filled!", null);
        else
        {
            if (!url.endsWith("/"))
            {
                url += "/";
            }
            url += config.getGeneralPrefix();
        }

        if (url.endsWith("/"))
            url = url.substring(0, url.lastIndexOf('/')) + "/";

        url = removeRESTPrefix(url);

        return url;
    }

    /**
     * @param httpRequest
     *            the http request to parse
     * @return the relative URL constructed from the context
     */
    static public String getBaseURL(HttpServletRequest httpRequest)
    {
        String url = HTTPControllerConfiguration.getInstance().getServerBaseURL();

        if (url == null || "".equals(url))
            url = removeListenerPrefix(getRequestURL(httpRequest));
        else
            url += getRequestContextPath(httpRequest);

        if (url.endsWith("/"))
            url = url.substring(0, url.lastIndexOf('/')) + "/";

        url = removeRESTPrefix(url);

        return url;
    }

    /**
     * @param context
     *            the context to parse
     * @return the relative URL constructed from the context
     */
    static public String getBaseURL(IDIFContext context)
    {
        HttpServletRequest httpRequest = (HttpServletRequest) context.getRequest()
                .getAttribute(DIFRequest.ORIGINAL_REQUEST);

        return getBaseURL(httpRequest);
    }

    /**
     * Get the manifest
     * 
     * @return the manifest
     */
    public static Manifest getManifest()
    {

        IDEMManager demManager = DIFIoCRegistry.getRegistry().getImplementation(IDEMManager.class);
        IStage mainStage = null;
        Manifest manifest = null;

        try
        {
            mainStage = demManager.getStage(HTTPControllerConfiguration.getInstance().getHomeStageID());
            Class<?> originalStageClass = Class.forName(mainStage.getOriginalClassName());

            // Hack. Get the index.html to infer the place of the MANIFEST...
            String manifestPath = originalStageClass.getClassLoader().getResource("index.htm").toString()
                    .replace("index.htm", "META-INF/MANIFEST.MF");
            manifest = new Manifest(new URL(manifestPath).openStream());
        }
        catch (Exception e)
        {
        }
        return manifest;
    }

    /**
     * @param request
     *            the request to parse
     * @return the URL constructed from the ServletRequest
     */
    static public String getRequestContextPath(ServletRequest request)
    {
        HttpServletRequest httpRequest = (HttpServletRequest) request;
        return httpRequest.getContextPath();
    }

    /**
     * @param request
     *            the request to parse
     * @return the relative URL constructed from the ServletRequest
     */
    static public String getRequestRelativeURL(ServletRequest request)
    {

        HttpServletRequest httpRequest = (HttpServletRequest) request;

        return removeRESTPrefix(httpRequest.getServletPath());
    }

    /**
     * @param context
     *            the context to parse
     * @return the relative URL constructed from the context
     */
    static public String getRequestURL(IDIFContext context)
    {

        HttpServletRequest httpRequest = (HttpServletRequest) context.getRequest()
                .getAttribute(DIFRequest.ORIGINAL_REQUEST);

        return getRequestURL(httpRequest);
    }

    /**
     * @param request
     *            the request to parse
     * @return the URL constructed from the ServletRequest
     */
    static public String getRequestURL(ServletRequest request)
    {
        String url = HTTPControllerConfiguration.getInstance().getServerBaseURL();

        if (url == null || "".equals(url))
        {
            HttpServletRequest httpRequest = (HttpServletRequest) request;
            return httpRequest.getRequestURL().toString();
        }
        else
        {
            if (url.endsWith("/"))
                url = url.substring(0, url.lastIndexOf('/')) + "/";

            url = removeRESTPrefix(url);
            url += getRequestContextPath(request) + "/" + HTTPControllerConfiguration.getInstance().getListenerName();

            return url;
        }
    }

    /**
     * @param stageID
     *            the stage to create the link to
     * @return a relative URL for the specified stage
     */
    static public String getStageLink(String stageID)
    {
        return getStageLinkWithParameters(stageID, null);
    }

    /**
     * @param stageID
     *            the stage to create the link to
     * @param listener
     *            the listener to call the stage from
     * @param parameters
     *            the parameter in URL notation. i.e. "param1=value1&param2=value2..."
     * @return a relative URL for the specified stage
     */
    static public String getStageLink(String stageID, String listener, String parameters)
    {
        try
        {
            if (StringUtils.isBlank(parameters))
                parameters = "";
            else
                parameters = "&" + parameters;

            return listener + "?" + HTTPConstants.STAGE_PARAMETER + "=" + stageID + parameters;
        }
        catch (Exception e)
        {
            return "";
        }
    }

    /**
     * @param stageID
     *            the stage to create the link to
     * @param parameters
     *            the parameter in URL notation. i.e. "param1=value1&param2=value2..."
     * @return a relative URL for the specified stage
     */
    static public String getStageLinkWithParameters(String stageID, String parameters)
    {
        return getStageLink(stageID, HTTPControllerConfiguration.getInstance().getListenerName(), parameters);
    }

    /**
     * Get the application version
     * 
     * @return the version
     */
    static public String getVersion()
    {

        if (version == null)
        {
            Manifest manifest = getManifest();

            if (manifest == null)
            {
                version = "";
            }
            else
                version = manifest.getMainAttributes().getValue(Attributes.Name.IMPLEMENTATION_VERSION);
        }

        return version;
    }

    /**
     * Converts basic HTML tags to simple plain text formatting
     * 
     * @param source
     *            the HTML source
     * @return the converted plain test
     */
    public static String htmlContentToPlainText(String source)
    {
        String result = source;

        result = result.replaceAll("<br[ ]*/>\\n", "\n");
        result = result.replaceAll("<br[ ]*/>", "\n");
        result = result.replaceAll("</li>", "\n");
        result = result.replaceAll("<li>", "  * ");
        result = result.replaceAll("<ul>", "\n");
        result = result.replaceAll("</ul>", "\n");
        result = result.replaceAll("<[b,i,u]>", "");
        result = result.replaceAll("</[b,i,u]>", "");
        result = result.replaceAll("<hr>",
                "-----------------------------------------------------------------------------");
        result = result.replaceAll("<span[^>]*>", "");
        result = result.replaceAll("</span>", "");
        result = result.replaceAll("<div[^>]*>", "");
        result = result.replaceAll("</div>", "\n");

        return result;
    }

    /**
     * Removes all listener or AJAX URLs parts to return only the base URL
     * 
     * @param path
     *            the original REST URL
     * @return the URL without the REST information
     */
    private static String removeListenerPrefix(String path)
    {
        path = path.replaceAll("/page[/\\?]?.*", "/");
        path = path.replaceAll("/ajax[/\\?]?.*", "/");
        path = path.replaceAll("/doc[/\\?]?.*", "/");
        path = path.replaceAll("/asset[/\\?]?.*", "/");

        return path;
    }

    /**
     * Converts a REST URL to a normal servlet call URL
     * 
     * @param path
     *            the original REST URL
     * @return the URL without the REST information
     */
    private static String removeRESTPrefix(String path)
    {
        path = path.replaceAll("/page/.*", "/page");
        path = path.replaceAll("/ajax/.*", "/ajax");
        path = path.replaceAll("/doc/.*", "/doc");
        path = path.replaceAll("/asset/.*", "/asset");

        return path;
    }

    /**
     * @param string
     *            String to escape the HTML markup characters
     * @return the escaped string
     */
    public static String stringToHTMLString(String string)
    {
        StringBuffer sb = new StringBuffer(string.length());
        // true if last char was blank
        boolean lastWasBlankChar = false;
        int len = string.length();
        char c;

        for (int i = 0; i < len; i++)
        {
            c = string.charAt(i);
            if (c == ' ')
            {
                // blank gets extra work,
                // this solves the problem you get if you replace all
                // blanks with &nbsp;, if you do that you loss
                // word breaking
                if (lastWasBlankChar)
                {
                    lastWasBlankChar = false;
                    sb.append("&nbsp;");
                }
                else
                {
                    lastWasBlankChar = true;
                    sb.append(' ');
                }
            }
            else
            {
                lastWasBlankChar = false;
                //
                // HTML Special Chars
                if (c == '"')
                    sb.append("&quot;");
                else if (c == '&')
                    sb.append("&amp;");
                else if (c == '<')
                    sb.append("&lt;");
                else if (c == '>')
                    sb.append("&gt;");
                else if (c == '\n')
                    // Handle Newline
                    sb.append("&lt;br/&gt;");
                else
                {
                    int ci = 0xffff & c;
                    if (ci < 160)
                        // nothing special only 7 Bit
                        sb.append(c);
                    else
                    {
                        // Not 7 Bit use the unicode system
                        sb.append("&#");
                        sb.append(new Integer(ci).toString());
                        sb.append(';');
                    }
                }
            }
        }
        return sb.toString();
    }
}
