/**
 * 2009, Digitalis Informatica. All rights reserved. Distribuicao e Gestao de Informatica, Lda. Estrada de Paco de Arcos
 * num.9 - Piso -1 2780-666 Paco de Arcos Telefone: (351) 21 4408990 Fax: (351) 21 4408999 http://www.digitalis.pt
 */

package pt.digitalis.dif.utils.pdf;

import java.io.FileInputStream;
import java.io.IOException;
import java.security.KeyStore;
import java.security.KeyStoreException;
import java.security.NoSuchAlgorithmException;
import java.security.cert.Certificate;
import java.security.cert.CertificateException;
import java.security.cert.X509Certificate;
import java.util.Enumeration;

import pt.digitalis.dif.ioc.DIFIoCRegistry;
import pt.digitalis.utils.config.IConfigurations;
import pt.digitalis.utils.config.annotations.ConfigDefault;
import pt.digitalis.utils.config.annotations.ConfigID;
import pt.digitalis.utils.config.annotations.ConfigIgnore;
import pt.digitalis.utils.config.annotations.ConfigSectionID;

/**
 * The Class Digital Certificates Security Configuration.
 * 
 * @author Luis Pinto <a href="mailto:lpinto@digitalis.pt">lpinto@digitalis.pt</a><br/>
 * @created Jan 31, 2012
 */
@ConfigID("dif2")
@ConfigSectionID("Security/DigitalCertificate")
public class DigitalCertificateConfiguration {

    /** The singleton instance property. */
    private static DigitalCertificateConfiguration instance;

    /**
     * Returns the active configuration object instance.
     * 
     * @return the configuration instance
     * @throws Exception
     *             the exception
     */
    @ConfigIgnore
    static public DigitalCertificateConfiguration getInstance() throws Exception
    {
        if (instance == null)
        {
            try
            {
                instance = DIFIoCRegistry.getRegistry().getImplementation(IConfigurations.class)
                        .readConfiguration(DigitalCertificateConfiguration.class);
            }
            catch (Exception e)
            {
                e.printStackTrace();
                instance = null;
            }
            if (instance != null)
            {
                instance.initialize();
            }
        }

        return instance;
    }

    /**
     * Save configuration.
     * 
     * @param path
     *            the path
     * @param password
     *            the password
     * @param showSignature
     *            the show signature
     * @param lowerLeftX
     *            the lower left x
     * @param lowerLeftY
     *            the lower left y
     * @param upperRightX
     *            the upper right x
     * @param upperRightY
     *            the upper right y
     * @param contact
     *            the contact
     * @param reason
     *            the reason
     * @param location
     *            the location
     * @throws Exception
     *             the exception
     */
    public static void saveConfiguration(String path, String password, Boolean showSignature, Double lowerLeftX,
            Double lowerLeftY, Double upperRightX, Double upperRightY, String contact, String reason, String location)
            throws Exception
    {
        try
        {
            DigitalCertificateConfiguration.getInstance();
        }
        catch (Exception e)
        {
            // ignore
        }
        if (path != null && !"".equals(path))
        {
            instance.setPath(path);
        }

        if (password != null && !"".equals(password))
        {

            instance.setPassword(password);
        }

        DigitalCertificateConfiguration.getInstance().setShowSignature(showSignature);

        if (lowerLeftX != null)
        {
            instance.setLowerLeftX(lowerLeftX.toString());
        }
        if (lowerLeftY != null)
        {
            instance.setLowerLeftY(lowerLeftY.toString());
        }
        if (upperRightX != null)
        {
            instance.setUpperRightX(upperRightX.toString());
        }
        if (upperRightY != null)
        {
            instance.setUpperRightY(upperRightY.toString());
        }

        if (location != null)
        {
            instance.setLocation(location);
        }
        if (reason != null)
        {
            instance.setReason(reason);
        }
        if (contact != null)
        {
            instance.setContact(contact);
        }

        DIFIoCRegistry.getRegistry().getImplementation(IConfigurations.class)
                .writeConfiguration(DigitalCertificateConfiguration.getInstance());
        instance = null;
    }

    /** The alias. */
    private String alias = null;

    /** The contact. */
    private String contact;

    /** The initialized. */
    private boolean initialized = false;

    /** The key store. */
    private KeyStore keystore = null;

    /** The key store. */
    private KeyStore keyStoreAll = null;

    /** The location. */
    private String location;

    /** The lower left x. */
    private String lowerLeftX;

    /** The lower left y. */
    private String lowerLeftY;

    /** The password. */
    private String password;

    /** The path. */
    private String path;

    /** The reason. */
    private String reason;

    /** The show signature. */
    private Boolean showSignature;

    /** The type. */
    private String type;

    /** The upper right x. */
    private String upperRightX;

    /** The upper right y. */
    private String upperRightY;

    /**
     * Inspector for the 'alias' attribute.
     * 
     * @return the alias value
     * @throws IOException
     * @throws CertificateException
     * @throws NoSuchAlgorithmException
     * @throws KeyStoreException
     */
    @ConfigIgnore
    public String getAlias() throws KeyStoreException, NoSuchAlgorithmException, CertificateException, IOException
    {
        if (!initialized)
        {
            initialize();
        }
        return alias;
    }

    /**
     * Inspector for the 'contact' attribute.
     * 
     * @return the contact value
     */
    @ConfigDefault("")
    public String getContact()
    {
        return contact;
    }

    /**
     * Inspector for the 'keystore' attribute.
     * 
     * @return the keystore value
     * @throws IOException
     * @throws CertificateException
     * @throws NoSuchAlgorithmException
     * @throws KeyStoreException
     */
    @ConfigIgnore
    public KeyStore getKeystore() throws KeyStoreException, NoSuchAlgorithmException, CertificateException, IOException
    {
        if (!initialized)
        {
            initialize();
        }
        return keystore;
    }

    /**
     * Inspector for the 'keyStoreAll' attribute.
     * 
     * @return the keyStoreAll value
     * @throws IOException
     * @throws CertificateException
     * @throws NoSuchAlgorithmException
     * @throws KeyStoreException
     */
    @ConfigIgnore
    public KeyStore getKeyStoreAll()
            throws KeyStoreException, NoSuchAlgorithmException, CertificateException, IOException
    {
        if (!initialized)
        {
            initialize();
        }
        return keyStoreAll;
    }

    /**
     * Inspector for the 'location' attribute.
     * 
     * @return the location value
     */
    @ConfigDefault("")
    public String getLocation()
    {
        return location;
    }

    /**
     * Inspector for the 'lowerLeftX' attribute.
     * 
     * @return the lowerLeftX value
     */
    @ConfigDefault("")
    public Double getLowerLeftX()
    {
        if (lowerLeftX != null && !"".equals(lowerLeftX))
        {
            return Double.valueOf(lowerLeftX);
        }
        else
        {
            return null;
        }
    }

    /**
     * Inspector for the 'lowerLeftY' attribute.
     * 
     * @return the lowerLeftY value
     */
    @ConfigDefault("")
    public Double getLowerLeftY()
    {
        if (lowerLeftY != null && !"".equals(lowerLeftY))
        {
            return Double.valueOf(lowerLeftY);
        }
        else
        {
            return null;
        }
    }

    /**
     * Inspector for the 'password' attribute.
     * 
     * @return the password value
     */
    @ConfigDefault("")
    public String getPassword()
    {
        return password;
    }

    /**
     * Inspector for the 'path' attribute.
     * 
     * @return the path value
     */
    @ConfigDefault("")
    public String getPath()
    {
        return path;
    }

    /**
     * Inspector for the 'reason' attribute.
     * 
     * @return the reason value
     */
    @ConfigDefault("")
    public String getReason()
    {
        return reason;
    }

    /**
     * Inspector for the 'showSignature' attribute.
     * 
     * @return the showSignature value
     */
    @ConfigDefault("false")
    public Boolean getShowSignature()
    {
        return showSignature;
    }

    /**
     * Inspector for the 'type' attribute.
     * 
     * @return the type value
     */
    @ConfigDefault("pkcs12")
    public String getType()
    {
        return type;
    }

    /**
     * Inspector for the 'upperRightX' attribute.
     * 
     * @return the upperRightX value
     */
    @ConfigDefault("")
    public Double getUpperRightX()
    {
        if (upperRightX != null && !"".equals(upperRightX))
        {
            return Double.valueOf(upperRightX);
        }
        else
        {
            return null;
        }
    }

    /**
     * Inspector for the 'upperRightY' attribute.
     * 
     * @return the upperRightY value
     */
    @ConfigDefault("")
    public Double getUpperRightY()
    {
        if (upperRightY != null && !"".equals(upperRightY))
        {
            return Double.valueOf(upperRightY);
        }
        else
        {
            return null;
        }

    }

    /**
     * Initialize.
     * 
     * @throws KeyStoreException
     *             the key store exception
     * @throws NoSuchAlgorithmException
     *             the no such algorithm exception
     * @throws CertificateException
     *             the certificate exception
     * @throws IOException
     *             Signals that an I/O exception has occurred.
     */
    private void initialize() throws KeyStoreException, NoSuchAlgorithmException, CertificateException, IOException
    {
        if (!initialized && this.path != null && this.password != null && !"".equals(this.path)
                && !"".equals(this.password))
        {

            keyStoreAll = KeyStore.getInstance("JKS");
            keyStoreAll.load(null, null);

            keystore = KeyStore.getInstance(this.getType());
            keystore.load(new FileInputStream(this.path), this.password.toCharArray());

            alias = keystore.aliases().nextElement();

            Enumeration<String> e = keystore.aliases();

            /* Add the certificate chain to the store */
            for (; e.hasMoreElements();)
            {
                String a = e.nextElement();
                Certificate mainCert = keystore.getCertificate(a);
                keyStoreAll.setCertificateEntry(a, mainCert);

                Certificate[] certificados = keystore.getCertificateChain(a);

                for (Certificate cert: certificados)
                {
                    X509Certificate x509cert = (X509Certificate) cert;
                    keyStoreAll.setCertificateEntry(x509cert.getIssuerDN().getName(), x509cert);
                }
            }
            initialized = true;
        }
    }

    /**
     * Modifier for the 'contact' attribute.
     * 
     * @param contact
     *            the new contact value to set
     */
    public void setContact(String contact)
    {
        this.contact = contact;
    }

    /**
     * Modifier for the 'location' attribute.
     * 
     * @param location
     *            the new location value to set
     */
    public void setLocation(String location)
    {
        this.location = location;
    }

    /**
     * Modifier for the 'lowerLeftX' attribute.
     * 
     * @param lowerLeftX
     *            the new lowerLeftX value to set
     */
    public void setLowerLeftX(String lowerLeftX)
    {
        this.lowerLeftX = lowerLeftX;
    }

    /**
     * Modifier for the 'lowerLeftY' attribute.
     * 
     * @param lowerLeftY
     *            the new lowerLeftY value to set
     */
    public void setLowerLeftY(String lowerLeftY)
    {
        this.lowerLeftY = lowerLeftY;
    }

    /**
     * Modifier for the 'password' attribute.
     * 
     * @param password
     *            the new password value to set
     */
    public void setPassword(String password)
    {
        this.password = password;
    }

    /**
     * Modifier for the 'path' attribute.
     * 
     * @param path
     *            the new path value to set
     */
    public void setPath(String path)
    {
        this.path = path;
    }

    /**
     * Modifier for the 'reason' attribute.
     * 
     * @param reason
     *            the new reason value to set
     */
    public void setReason(String reason)
    {
        this.reason = reason;
    }

    /**
     * Modifier for the 'showSignature' attribute.
     * 
     * @param showSignature
     *            the new showSignature value to set
     */
    public void setShowSignature(Boolean showSignature)
    {
        this.showSignature = showSignature;
    }

    /**
     * Modifier for the 'type' attribute.
     * 
     * @param type
     *            the new type value to set
     */
    public void setType(String type)
    {
        this.type = type;
    }

    /**
     * Modifier for the 'upperRightX' attribute.
     * 
     * @param upperRightX
     *            the new upperRightX value to set
     */
    public void setUpperRightX(String upperRightX)
    {
        this.upperRightX = upperRightX;
    }

    /**
     * Modifier for the 'upperRightY' attribute.
     * 
     * @param upperRightY
     *            the new upperRightY value to set
     */
    public void setUpperRightY(String upperRightY)
    {
        this.upperRightY = upperRightY;
    }

}
