/**
 * 2009, Digitalis Informatica. All rights reserved. Distribuicao e Gestao de Informatica, Lda. Estrada de Paco de Arcos
 * num.9 - Piso -1 2780-666 Paco de Arcos Telefone: (351) 21 4408990 Fax: (351) 21 4408999 http://www.digitalis.pt
 */

package pt.digitalis.dif.utils.pdf;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.security.KeyStore;
import java.security.PrivateKey;
import java.security.cert.Certificate;
import java.security.cert.X509Certificate;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Date;
import java.util.LinkedHashMap;
import java.util.Map;

import javax.security.auth.x500.X500Principal;

import com.lowagie.text.Document;
import com.lowagie.text.DocumentException;
import com.lowagie.text.Paragraph;
import com.lowagie.text.Rectangle;
import com.lowagie.text.pdf.AcroFields;
import com.lowagie.text.pdf.PdfPKCS7;
import com.lowagie.text.pdf.PdfReader;
import com.lowagie.text.pdf.PdfSignatureAppearance;
import com.lowagie.text.pdf.PdfStamper;
import com.lowagie.text.pdf.PdfWriter;

import pt.digitalis.utils.common.StringUtils;

/**
 * The Class PDFUtils.
 * 
 * @author Luis Pinto <a href="mailto:lpinto@digitalis.pt">lpinto@digitalis.pt</a><br/>
 * @created Jan 31, 2012
 */
public class PDFUtils {

    /** The Constant SIGNATURE_NAME. */
    private static final String SIGNATURE_NAME = "SignatureAddedByDigitalis";

    /**
     * Creates the dummy pdf.
     * 
     * @return the {@link ByteArrayOutputStream} PDF
     * @throws DocumentException
     *             the document exception
     * @throws IOException
     *             Signals that an I/O exception has occurred.
     */
    public static ByteArrayOutputStream createDummyPDF() throws DocumentException, IOException
    {
        ByteArrayOutputStream result = new ByteArrayOutputStream();

        Document document = new Document();
        PdfWriter.getInstance(document, result);
        document.open();
        document.add(new Paragraph("Create by Digitalis Development Team (For Tests Purpose)"));
        document.add(new Paragraph(new Date().toString()));

        document.close();

        return result;
    }

    /**
     * Gets if certificate available
     * 
     * @return if certificate available
     * @throws Exception
     */
    public static boolean getCertificateAvailable() throws Exception
    {
        DigitalCertificateConfiguration conf = DigitalCertificateConfiguration.getInstance();
        return StringUtils.isNotBlank(conf.getPath());
    }

    /**
     * Gets the certificate data.
     * 
     * @return the certificate data
     * @throws Exception
     */
    public static Map<String, String> getCertificateData() throws Exception
    {
        Map<String, String> result = new LinkedHashMap<String, String>();

        DigitalCertificateConfiguration conf = DigitalCertificateConfiguration.getInstance();
        KeyStore keystore = conf.getKeystore();
        String alias = conf.getAlias();

        X509Certificate cert;

        cert = (X509Certificate) keystore.getCertificate(alias);

        for (String value: cert.getSubjectX500Principal().getName(X500Principal.RFC1779).split(","))
        {
            String[] values = value.split("=");
            if (values.length == 2)
            {
                result.put("Issue To \"" + values[0] + "\"", values[1]);
            }
        }

        result.put("Issuer", cert.getIssuerX500Principal().getName());
        if (cert.getNotBefore() != null)
        {
            result.put("Validity From", cert.getNotBefore().toString());
        }
        if (cert.getNotAfter() != null)
        {
            result.put("Validity To", cert.getNotAfter().toString());
        }
        result.put("SerialNumber", cert.getSerialNumber().toString());
        result.put("Type", cert.getType());

        return result;
    }

    /**
     * Sign PDF.
     * 
     * @param pdfToSign
     *            The PDF to Sign
     * @return Then PDF Signed in
     * @throws Exception
     */
    public static ByteArrayOutputStream signPDF(byte[] pdfToSign) throws Exception
    {
        DigitalCertificateConfiguration conf = DigitalCertificateConfiguration.getInstance();
        return signPDF(pdfToSign, conf.getShowSignature(), conf.getReason(), conf.getLocation(), conf.getContact(),
                conf.getLowerLeftX(), conf.getLowerLeftY(), conf.getUpperRightX(), conf.getUpperRightY());
    }

    /**
     * Sign PDF.
     * 
     * @param pdfToSign
     *            The PDF to Sign
     * @param showSignature
     *            <code>false</code> to make the signature invisible
     * @param reason
     *            the reason (signature property)
     * @param location
     *            the location (signature property)
     * @param contact
     *            the contact (signature property)
     * @param lowerLeftX
     *            the lower left x
     * @param lowerLeftY
     *            the lower left y
     * @param upperRightX
     *            the upper right x
     * @param upperRightY
     *            the upper right y
     * @return Then PDF Signed
     * @throws Exception
     */
    public static ByteArrayOutputStream signPDF(byte[] pdfToSign, boolean showSignature, String reason, String location,
            String contact, Double lowerLeftX, Double lowerLeftY, Double upperRightX, Double upperRightY)
            throws Exception
    {
        ByteArrayOutputStream fout = null;

        DigitalCertificateConfiguration conf = DigitalCertificateConfiguration.getInstance();
        KeyStore keystore = conf.getKeystore();
        String alias = conf.getAlias();
        PrivateKey key = (PrivateKey) keystore.getKey(alias, conf.getPassword().toCharArray());
        Certificate[] chain = keystore.getCertificateChain(alias);

        PdfReader reader = new PdfReader(pdfToSign);
        fout = new ByteArrayOutputStream();
        PdfStamper stp = PdfStamper.createSignature(reader, fout, '\0');
        PdfSignatureAppearance sap = stp.getSignatureAppearance();
        sap.setCrypto(key, chain, null, PdfSignatureAppearance.WINCER_SIGNED);
        if (reason != null)
        {
            sap.setReason(reason);
        }
        if (location != null)
        {
            sap.setLocation(location);
        }
        if (contact != null)
        {
            sap.setContact(contact);
        }

        // comment next line to have an invisible signature
        if (showSignature)
        {
            // If some of the coordinates aren't passed, default values come into action
            Float lowerLeftXAux = (lowerLeftX == null ? 100 : lowerLeftX.floatValue());
            Float lowerLeftYAux = (lowerLeftY == null ? 100 : lowerLeftY.floatValue());
            Float upperRightXAux = (upperRightX == null ? 200 : upperRightX.floatValue());
            Float upperRightYAux = (upperRightY == null ? 200 : upperRightY.floatValue());
            sap.setVisibleSignature(new Rectangle(lowerLeftXAux, lowerLeftYAux, upperRightXAux, upperRightYAux), 1,
                    SIGNATURE_NAME);
        }

        sap.setCertificationLevel(PdfSignatureAppearance.CERTIFIED_NO_CHANGES_ALLOWED);

        stp.close();

        return fout;
    }

    /**
     * Validate signature.
     * 
     * @param pdfToValidateSingature
     *            the pdf to validate Signature
     * @return true, if successful
     * @throws Exception
     *             the exception
     */
    @SuppressWarnings("rawtypes")
    public static boolean validateSignature(byte[] pdfToValidateSingature) throws Exception
    {
        Boolean result = null;

        DigitalCertificateConfiguration conf = DigitalCertificateConfiguration.getInstance();

        PdfReader reader = new PdfReader(pdfToValidateSingature);
        AcroFields af = reader.getAcroFields();
        ArrayList names = af.getSignatureNames();
        for (int k = 0; k < names.size(); ++k)
        {
            String name = (String) names.get(k);
            PdfPKCS7 pk = af.verifySignature(name);
            Calendar cal = pk.getSignDate();
            X509Certificate pkc[] = (X509Certificate[]) pk.getCertificates();

            Object fails[] = PdfPKCS7.verifyCertificates(pkc, conf.getKeyStoreAll(), null, cal);
            if (fails == null)
            {
                if (result == null)
                {
                    result = true;
                }
                else
                {
                    result = result && true;
                }
            }
        }

        if (result == null)
        {
            result = false;
        }
        return result;
    }
}
