/**
 * 2009, Digitalis Informatica. All rights reserved. Distribuicao e Gestao de Informatica, Lda. Estrada de Paco de Arcos
 * num.9 - Piso -1 2780-666 Paco de Arcos Telefone: (351) 21 4408990 Fax: (351) 21 4408999 http://www.digitalis.pt
 */

package pt.digitalis.dif.utils.security;

import javax.servlet.http.HttpServletResponse;

import pt.digitalis.dif.ioc.DIFIoCRegistry;
import pt.digitalis.utils.common.StringUtils;
import pt.digitalis.utils.config.IConfigurations;
import pt.digitalis.utils.config.annotations.ConfigDefault;
import pt.digitalis.utils.config.annotations.ConfigID;
import pt.digitalis.utils.config.annotations.ConfigIgnore;
import pt.digitalis.utils.config.annotations.ConfigSectionID;

/**
 * The Class HTTPSecurityConfiguration.
 * 
 * @author Pedro Viegas <a href="mailto:pviegas@digitalis.pt">pviegas@digitalis.pt</a><br/>
 * @created 13/11/2014
 */
@ConfigID("dif2")
@ConfigSectionID("Security/HTTP")
public class HTTPSecurityConfiguration {

    /** The singleton instance property. */
    private static HTTPSecurityConfiguration instance;

    /**
     * Returns the active configuration object instance.
     * 
     * @return the configuration instance
     * @throws Exception
     *             the exception
     */
    @ConfigIgnore
    static public HTTPSecurityConfiguration getInstance() throws Exception
    {
        if (instance == null)
        {
            try
            {
                instance = DIFIoCRegistry.getRegistry().getImplementation(IConfigurations.class)
                        .readConfiguration(HTTPSecurityConfiguration.class);
            }
            catch (Exception e)
            {
                e.printStackTrace();
                instance = null;
            }
        }

        return instance;
    }

    /**
     * Mail destination (in production mode) for a security report each time a security authorization is not respected.
     * The reports will be generated by vialoations of the "validateRequestAuthorizations" and
     * "xssParameterSanitization" rules.
     */
    private String securityViolationMailReportDestination;

    /**
     * Mail destination (in testing mode) for a security report each time a security authorization is not respected. The
     * reports will be generated by vialoations of the "validateRequestAuthorizations" and "xssParameterSanitization"
     * rules.
     */
    private String securityViolationMailReportTestingDestination;

    /**
     * Activates the security validator or authorized actions. This validator monitors all JSON Dataset requests, the
     * authorized actions and fields modified in these actions.
     */
    private Boolean validateRequestAuthorizations;

    /**
     * the X-Content-Type-Options HTTP header<br/>
     * This header can be set to protect against MIME type confusion attacks in Internet Explorer 9, Chrome and Safari.
     * Firefox is currently debating the implementation. Content sniffing is a method browsers use to attempt to
     * determine the 'real' content type of a response by looking at the content itself, instead of the response
     * header's content-type value. By returning X-Content-Type-Options: nosniff, certain elements will only load
     * external resources if their content-type matches what is expected.
     * <ul>
     * <li>https://www.veracode.com/blog/2014/03/guidelines-for-setting-security-headers</li>
     * </ul>
     * Valid settings:
     * <ul>
     * <li>nosniff - only possible valid for this header>/li>
     * <li>empty - to disable</li>
     * </ul>
     */
    private String xContentTypeOptions;

    /**
     * the X-Frame-Options HTTP header<br/>
     * This header is for configuring which sites are allowed to frame the loaded resource. Its primary purpose is to
     * protect against UI redressing style attacks. Internet Explorer has supported the ALLOW-FROM directive since IE8
     * and Firefox from 18. Both Chrome and Safari do not support ALLOW-FROM, however WebKit is currently discussing it.
     * <br/>
     * Valid Settings:
     * <ul>
     * <li>DENY - Denies any resource (local or remote) from attempting to frame the resource that also supplied the
     * X-Frame-Options header.</li>
     * <li>SAMEORIGIN - Allows only resources which are apart of the Same Origin Policy to frame the protected resource.
     * </li>
     * <li>ALLOW-FROM http://www.example.com - Allows a single serialized-origin (must have scheme) to frame the
     * protected resource. This is only valid in Internet Explorer and Firefox. The default of other browsers is to
     * allow any origin (as if X-Frame-Options was not set).</li>
     * </ul>
     */
    private String xFrameOptions;

    /** if DIF will sanitize every parameter it receives for XSS malicious content */
    private Boolean xssParameterSanitization;

    /**
     * the X-XSS-Protection HTTP header to prevent Cross-site scripting:<br/>
     * This response header can be used to configure a user-agent's built in reflective XSS protection. Currently, only
     * Microsoft's Internet Explorer, Google Chrome and Safari (WebKit) support this header.
     * <ul>
     * <li>http://en.wikipedia.org/wiki/Cross-site_scripting</li>
     * <li>https://www.veracode.com/blog/2014/03/guidelines-for-setting-security-headers</li>
     * </ul>
     * Valid Settings:
     * <ul>
     * <li>0 - Disables the XSS Protections offered by the user-agent.</li>
     * <li>1 - Enables the XSS Protections</li>
     * <li>1; mode=block - Enables XSS protections and instructs the user-agent to block the response in the event that
     * script has been inserted from user input, instead of sanitizing.</li>
     * <li>1; report=http://site.com/report - A Chrome and WebKit only directive that tells the user-agent to report
     * potential XSS attacks to a single URL. Data will be POST'd to the report URL in JSON format.</li/
     * </ul>
     */
    private String xssProtection;

    /**
     * Applies the configured security configurations
     * 
     * @param httpResponse
     *            the response object to apply security configs
     */
    public void applyHTTPSecurityHeaders(HttpServletResponse httpResponse)
    {
        if (StringUtils.isNotBlank(this.getXssProtection()))
            httpResponse.setHeader("X-XSS-Protection", this.getXssProtection());
        if (StringUtils.isNotBlank(this.getXssProtection()))
            httpResponse.setHeader("X-Content-Type-Options", this.getxContentTypeOptions());
        if (StringUtils.isNotBlank(this.getXssProtection()))
            httpResponse.setHeader("X-Frame-Options", this.getxFrameOptions());
    }

    /**
     * Inspector for the 'securityViolationMailReportDestination' attribute.
     * 
     * @return the securityViolationMailReportDestination value
     */
    @ConfigDefault("tech-support@digitalis.pt")
    public String getSecurityViolationMailReportDestination()
    {
        return securityViolationMailReportDestination;
    }

    /**
     * Inspector for the 'securityViolationMailReportTestingDestination' attribute.
     * 
     * @return the securityViolationMailReportTestingDestination value
     */
    public String getSecurityViolationMailReportTestingDestination()
    {
        return securityViolationMailReportTestingDestination;
    }

    /**
     * Inspector for the 'validateRequestAuthorizations' attribute.
     * 
     * @return the validateRequestAuthorizations value
     */
    @ConfigDefault("true")
    public Boolean getValidateRequestAuthorizations()
    {
        return validateRequestAuthorizations;
    }

    /**
     * Inspector for the 'xContentTypeOptions' attribute.
     * 
     * @return the xContentTypeOptions value
     */
    @ConfigDefault("nosniff")
    public String getxContentTypeOptions()
    {
        return xContentTypeOptions;
    }

    /**
     * Inspector for the 'xFrameOptions' attribute.
     * 
     * @return the xFrameOptions value
     */
    @ConfigDefault("DENY")
    public String getxFrameOptions()
    {
        return xFrameOptions;
    }

    /**
     * Inspector for the 'xssParameterSanitization' attribute.
     * 
     * @return the xssParameterSanitization value
     */
    @ConfigDefault("true")
    public Boolean getXssParameterSanitization()
    {
        return xssParameterSanitization;
    }

    /**
     * Inspector for the 'xssProtection' attribute.
     * 
     * @return the xssProtection value
     */
    @ConfigDefault("1; mode=block")
    public String getXssProtection()
    {
        return xssProtection;
    }

    /**
     * Modifier for the 'securityViolationMailReportDestination' attribute.
     * 
     * @param securityViolationMailReportDestination
     *            the new securityViolationMailReportDestination value to set
     */
    public void setSecurityViolationMailReportDestination(String securityViolationMailReportDestination)
    {
        this.securityViolationMailReportDestination = securityViolationMailReportDestination;
    }

    /**
     * Modifier for the 'securityViolationMailReportTestingDestination' attribute.
     * 
     * @param securityViolationMailReportTestingDestination
     *            the new securityViolationMailReportTestingDestination value to set
     */
    public void setSecurityViolationMailReportTestingDestination(String securityViolationMailReportTestingDestination)
    {
        this.securityViolationMailReportTestingDestination = securityViolationMailReportTestingDestination;
    }

    /**
     * Modifier for the 'validateRequestAuthorizations' attribute.
     * 
     * @param validateRequestAuthorizations
     *            the new validateRequestAuthorizations value to set
     */
    public void setValidateRequestAuthorizations(Boolean validateRequestAuthorizations)
    {
        this.validateRequestAuthorizations = validateRequestAuthorizations;
    }

    /**
     * Modifier for the 'xContentTypeOptions' attribute.
     * 
     * @param xContentTypeOptions
     *            the new xContentTypeOptions value to set
     */
    public void setxContentTypeOptions(String xContentTypeOptions)
    {
        this.xContentTypeOptions = xContentTypeOptions;
    }

    /**
     * Modifier for the 'xFrameOptions' attribute.
     * 
     * @param xFrameOptions
     *            the new xFrameOptions value to set
     */
    public void setxFrameOptions(String xFrameOptions)
    {
        this.xFrameOptions = xFrameOptions;
    }

    /**
     * Modifier for the 'xssParameterSanitization' attribute.
     * 
     * @param xssParameterSanitization
     *            the new xssParameterSanitization value to set
     */
    public void setXssParameterSanitization(Boolean xssParameterSanitization)
    {
        this.xssParameterSanitization = xssParameterSanitization;
    }

    /**
     * Modifier for the 'xssProtection' attribute.
     * 
     * @param xssProtection
     *            the new xssProtection value to set
     */
    public void setXssProtection(String xssProtection)
    {
        this.xssProtection = xssProtection;
    }
}
