/**
 * 2010, Digitalis Informatica. All rights reserved. Distribuicao e Gestao de Informatica, Lda. Estrada de Paco de Arcos
 * num.9 - Piso -1 2780-666 Paco de Arcos Telefone: (351) 21 4408990 Fax: (351) 21 4408999 http://www.digitalis.pt
 */

package pt.digitalis.dif.rules.objects;

import java.util.HashMap;
import java.util.Map;

import pt.digitalis.dif.ioc.DIFIoCRegistry;
import pt.digitalis.dif.rules.IFlowManager;
import pt.digitalis.dif.rules.IPrivateFlowManager;
import pt.digitalis.dif.rules.IPrivateRulesManager;
import pt.digitalis.dif.rules.IRulesManager;
import pt.digitalis.dif.rules.annotations.ContextParameter;
import pt.digitalis.dif.rules.annotations.Flow;
import pt.digitalis.dif.rules.annotations.RuleGroup;
import pt.digitalis.dif.rules.condegen.IContextParameters;
import pt.digitalis.dif.rules.condegen.RuleExceptionThrower;
import pt.digitalis.dif.rules.exceptions.MissingContextException;
import pt.digitalis.dif.rules.exceptions.rules.RuleException;
import pt.digitalis.dif.rules.exceptions.rules.RuleGroupException;
import pt.digitalis.dif.rules.objects.rules.AbstractRuleGroup;
import pt.digitalis.dif.rules.objects.rules.RuleGroupDescriptor;

/**
 * Template class for {@link RuleGroup} or {@link Flow} classes.
 * 
 * @author Pedro Viegas <a href="mailto:pviegas@digitalis.pt">pviegas@digitalis.pt</a><br/>
 * @created 2010/07/22
 */
public abstract class AbstractContribution {

    /** the cached rule group instances to use for condition rules */
    private Map<String, AbstractRuleGroup> cachedRuleGroupInstances = new HashMap<String, AbstractRuleGroup>();

    /** the Flow Manager instance */
    protected IPrivateFlowManager flowManager = null;

    /** the Rule Manager instance */
    protected IPrivateRulesManager ruleManager = null;

    /**
     * Helper inner method for building a parameter map to call Rules
     * 
     * @param parameterNames
     *            all parameter names comma separated
     * @param values
     *            the values to assign in the map, binded by position in the parameterNames attribute
     * @return the parameter Map
     */
    protected Map<String, Object> buildParameterMap(String parameterNames, Object... values)
    {
        Map<String, Object> result = new HashMap<String, Object>();
        String[] params = parameterNames.split(",");
        int index = 0;

        for (String paramName: params)
            result.put(paramName, values[index++]);

        return result;
    }

    /**
     * Inspector for the 'ruleManager' attribute.
     * 
     * @return the ruleManager value
     */
    protected IPrivateFlowManager getFlowManager()
    {
        if (flowManager == null)
            flowManager = (IPrivateFlowManager) DIFIoCRegistry.getRegistry().getImplementation(IFlowManager.class);

        return flowManager;
    }

    /**
     * @return the instance map of parameters, defined by {@link ContextParameter}
     */
    abstract protected Map<String, Object> getParameterValues();

    /**
     * Get the RouleGroup instance
     * 
     * @param <T>
     *            The RoleGroup
     * @param ruleGroupID
     *            the unique RoleGroup id
     * @return RuleGroup instance
     * @throws RuleGroupException
     *             if a RuleGroupException exception occurrs
     * @throws MissingContextException
     *             if a MissingContextException occurrs
     */
    @SuppressWarnings("unchecked")
    protected <T extends AbstractRuleGroup> T getRuleGroup(String ruleGroupID) throws MissingContextException,
            RuleGroupException
    {
        return (T) getRuleGroupCachedInstance(ruleGroupID);
    }

    /**
     * Get the {@link RuleGroup} cached instance. For reuse purposes in the context of this flow instance
     * 
     * @param <T>
     * @param clazz
     *            the ruleGroup definition class
     * @return the cached {@link RuleGroup} instance, or a newly created and cached one
     * @throws RuleGroupException
     * @throws MissingContextException
     */
    @SuppressWarnings("unchecked")
    public <T extends AbstractRuleGroup> T getRuleGroupCachedInstance(Class<T> clazz) throws RuleGroupException,
            MissingContextException
    {
        RuleGroupDescriptor ruleGroupDescriptor = getRuleManager().getRuleGroup(clazz);

        AbstractRuleGroup ruleGroup = cachedRuleGroupInstances.get(ruleGroupDescriptor.getUniqueName());

        if (ruleGroup == null)
        {
            ruleGroup = getRuleManager().getRuleGroupInstance(clazz, getParameterValues());

            cachedRuleGroupInstances.put(ruleGroupDescriptor.getUniqueName(), ruleGroup);
        }
        else
        {
            ((IContextParameters) ruleGroup).initializeContext(getParameterValues());
        }

        return (T) ruleGroup;

    }

    /**
     * Get the {@link RuleGroup} cached instance. For reuse purposes in the context of this flow instance
     * 
     * @param <T>
     * @param ruleGroupID
     *            the ruleGroup unique name
     * @return the cached {@link RuleGroup} instance, or a newly created and cached one
     * @throws RuleGroupException
     * @throws MissingContextException
     */
    @SuppressWarnings("unchecked")
    protected <T extends AbstractRuleGroup> T getRuleGroupCachedInstance(String ruleGroupID)
            throws MissingContextException, RuleGroupException
    {
        T ruleGroup = (T) cachedRuleGroupInstances.get(ruleGroupID);

        if (ruleGroup == null)
        {
            RuleGroupDescriptor ruleGroupDescriptor = getRuleManager().getRuleGroup(ruleGroupID);

            Class<T> clazz = (Class<T>) ruleGroupDescriptor.getClazz();
            ruleGroup = getRuleManager().getRuleGroupInstance(clazz, getParameterValues());

            cachedRuleGroupInstances.put(ruleGroupID, ruleGroup);
        }

        return ruleGroup;
    }

    /**
     * Inspector for the 'ruleManager' attribute.
     * 
     * @return the ruleManager value
     */
    protected IPrivateRulesManager getRuleManager()
    {
        if (ruleManager == null)
            ruleManager = (IPrivateRulesManager) DIFIoCRegistry.getRegistry().getImplementation(IRulesManager.class);

        return ruleManager;
    }

    /**
     * @return the {@link RuleException} thrower utility
     */
    public RuleExceptionThrower getThrower()
    {
        return RuleExceptionThrower.getInstance();
    }

    /**
     * Modifier for the 'flowManager' attribute.
     * 
     * @param flowManager
     *            the new flowManager value to set
     */
    public void setFlowManager(IPrivateFlowManager flowManager)
    {
        this.flowManager = flowManager;
    }

    /**
     * Modifier for the 'ruleManager' attribute.
     * 
     * @param ruleManager
     *            the new ruleManager value to set
     */
    public void setRuleManager(IPrivateRulesManager ruleManager)
    {
        this.ruleManager = ruleManager;
    }

}
