/**
 * 2010, Digitalis Informatica. All rights reserved. Distribuicao e Gestao de Informatica, Lda. Estrada de Paco de Arcos
 * num.9 - Piso -1 2780-666 Paco de Arcos Telefone: (351) 21 4408990 Fax: (351) 21 4408999 http://www.digitalis.pt
 */

package pt.digitalis.dif.rules.objects.flow;

import java.util.ArrayList;
import java.util.List;

import pt.digitalis.dif.dem.annotations.AnnotationTags;
import pt.digitalis.dif.rules.IFlowManager;
import pt.digitalis.dif.rules.objects.AbstractClassDescriptor;
import pt.digitalis.dif.rules.objects.AbstractContribution;
import pt.digitalis.dif.rules.objects.AbstractMethodDescriptor;

/**
 * Defines a Flow
 * 
 * @author Pedro Viegas <a href="mailto:pviegas@digitalis.pt">pviegas@digitalis.pt</a><br/>
 * @created 2010/07/27
 */
public class FlowDescriptor extends AbstractClassDescriptor {

    /** The action IDs of the flow */
    private List<String> actions;

    /** the container flow manager */
    private final IFlowManager flowManager;

    /** the ID of the parent group */
    private String parentGroupName;

    /**
     * Default constructor
     * 
     * @param clazz
     * @param name
     * @param flowManager
     * @param parentGroupName
     */
    public FlowDescriptor(IFlowManager flowManager, Class<?> clazz, String name, String parentGroupName)
    {
        if (AnnotationTags.NONE.equals(parentGroupName))
            parentGroupName = null;

        if (AnnotationTags.GENERATE_ID.equals(name))
            name = clazz.getSimpleName();

        setName(name.toLowerCase());
        setClazz(clazz);

        this.flowManager = flowManager;
        this.parentGroupName = parentGroupName;
    }

    /**
     * Inspector for the 'flow' attribute.
     * 
     * @return the flow value
     */
    public List<String> getActionIDs()
    {
        return actions;
    }

    /**
     * @return the flow actions
     */
    public List<FlowActionDescriptor> getActions()
    {
        List<String> actionIDs = getActionIDs();
        List<FlowActionDescriptor> actions = new ArrayList<FlowActionDescriptor>();

        if (actionIDs != null)
        {
            for (String actionID: actionIDs)
                actions.add(flowManager.getFlowAction(getName(), actionID));
        }

        return actions;
    }

    /**
     * @see pt.digitalis.dif.rules.objects.AbstractClassDescriptor#getChildIDs()
     */
    @Override
    public List<String> getChildIDs()
    {
        return getActionIDs();
    }

    /**
     * @see pt.digitalis.dif.rules.objects.AbstractClassDescriptor#getChilds()
     */
    @Override
    public List<? extends AbstractMethodDescriptor> getChilds()
    {
        return getActions();
    }

    /**
     * @return the group rules
     */
    public List<FlowDescriptor> getFlowGroups()
    {
        List<String> flowGroupIDs = flowManager.getRuleGroupChildrenGroups(getUniqueName().toLowerCase());
        List<FlowDescriptor> flows = new ArrayList<FlowDescriptor>();

        if (flowGroupIDs != null)
        {
            for (String flowGroupID: flowGroupIDs)
                flows.add(flowManager.getFlow(getUniqueName() + "." + flowGroupID));
        }

        return flows;
    }

    /**
     * @see pt.digitalis.dif.rules.objects.AbstractClassDescriptor#getInstanceBaseImplementationClass()
     */
    @Override
    public Class<? extends AbstractContribution> getInstanceBaseImplementationClass()
    {
        return AbstractFlow.class;
    }

    /**
     * Inspector for the 'parentGroupName' attribute.
     * 
     * @return the parentGroupName value
     */
    public String getParentGroupName()
    {
        return parentGroupName;
    }

    /**
     * Inspector for the 'name' attribute.
     * 
     * @return the name value
     */
    public String getUniqueName()
    {
        if (parentGroupName == null)
            return getName();
        else
            return parentGroupName + "." + getName();
    }

    /**
     * Modifier for the 'actions' attribute.
     * 
     * @param actions
     *            the new actions value to set
     */
    public void setActionIDs(List<String> actions)
    {
        this.actions = actions;
    }

    /**
     * Modifier for the 'parentGroupName' attribute.
     * 
     * @param parentGroupName
     *            the new parentGroupName value to set
     */
    public void setParentGroupName(String parentGroupName)
    {
        this.parentGroupName = parentGroupName;
    }
}
