/**
 * 2010, Digitalis Informatica. All rights reserved. Distribuicao e Gestao de Informatica, Lda. Estrada de Paco de Arcos
 * num.9 - Piso -1 2780-666 Paco de Arcos Telefone: (351) 21 4408990 Fax: (351) 21 4408999 http://www.digitalis.pt
 */

package pt.digitalis.dif.rules.objects.rules;

import java.util.ArrayList;
import java.util.List;

import pt.digitalis.dif.dem.annotations.AnnotationTags;
import pt.digitalis.dif.rules.IRulesManager;
import pt.digitalis.dif.rules.objects.AbstractClassDescriptor;
import pt.digitalis.dif.rules.objects.AbstractContribution;
import pt.digitalis.dif.rules.objects.AbstractMethodDescriptor;

/**
 * Defines a Group of Rules
 * 
 * @author Pedro Viegas <a href="mailto:pviegas@digitalis.pt">pviegas@digitalis.pt</a><br/>
 * @created 2010/07/21
 */
public class RuleGroupDescriptor extends AbstractClassDescriptor {

    /** the ID of the parent group */
    private String parentGroupName;

    /** The rules IDs of the group's rules */
    private List<String> rules;

    /** the container rule manager */
    private final IRulesManager rulesManager;

    /**
     * Default constructor
     * 
     * @param clazz
     * @param name
     * @param parentGroupName
     * @param rulesManager
     */
    public RuleGroupDescriptor(IRulesManager rulesManager, Class<?> clazz, String name, String parentGroupName)
    {

        if (AnnotationTags.NONE.equals(parentGroupName))
            parentGroupName = null;

        if (AnnotationTags.GENERATE_ID.equals(name))
            name = clazz.getSimpleName();

        setClazz(clazz);
        setName(name.toLowerCase());

        if (parentGroupName != null)
        {
            this.parentGroupName = parentGroupName.toLowerCase();
        }
        this.rulesManager = rulesManager;
    }

    /**
     * @see pt.digitalis.dif.rules.objects.AbstractClassDescriptor#getChildIDs()
     */
    @Override
    public List<String> getChildIDs()
    {
        return getRuleIDs();
    }

    /**
     * @see pt.digitalis.dif.rules.objects.AbstractClassDescriptor#getChilds()
     */
    @Override
    public List<? extends AbstractMethodDescriptor> getChilds()
    {
        return getRules();
    }

    /**
     * @see pt.digitalis.dif.rules.objects.AbstractClassDescriptor#getInstanceBaseImplementationClass()
     */
    @Override
    public Class<? extends AbstractContribution> getInstanceBaseImplementationClass()
    {
        return AbstractRuleGroup.class;
    }

    /**
     * @return the parent group
     */
    public RuleGroupDescriptor getParentGroup()
    {
        return rulesManager.getRuleGroup(getParentGroupName().toLowerCase());
    }

    /**
     * Inspector for the 'parentGroupName' attribute.
     * 
     * @return the parentGroupName value
     */
    public String getParentGroupName()
    {
        return parentGroupName;
    }

    /**
     * @return the group rules
     */
    public List<RuleGroupDescriptor> getRuleGroups()
    {
        List<String> ruleGroupIDs = rulesManager.getRuleGroupChildrenGroups(getUniqueName().toLowerCase());
        List<RuleGroupDescriptor> ruleGroups = new ArrayList<RuleGroupDescriptor>();

        if (ruleGroupIDs != null)
        {
            for (String ruleGroupID: ruleGroupIDs)
                ruleGroups.add(rulesManager.getRuleGroup(getUniqueName() + "." + ruleGroupID));
        }

        return ruleGroups;
    }

    /**
     * Inspector for the 'rules' attribute.
     * 
     * @return the rules value
     */
    public List<String> getRuleIDs()
    {
        return rules;
    }

    /**
     * @return the group rules
     */
    public List<RuleDescriptor> getRules()
    {
        List<String> ruleIDs = getRuleIDs();
        List<RuleDescriptor> rules = new ArrayList<RuleDescriptor>();

        if (ruleIDs != null)
        {
            for (String ruleID: ruleIDs)
                rules.add(rulesManager.getRule(getUniqueName(), ruleID));
        }

        return rules;
    }

    /**
     * Inspector for the 'name' attribute.
     * 
     * @return the name value
     */
    public String getUniqueName()
    {
        if (parentGroupName == null)
            return getName();
        else
            return parentGroupName + "." + getName();
    }

    /**
     * Modifier for the 'parentGroupName' attribute.
     * 
     * @param parentGroupName
     *            the new parentGroupName value to set
     */
    public void setParentGroupName(String parentGroupName)
    {
        this.parentGroupName = parentGroupName;
    }

    /**
     * Modifier for the 'rules' attribute.
     * 
     * @param rules
     *            the new rules value to set
     */
    public void setRuleIDs(List<String> rules)
    {
        this.rules = rules;
    }
}
