/**
 * 2010, Digitalis Informatica. All rights reserved. Distribuicao e Gestao de Informatica, Lda. Estrada de Paco de Arcos
 * num.9 - Piso -1 2780-666 Paco de Arcos Telefone: (351) 21 4408990 Fax: (351) 21 4408999 http://www.digitalis.pt
 */

package pt.digitalis.dif.rules;

import java.lang.reflect.InvocationTargetException;
import java.util.HashMap;
import java.util.Map;
import java.util.Map.Entry;

import pt.digitalis.dif.dem.annotations.AnnotationTags;
import pt.digitalis.dif.ioc.DIFIoCRegistry;
import pt.digitalis.dif.rules.annotations.Flow;
import pt.digitalis.dif.rules.condegen.IContextParameters;
import pt.digitalis.dif.rules.exceptions.MissingContextException;
import pt.digitalis.dif.rules.exceptions.TooManyContextParamsException;
import pt.digitalis.dif.rules.exceptions.flow.FlowActionDoesNotExistException;
import pt.digitalis.dif.rules.exceptions.flow.FlowActionException;
import pt.digitalis.dif.rules.exceptions.flow.FlowException;
import pt.digitalis.dif.rules.exceptions.flow.InvalidFlow;
import pt.digitalis.dif.rules.objects.flow.AbstractFlow;
import pt.digitalis.dif.rules.objects.flow.FlowActionDescriptor;
import pt.digitalis.dif.rules.objects.flow.FlowActionResult;
import pt.digitalis.dif.rules.objects.flow.FlowDescriptor;
import pt.digitalis.dif.utils.logging.DIFLogger;

/**
 * Base implementation for the an {@link IFlowManager}
 * 
 * @author Pedro Viegas <a href="mailto:pviegas@digitalis.pt">pviegas@digitalis.pt</a><br/>
 * @created 2010/07/27
 */
abstract public class AbstractFlowManager extends AbstractManager implements IFlowManager, IPrivateFlowManager {

    /** Could not instantiate the Flow class error message */
    private final String FLOW_INSTANTIATE_ERROR = "Could not instanciate the Flow class";

    /** the rules manager */
    protected IRulesManager rulesManager;

    /**
     * @param flowName
     * @return a string with the flow definition
     */
    private String dumpFlow(String flowName)
    {
        StringBuffer buffer = new StringBuffer();
        FlowDescriptor flow = getFlow(flowName);

        buffer.append("    [" + flow.getName() + "]\n");

        // Add all flow actions
        for (FlowActionDescriptor flowAction: flow.getActions())
        {
            buffer.append("        - ");
            buffer.append(flowAction.getName());
            buffer.append("(");
            boolean first = true;

            for (String paramName: flowAction.getParameters())
            {
                if (first)
                    first = false;
                else
                    buffer.append(",");

                buffer.append(paramName);
            }

            buffer.append(")\n");

            String descPrefix = "            ";
            if (DIFLogger.getLogger().isDebugEnabled())
            {
                if (flowAction.getConditionRuleName() != null)
                    buffer.append(descPrefix + "(Execution condition rule: " + flowAction.getConditionRuleName()
                            + ")\n");
                buffer.append(descPrefix + flowAction.getDescription().replaceAll("\\n", descPrefix) + "\n\n.");
            }
        }

        return buffer.toString();
    }

    /**
     * @see pt.digitalis.dif.rules.AbstractManager#dumpRegistryToLog()
     */
    @Override
    protected void dumpRegistryToLog()
    {
        if (DIFLogger.getLogger().isDebugEnabled())
        {
            StringBuffer buffer = new StringBuffer();

            buffer.append("\n\n");
            buffer.append("DIF Flows ClassesRegistry:\n");
            buffer.append("------------------------------\n\n");

            for (FlowDescriptor flow: this.getFlows())
                buffer.append(dumpFlow(flow.getUniqueName()));

            buffer.append("\n\n");

            DIFLogger.getLogger().debug(buffer.toString());
        }
    }

    /**
     * @see pt.digitalis.dif.rules.IPrivateFlowManager#flowExecute(pt.digitalis.dif.rules.objects.flow.AbstractFlow,
     *      java.lang.String)
     */
    public FlowActionResult<?> flowExecute(AbstractFlow flowInstance, String uniqueActionID) throws FlowActionException
    {
        return flowExecute(flowInstance, uniqueActionID, new HashMap<String, Object>());
    }

    /**
     * @see pt.digitalis.dif.rules.IPrivateFlowManager#flowExecute(pt.digitalis.dif.rules.objects.flow.AbstractFlow,
     *      java.lang.String, java.util.Map)
     */
    public FlowActionResult<?> flowExecute(AbstractFlow flowInstance, String uniqueActionID,
            Map<String, Object> parameters) throws FlowActionException
    {
        FlowActionDescriptor flowAction = getFlowAction(uniqueActionID);
        FlowActionResult<?> result;

        try
        {
            result = (FlowActionResult<?>) flowAction.getMethodToImplementation().invoke(flowInstance,
                    buildExecutionArguments(flowAction, parameters));
        }
        catch (IllegalArgumentException e)
        {
            throw new FlowActionException(flowAction, e);
        }
        catch (IllegalAccessException e)
        {
            throw new FlowActionException(flowAction, e);
        }
        catch (InvocationTargetException e)
        {
            throw new FlowActionException(flowAction, e.getTargetException());
        }

        return result;
    }

    /**
     * @see pt.digitalis.dif.rules.IPrivateFlowManager#flowExecute(pt.digitalis.dif.rules.objects.flow.AbstractFlow,
     *      java.lang.String, java.lang.String)
     */
    public FlowActionResult<?> flowExecute(AbstractFlow flowInstance, String flowID, String actionID)
            throws FlowActionException
    {
        return flowExecute(flowInstance, flowID, actionID, null);
    }

    /**
     * @see pt.digitalis.dif.rules.IPrivateFlowManager#flowExecute(pt.digitalis.dif.rules.objects.flow.AbstractFlow,
     *      java.lang.String, java.lang.String, java.util.Map)
     */
    public FlowActionResult<?> flowExecute(AbstractFlow flowInstance, String flowID, String actionID,
            Map<String, Object> parameters) throws FlowActionException
    {
        return flowExecute(flowInstance, flowID + "." + actionID, parameters);
    }

    /**
     * @see pt.digitalis.dif.rules.IFlowManager#flowExecute(java.lang.String, java.util.Map)
     */
    public FlowActionResult<?> flowExecute(String uniqueActionID, Map<String, Object> parameters) throws FlowException,
            FlowActionException, MissingContextException
    {
        return flowExecute(getFlowInstance(uniqueActionID, parameters), uniqueActionID, parameters);
    }

    /**
     * @see pt.digitalis.dif.rules.IFlowManager#flowExecute(java.lang.String, java.lang.String, java.util.Map)
     */
    public FlowActionResult<?> flowExecute(String flowID, String actionID, Map<String, Object> parameters)
            throws FlowException, FlowActionException, MissingContextException
    {
        return flowExecute(flowID + "." + actionID, parameters);
    }

    /**
     * @see pt.digitalis.dif.rules.IFlowManager#getFlow(java.lang.Class)
     */
    @SuppressWarnings("unchecked")
    public <T extends AbstractFlow> FlowDescriptor getFlow(Class<T> clazz) throws InvalidFlow
    {
        // Gets the class RuleGroup declaration annotation
        Flow flowDef = clazz.getAnnotation(Flow.class);

        if (flowDef == null)
        {
            // Not found, test if it's an enriched class, and thus the class is it's super class
            Class<T> superClazz = (Class<T>) clazz.getSuperclass();
            flowDef = superClazz.getAnnotation(Flow.class);

            if (flowDef == null)
                throw new InvalidFlow(clazz);
            else
                clazz = superClazz;
        }

        String flowID = flowDef.name();

        if (AnnotationTags.GENERATE_ID.equals(flowID))
            flowID = clazz.getSimpleName();

        if (!AnnotationTags.NONE.equals(flowDef.parentGroup()))
            flowID = flowDef.parentGroup() + "." + flowID;

        FlowDescriptor flow = getFlow(flowID);

        if (flow == null)
            throw new InvalidFlow(clazz);
        else
            return flow;
    }

    /**
     * @see pt.digitalis.dif.rules.IFlowManager#getFlowAction(java.lang.String, java.lang.String)
     */
    public FlowActionDescriptor getFlowAction(String flowID, String actionID)
    {
        return getFlowAction(flowID + "." + actionID);
    }

    /**
     * @see pt.digitalis.dif.rules.IFlowManager#getFlowInstance(java.lang.Class)
     */
    public <T extends AbstractFlow> T getFlowInstance(Class<T> clazz) throws FlowException, MissingContextException
    {
        return getFlowInstance(clazz, new HashMap<String, Object>());
    }

    /**
     * @see pt.digitalis.dif.rules.IFlowManager#getFlowInstance(java.lang.Class, java.util.Map)
     */
    public <T extends AbstractFlow> T getFlowInstance(Class<T> clazz, Map<String, Object> parameters)
            throws FlowException, MissingContextException
    {
        // Get the RuleGroup from the class declaration
        FlowDescriptor flow = getFlow(clazz);

        try
        {
            @SuppressWarnings("unchecked")
            T instance = (T) flow.getClazz().newInstance();
            instance.setFlowManager(this);
            initializeInstanceContext(instance, flow, parameters);

            return instance;
        }
        catch (InstantiationException e)
        {
            throw new InvalidFlow(clazz, e);
        }
        catch (IllegalAccessException e)
        {
            throw new InvalidFlow(clazz, e);
        }
    }

    /**
     * @see pt.digitalis.dif.rules.IFlowManager#getFlowInstance(java.lang.Class, java.lang.Object[])
     */
    public <T extends AbstractFlow> T getFlowInstance(Class<T> clazz, Object... parameters) throws FlowException,
            TooManyContextParamsException, MissingContextException
    {
        FlowDescriptor flow = getFlow(clazz);
        return getFlowInstance(clazz, buildParameterMap(flow, flow.getContextParameters(), parameters));
    }

    /**
     * Creates a new instance of a flow for the given action
     * 
     * @param flowActionID
     *            the flow action
     * @return the Flow instance
     * @throws FlowException
     *             if the flow is not valid
     * @throws FlowActionException
     *             if the group instance could not be created
     * @throws MissingContextException
     */
    protected AbstractFlow getFlowInstance(String flowActionID) throws FlowException, FlowActionException,
            MissingContextException
    {
        return getFlowInstance(flowActionID.toLowerCase(), null);
    }

    /**
     * Creates a new instance of a flow for the given action
     * 
     * @param flowActionID
     *            the flow action
     * @param parameters
     *            the context parameters to pass to the RuleGroup instance
     * @return the Flow instance
     * @throws FlowException
     *             if the flow is not valid
     * @throws FlowActionException
     *             if the group instance could not be created
     * @throws MissingContextException
     */
    protected AbstractFlow getFlowInstance(String flowActionID, Map<String, Object> parameters) throws FlowException,
            FlowActionException, MissingContextException
    {
        FlowActionDescriptor flowAction = getFlowAction(flowActionID);

        if (flowAction == null)
            throw new FlowActionDoesNotExistException(flowActionID);
        else
        {
            Class<?> clazz = flowAction.getFlow().getClazz();

            try
            {
                FlowDescriptor flow = flowAction.getFlow();
                AbstractFlow instance = (AbstractFlow) flow.getClazz().newInstance();
                HashMap<String, Object> parsedParameters = new HashMap<String, Object>();

                // If the RuleGroup has context parameters, validate and initialize them
                if (!flow.getContextParameters().isEmpty())
                {
                    // LowerCase them to match
                    if (parameters != null)
                        for (Entry<String, Object> param: parameters.entrySet())
                            parsedParameters.put(param.getKey().toLowerCase(), param.getValue());

                    // No parameters, passed! Cannot continue.
                    if (parameters == null)
                        throw new MissingContextException(flow);

                    // Call initialization method with parameters
                    ((IContextParameters) instance).initializeContext(parsedParameters);
                }

                return instance;
            }
            catch (InstantiationException e)
            {
                throw new FlowActionException(flowAction, clazz.getName() + ": " + FLOW_INSTANTIATE_ERROR, e);
            }
            catch (IllegalAccessException e)
            {
                throw new FlowActionException(flowAction, clazz.getName() + ": " + FLOW_INSTANTIATE_ERROR, e);
            }
        }
    }

    /**
     * Inspector for the 'rulesManager' attribute.
     * 
     * @return the rulesManager value
     * @throws Exception
     */
    protected IRulesManager getRulesManager() throws Exception
    {
        if (rulesManager == null)
        {
            rulesManager = DIFIoCRegistry.getRegistry().getImplementation(IRulesManager.class);
            rulesManager.initialize();
        }

        return rulesManager;
    }
}
