package pt.digitalis.dif.rules;

import java.lang.reflect.InvocationTargetException;
import java.util.HashMap;
import java.util.Map;
import java.util.Map.Entry;

import pt.digitalis.dif.dem.annotations.AnnotationTags;
import pt.digitalis.dif.rules.annotations.RuleGroup;
import pt.digitalis.dif.rules.condegen.IContextParameters;
import pt.digitalis.dif.rules.exceptions.MissingContextException;
import pt.digitalis.dif.rules.exceptions.TooManyContextParamsException;
import pt.digitalis.dif.rules.exceptions.rules.InvalidRuleGroup;
import pt.digitalis.dif.rules.exceptions.rules.RuleDoesNotExistException;
import pt.digitalis.dif.rules.exceptions.rules.RuleException;
import pt.digitalis.dif.rules.exceptions.rules.RuleGroupException;
import pt.digitalis.dif.rules.objects.rules.AbstractRuleGroup;
import pt.digitalis.dif.rules.objects.rules.RuleDescriptor;
import pt.digitalis.dif.rules.objects.rules.RuleGroupDescriptor;
import pt.digitalis.dif.rules.objects.rules.RuleResult;
import pt.digitalis.dif.utils.logging.DIFLogger;

/**
 * 2010, Digitalis Informatica. All rights reserved. Distribuicao e Gestao de Informatica, Lda. Estrada de Paco de Arcos
 * num.9 - Piso -1 2780-666 Paco de Arcos Telefone: (351) 21 4408990 Fax: (351) 21 4408999 http://www.digitalis.pt
 */

/**
 * Base implementation for the an {@link IRulesManager}
 * 
 * @author Pedro Viegas <a href="mailto:pviegas@digitalis.pt">pviegas@digitalis.pt</a><br/>
 * @created 2010/07/21
 */
public abstract class AbstractRulesManager extends AbstractManager implements IRulesManager, IPrivateRulesManager {

    /** Could not instantiate the Rule Group class error message */
    private final String RULE_GROUP_INSTANTIATE_ERROR = "Could not instanciate the Rule Group class";

    /**
     * @see pt.digitalis.dif.rules.AbstractManager#dumpRegistryToLog()
     */
    @Override
    protected void dumpRegistryToLog()
    {
        if (DIFLogger.getLogger().isDebugEnabled())
        {
            StringBuffer buffer = new StringBuffer();

            buffer.append("\n\n");
            buffer.append("DIF Rules Classes Registry:\n");
            buffer.append("------------------------------\n\n");

            for (RuleGroupDescriptor ruleGroup: this.getRootRuleGroups())
                buffer.append(dumpRuleGroup(ruleGroup.getUniqueName(), "    "));

            buffer.append("\n\n.");

            DIFLogger.getLogger().debug(buffer.toString());
        }
    }

    /**
     * @param ruleGroupUniqueName
     * @param prefix
     *            the current group prefix for rendering purposes
     * @return a string with the group definition
     */
    private String dumpRuleGroup(String ruleGroupUniqueName, String prefix)
    {
        StringBuffer buffer = new StringBuffer();
        RuleGroupDescriptor ruleGroup = getRuleGroup(ruleGroupUniqueName);

        buffer.append(prefix);
        buffer.append("[" + ruleGroup.getName() + "]\n");

        // Add all rules
        for (RuleDescriptor rule: ruleGroup.getRules())
        {
            buffer.append(prefix + "    - ");
            buffer.append(rule.getName());
            buffer.append(": " + rule.getType().toString() + "(");
            boolean first = true;

            for (String paramName: rule.getParameters())
            {
                if (first)
                    first = false;
                else
                    buffer.append(",");

                buffer.append(paramName);
            }

            buffer.append(")\n");

            String descPrefix = prefix + "        ";
            if (DIFLogger.getLogger().isDebugEnabled())
            {
                if (rule.getConditionRuleName() != null)
                    buffer.append(prefix + "        (Execution condition rule: " + rule.getConditionRuleName() + ")\n");
                buffer.append(descPrefix + rule.getDescription().replaceAll("\\n", descPrefix) + "\n\n");
            }
        }

        // Add all subgroups
        for (RuleGroupDescriptor childGroup: ruleGroup.getRuleGroups())
            buffer.append(dumpRuleGroup(childGroup.getUniqueName(), prefix + "    "));

        return buffer.toString();
    }

    /**
     * @see pt.digitalis.dif.rules.IPrivateRulesManager#evaluateRule(pt.digitalis.dif.rules.objects.rules.AbstractRuleGroup,
     *      java.lang.String, java.util.Map)
     */
    public boolean evaluateRule(AbstractRuleGroup ruleGroupInstance, String uniqueRuleID, Map<String, Object> parameters)
            throws RuleException, Throwable
    {
        RuleDescriptor rule = getRule(uniqueRuleID);

        if (rule == null)
            throw new RuleDoesNotExistException(uniqueRuleID);

        try
        {
            return (Boolean) rule.getMethodToImplementation().invoke(ruleGroupInstance,
                    buildExecutionArguments(rule, parameters));
        }
        catch (IllegalArgumentException e)
        {
            throw new RuleException(rule, e);
        }
        catch (IllegalAccessException e)
        {
            throw new RuleException(rule, e);
        }
        catch (InvocationTargetException e)
        {
            Throwable exception = e.getTargetException();

            for (Class<?> throwable: rule.getMethodToImplementation().getExceptionTypes())
                if (throwable.isAssignableFrom(exception.getClass()))
                    throw exception;

            throw new RuleException(rule, e);
        }
        catch (ClassCastException e)
        {
            throw new RuleException(rule, e);
        }
    }

    /**
     * @see pt.digitalis.dif.rules.IRulesManager#evaluateRule(java.lang.String, java.util.Map)
     */
    public boolean evaluateRule(String uniqueRuleID, Map<String, Object> parameters) throws RuleException,
            RuleGroupException, MissingContextException, Throwable
    {
        return evaluateRule(getRuleGroupInstance(uniqueRuleID, parameters), uniqueRuleID, parameters);
    }

    /**
     * @see pt.digitalis.dif.rules.IRulesManager#evaluateRule(java.lang.String, java.lang.String, java.util.Map)
     */
    public boolean evaluateRule(String groupID, String ruleID, Map<String, Object> parameters) throws RuleException,
            RuleGroupException, MissingContextException, Throwable
    {
        return evaluateRule(groupID + "." + ruleID, parameters);
    }

    /**
     * @see pt.digitalis.dif.rules.IPrivateRulesManager#executeRule(pt.digitalis.dif.rules.objects.rules.AbstractRuleGroup,
     *      java.lang.String, java.util.Map)
     */
    public RuleResult<?> executeRule(AbstractRuleGroup ruleGroupInstance, String uniqueRuleID,
            Map<String, Object> parameters) throws RuleException
    {
        RuleDescriptor rule = getRule(uniqueRuleID);
        RuleResult<?> result;

        try
        {
            result = (RuleResult<?>) rule.getMethodToImplementation().invoke(ruleGroupInstance,
                    buildExecutionArguments(rule, parameters));
        }
        catch (IllegalArgumentException e)
        {
            throw new RuleException(rule, e);
        }
        catch (IllegalAccessException e)
        {
            throw new RuleException(rule, e);
        }
        catch (InvocationTargetException e)
        {
            throw new RuleException(rule, e.getTargetException());
        }

        return result;
    }

    /**
     * @see pt.digitalis.dif.rules.IRulesManager#executeRule(java.lang.String, java.util.Map)
     */
    public RuleResult<?> executeRule(String uniqueRuleID, Map<String, Object> parameters) throws RuleException,
            RuleGroupException, MissingContextException
    {
        return executeRule(getRuleGroupInstance(uniqueRuleID, parameters), uniqueRuleID, parameters);
    }

    /**
     * @see pt.digitalis.dif.rules.IRulesManager#executeRule(java.lang.String, java.lang.String, java.util.Map)
     */
    public RuleResult<?> executeRule(String groupID, String ruleID, Map<String, Object> parameters)
            throws RuleException, RuleGroupException, MissingContextException
    {
        return executeRule(groupID + "." + ruleID, parameters);
    }

    /**
     * @see pt.digitalis.dif.rules.IRulesManager#getRule(java.lang.String, java.lang.String)
     */
    public RuleDescriptor getRule(String groupID, String ruleID)
    {
        return getRule(groupID + "." + ruleID);
    }

    /**
     * @see pt.digitalis.dif.rules.IRulesManager#getRuleGroup(java.lang.Class)
     */
    @SuppressWarnings("unchecked")
    public <T extends AbstractRuleGroup> RuleGroupDescriptor getRuleGroup(Class<T> clazz) throws InvalidRuleGroup
    {
        // Gets the class RuleGroup declaration annotation
        RuleGroup groupDef = clazz.getAnnotation(RuleGroup.class);

        if (groupDef == null)
        {
            // Not found, test if it's an enriched class, and thus the class is it's super class
            Class<T> superClazz = (Class<T>) clazz.getSuperclass();
            groupDef = superClazz.getAnnotation(RuleGroup.class);

            if (groupDef == null)
                throw new InvalidRuleGroup(clazz);
            else
                clazz = superClazz;
        }

        String groupID = groupDef.name();

        if (AnnotationTags.GENERATE_ID.equals(groupID))
            groupID = clazz.getSimpleName();

        if (!AnnotationTags.NONE.equals(groupDef.parentGroup()))
            groupID = groupDef.parentGroup() + "." + groupID;

        RuleGroupDescriptor group = getRuleGroup(groupID);

        if (group == null)
            throw new InvalidRuleGroup(clazz);
        else
            return group;
    }

    /**
     * @see pt.digitalis.dif.rules.IRulesManager#getRuleGroupInstance(java.lang.Class)
     */
    public <T extends AbstractRuleGroup> T getRuleGroupInstance(Class<T> clazz) throws RuleGroupException,
            MissingContextException
    {
        return getRuleGroupInstance(clazz, new HashMap<String, Object>());
    }

    /**
     * @see pt.digitalis.dif.rules.IRulesManager#getRuleGroupInstance(java.lang.Class, java.util.Map)
     */
    public <T extends AbstractRuleGroup> T getRuleGroupInstance(Class<T> clazz, Map<String, Object> parameters)
            throws RuleGroupException, MissingContextException
    {
        // Get the RuleGroup from the class declaration
        RuleGroupDescriptor group = getRuleGroup(clazz);

        try
        {
            @SuppressWarnings("unchecked")
            T instance = (T) group.getClazz().newInstance();
            instance.setRuleManager(this);
            initializeInstanceContext(instance, group, parameters);

            return instance;
        }
        catch (InstantiationException e)
        {
            throw new InvalidRuleGroup(clazz, e);
        }
        catch (IllegalAccessException e)
        {
            throw new InvalidRuleGroup(clazz, e);
        }
    }

    /**
     * @see pt.digitalis.dif.rules.IRulesManager#getRuleGroupInstance(java.lang.Class, java.lang.Object[])
     */
    public <T extends AbstractRuleGroup> T getRuleGroupInstance(Class<T> clazz, Object... parameters)
            throws RuleGroupException, TooManyContextParamsException, MissingContextException
    {
        RuleGroupDescriptor ruleGroup = getRuleGroup(clazz);

        return getRuleGroupInstance(clazz, buildParameterMap(ruleGroup, ruleGroup.getContextParameters(), parameters));
    }

    /**
     * Creates a new instance of a rule group for the given rule
     * 
     * @param uniqueRuleID
     *            the rule
     * @return the RuleGroup instance
     * @throws RuleException
     *             if the rule is not valid
     * @throws RuleGroupException
     *             if the group instance could not be created
     * @throws MissingContextException
     */
    protected AbstractRuleGroup getRuleGroupInstance(String uniqueRuleID) throws RuleException, RuleGroupException,
            MissingContextException
    {
        return getRuleGroupInstance(uniqueRuleID.toLowerCase(), null);
    }

    /**
     * Creates a new instance of a rule group for the given rule
     * 
     * @param uniqueRuleID
     *            the rule
     * @param parameters
     *            the context parameters to pass to the RuleGroup instance
     * @return the RuleGroup instance
     * @throws RuleException
     *             if the rule is not valid
     * @throws RuleGroupException
     *             if the group instance could not be created
     * @throws MissingContextException
     */
    protected AbstractRuleGroup getRuleGroupInstance(String uniqueRuleID, Map<String, Object> parameters)
            throws RuleException, RuleGroupException, MissingContextException
    {
        RuleDescriptor rule = getRule(uniqueRuleID);

        if (rule == null)
            throw new RuleDoesNotExistException(uniqueRuleID);
        else
        {
            Class<?> clazz = rule.getRuleGroup().getClazz();

            try
            {
                RuleGroupDescriptor ruleGroup = rule.getRuleGroup();
                AbstractRuleGroup instance = (AbstractRuleGroup) ruleGroup.getClazz().newInstance();
                HashMap<String, Object> parsedParameters = new HashMap<String, Object>();

                // If the RuleGroup has context parameters, validate and initialize them
                if (!ruleGroup.getContextParameters().isEmpty())
                {
                    // LowerCase them to match
                    if (parameters != null)
                        for (Entry<String, Object> param: parameters.entrySet())
                            parsedParameters.put(param.getKey().toLowerCase(), param.getValue());

                    // No parameters, passed! Cannot continue.
                    if (parameters == null)
                        throw new MissingContextException(ruleGroup);

                    // Call initialization method with parameters
                    ((IContextParameters) instance).initializeContext(parsedParameters);
                }

                return instance;
            }
            catch (InstantiationException e)
            {
                throw new RuleException(rule, clazz.getName() + ": " + RULE_GROUP_INSTANTIATE_ERROR, e);
            }
            catch (IllegalAccessException e)
            {
                throw new RuleException(rule, clazz.getName() + ": " + RULE_GROUP_INSTANTIATE_ERROR, e);
            }
        }
    }
}