package pt.digitalis.dif.rules;

import java.lang.reflect.Method;
import java.lang.reflect.Modifier;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import pt.digitalis.dif.ioc.DIFIoCRegistry;
import pt.digitalis.dif.rules.annotations.Flow;
import pt.digitalis.dif.rules.annotations.FlowAction;
import pt.digitalis.dif.rules.annotations.FlowActionOverride;
import pt.digitalis.dif.rules.condegen.RuleClassEnhancer;
import pt.digitalis.dif.rules.objects.flow.FlowActionDescriptor;
import pt.digitalis.dif.rules.objects.flow.FlowDescriptor;
import pt.digitalis.dif.rules.registration.ClassesRegistry;
import pt.digitalis.dif.rules.registration.IFlowRegistrator;
import pt.digitalis.dif.utils.logging.DIFLogger;
import pt.digitalis.utils.bytecode.holders.HolderRepository;

/**
 * 2010, Digitalis Informatica. All rights reserved. Distribuicao e Gestao de Informatica, Lda. Estrada de Paco de Arcos
 * num.9 - Piso -1 2780-666 Paco de Arcos Telefone: (351) 21 4408990 Fax: (351) 21 4408999 http://www.digitalis.pt
 */

/**
 * @author Pedro Viegas <a href="mailto:pviegas@digitalis.pt">pviegas@digitalis.pt</a><br/>
 * @created 2010/06/30
 */
public class FlowManager extends AbstractFlowManager {

    /** The flow actions directory */
    static private Map<String, FlowActionDescriptor> flowActions;

    /** The group hierarchy */
    static private Map<String, List<String>> flowGroupChildGroups;

    /**  */
    static private ClassesRegistry flowRegistry = new ClassesRegistry();

    /** The root directory */
    static private List<String> flowRootGroups;

    /** The flow directory */
    static private Map<String, FlowDescriptor> flows;

    /** T if initialization has took place */
    static private boolean isInitialized;

    /**
     * Parses a given class for Flow definitions and adds them to the Flow Manager. Will add both Flow and all inner
     * FlowActions
     * 
     * @param flowClass
     *            the class to infer the flow from
     * @return the {@link FlowDescriptor} added
     * @throws Exception
     */
    protected FlowDescriptor addFlowActionsFromClass(Class<?> flowClass) throws Exception
    {
        FlowDescriptor flow = null;

        // Gets the class Flow declaration annotation
        Flow flowDef = flowClass.getAnnotation(Flow.class);

        if (flowDef == null)
            // Was added a class to the Flow registry that is not a Flow declaration
            DIFLogger.getLogger().warn("   - " + flowClass.getSimpleName() + ": Does not declare a Flow!");

        else if (!Modifier.isAbstract(flowClass.getModifiers()))
            // Was added a class to the Flow registry that is not abstract
            DIFLogger.getLogger().warn(
                    "   - " + flowClass.getSimpleName() + ": All Flow declarations classes must be abstract!");
        else
        {

            // Creates a new instance
            flow = new FlowDescriptor(this, flowClass, flowDef.name(), flowDef.parentGroup());
            if (!flows.containsKey(flow.getUniqueName()))
            {
                // The rule group rule IDs list
                List<String> innerActions = new ArrayList<String>();
                List<FlowActionDescriptor> flowTempActions = new ArrayList<FlowActionDescriptor>();

                // Parse the inner actions
                for (Method method: flowClass.getMethods())
                {
                    FlowAction action = method.getAnnotation(FlowAction.class);
                    Method declaringMethod = method;

                    // Is an overridden Flow Action. Get parent method FlowAction definition
                    if (action == null && method.getAnnotation(FlowActionOverride.class) != null)
                    {
                        declaringMethod = this.findParentMethodAnnotededBy(method, FlowAction.class);
                        if (declaringMethod != null)
                            action = declaringMethod.getAnnotation(FlowAction.class);
                    }

                    if (action != null)
                    {
                        FlowActionDescriptor flowAction = new FlowActionDescriptor(this, action.name(),
                                action.description(), flow.getName(), method, declaringMethod, action.conditionRule());

                        // Add method parameters will be added in enhancement step ahead
                        innerActions.add(flowAction.getName().toLowerCase());
                        flowTempActions.add(flowAction);
                    }
                }

                flow.setActionIDs(innerActions);

                flow = RuleClassEnhancer.enhanceFlow(flow, flowTempActions);

                // Since we want that a Flow can be partially imported, we will use a logic of clean up erroneous
                // flowActions instead of error aborting the RuleGroup enhance process
                // Adds all non-invalid actions to the manager
                for (FlowActionDescriptor action: flowTempActions)
                    if (flow.getActionIDs().contains(action.getName().toLowerCase()))
                        flowActions.put(action.getUniqueName().toLowerCase(), action);

                // Adds the group to the directory indexed by the unique name
                flows.put(flow.getUniqueName().toLowerCase(), flow);

                // Handle the RuleGroup hierarchy
                if (flow.getParentGroupName() != null)
                {
                    FlowDescriptor parentGroupDesc = this.getFlow(flow.getParentGroupName());

                    // The Rule may not be register
                    if (parentGroupDesc == null)
                    {
                        String[] parentGroup = flow.getParentGroupName().replace(".", "###").split("###");

                        Class<?> clazz = flowRegistry.getClassesByName().get(parentGroup[parentGroup.length - 1]);
                        if (clazz != null)
                        {
                            parentGroupDesc = addFlowActionsFromClass(clazz);
                        }
                        else
                        {
                            throw new Exception("The Flow Action with the name '" + flow.getParentGroupName()
                                    + "' wasn't found.");
                        }
                    }

                    // If it is an inner group adds it to the parent group child list
                    if (flowGroupChildGroups.get(flow.getParentGroupName().toLowerCase()) == null)
                        flowGroupChildGroups.put(flow.getParentGroupName().toLowerCase(), new ArrayList<String>());

                    flowGroupChildGroups.get(flow.getParentGroupName().toLowerCase()).add(flow.getName().toLowerCase());
                }
                else
                    // If it is a root group add it to the root list
                    flowRootGroups.add(flow.getName().toLowerCase());
            }
        }

        return flow;
    }

    /**
     * @see pt.digitalis.dif.rules.IFlowManager#getFlow(java.lang.String)
     */
    public FlowDescriptor getFlow(String flowID)
    {
        return flows.get(flowID.toLowerCase());
    }

    /**
     * @see pt.digitalis.dif.rules.IFlowManager#getFlowAction(java.lang.String)
     */
    public FlowActionDescriptor getFlowAction(String uniqueActionID)
    {
        return flowActions.get(uniqueActionID.toLowerCase());
    }

    /**
     * @see pt.digitalis.dif.rules.IFlowManager#getFlowActions()
     */
    public Map<String, FlowActionDescriptor> getFlowActions()
    {
        return flowActions;
    }

    /**
     * @see pt.digitalis.dif.rules.IFlowManager#getFlows()
     */
    public List<FlowDescriptor> getFlows()
    {
        return new ArrayList<FlowDescriptor>(flows.values());
    }

    /**
     * @see pt.digitalis.dif.rules.IFlowManager#getRootFlowGroups()
     */
    public List<FlowDescriptor> getRootFlowGroups()
    {
        List<FlowDescriptor> rootGroupList = new ArrayList<FlowDescriptor>();

        for (String groupID: flowRootGroups)
            rootGroupList.add(flows.get(groupID));

        return rootGroupList;
    }

    /**
     * @see pt.digitalis.dif.rules.IFlowManager#getRuleGroupChildrenGroups(java.lang.String)
     */
    public List<String> getRuleGroupChildrenGroups(String groupUniqueID)
    {
        return flowGroupChildGroups.get(groupUniqueID.toLowerCase());
    }

    /**
     * @see pt.digitalis.dif.rules.AbstractManager#initialize()
     */
    @Override
    public void initialize() throws Exception
    {
        // Trigger rules engine initialization, since the Flows might reference them...
        getRulesManager().getRootRuleGroups();

        if (!isInitialized)
        {
            long startTime = System.currentTimeMillis();
            DIFLogger.getLogger().info("DIF Flow Manager Initialization started...");
            HolderRepository.initializeInternals();

            flows = new HashMap<String, FlowDescriptor>();
            flowActions = new HashMap<String, FlowActionDescriptor>();
            flowGroupChildGroups = new HashMap<String, List<String>>();
            flowRootGroups = new ArrayList<String>();
            flowRegistry = new ClassesRegistry();

            // Read all flow registrators declare by the applications from the IoC bindings and add them to the registry
            for (IFlowRegistrator registrator: DIFIoCRegistry.getRegistry().getImplementations(IFlowRegistrator.class))
                registrator.registerFlow(flowRegistry);

            // Parse the registry and add all rule groups and their rules to the Manager rule directory
            for (Class<?> flowClass: flowRegistry.getClasses())
                addFlowActionsFromClass(flowClass);

            // CondtitionRules validation
            StringBuffer errors = new StringBuffer();
            for (FlowActionDescriptor flow: flowActions.values())
            {
                if (flow.getConditionRuleName() != null
                        && !rulesManager.getRules().containsKey(flow.getConditionRuleName().replace("!", "")))
                {
                    errors.append("\nThe condition rule '" + flow.getConditionRuleName()
                            + "' isn't avaliable for rule '" + flow.getUniqueName() + "'");
                }
            }

            HolderRepository.cleanUp();

            // Throw critical errors if they exist
            if (errors.length() > 0)
            {
                throw new Exception(errors.toString());
            }
            else
            {
                dumpRegistryToLog();
                DIFLogger.getLogger().info(
                        "DIF Flow Manager Initialized in " + (System.currentTimeMillis() - startTime) + " ms.");
            }
            isInitialized = true;
        }
    }
}
