/**
 * 2010, Digitalis Informatica. All rights reserved. Distribuicao e Gestao de Informatica, Lda. Estrada de Paco de Arcos
 * num.9 - Piso -1 2780-666 Paco de Arcos Telefone: (351) 21 4408990 Fax: (351) 21 4408999 http://www.digitalis.pt
 */

package pt.digitalis.dif.rules.condegen;

import java.lang.reflect.Field;
import java.lang.reflect.Modifier;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;

import javassist.CannotCompileException;
import pt.digitalis.dif.dem.annotations.AnnotationTags;
import pt.digitalis.dif.rules.IFlowManager;
import pt.digitalis.dif.rules.IRulesManager;
import pt.digitalis.dif.rules.annotations.ContextParameter;
import pt.digitalis.dif.rules.exceptions.rules.RuleDoesNotExistException;
import pt.digitalis.dif.rules.objects.AbstractClassDescriptor;
import pt.digitalis.dif.rules.objects.AbstractMethodDescriptor;
import pt.digitalis.dif.rules.objects.flow.FlowActionDescriptor;
import pt.digitalis.dif.rules.objects.flow.FlowActionResult;
import pt.digitalis.dif.rules.objects.flow.FlowDescriptor;
import pt.digitalis.dif.rules.objects.rules.AbstractRuleGroup;
import pt.digitalis.dif.rules.objects.rules.RuleDescriptor;
import pt.digitalis.dif.rules.objects.rules.RuleGroupDescriptor;
import pt.digitalis.dif.rules.objects.rules.RuleResult;
import pt.digitalis.dif.utils.logging.DIFLogger;
import pt.digitalis.utils.CodeGenUtil4Javassist;
import pt.digitalis.utils.bytecode.exceptions.CodeGenerationException;
import pt.digitalis.utils.bytecode.holders.ClassHolder;
import pt.digitalis.utils.inspection.ReflectionUtils;
import pt.digitalis.utils.inspection.exception.ResourceNotFoundException;

/**
 * Class enhancer utility class for Rules
 * 
 * @author Pedro Viegas <a href="mailto:pviegas@digitalis.pt">pviegas@digitalis.pt</a><br/>
 * @created 2010/07/22
 */
public class RuleClassEnhancer {

    /** Error regarding the super class of all instances of Rules/Flows */
    private static final String MUST_EXTEND_TEMPLATE = "All $1 must extend " + AbstractRuleGroup.class.getSimpleName();

    /** Error regarding the naming of all parameters */
    private static final String PARAMETER_NOT_NAMED = "All method parameters must be named with @Named.";

    /**
     * Receives a RuleGroup definition and creates an implementation class for it. this implementation class will be the
     * one that will be used by the {@link IRulesManager} at all times.
     * 
     * @param <T>
     * @param clazz
     * @param classDescriptor
     * @param tempChildItems
     * @return the rule group implementation class
     * @throws ResourceNotFoundException
     * @throws CodeGenerationException
     * @throws CannotCompileException
     * @throws ClassNotFoundException
     * @throws NoSuchMethodException
     * @throws SecurityException
     */
    static private <T extends AbstractClassDescriptor> T enhanceClassInstance(Class<T> clazz, T classDescriptor,
            List<? extends AbstractMethodDescriptor> tempChildItems) throws CodeGenerationException,
            ResourceNotFoundException, CannotCompileException, ClassNotFoundException, SecurityException,
            NoSuchMethodException

    {
        // Tests the super class of the ClassInstance type (according to the descriptor)
        if (!classDescriptor.getInstanceBaseImplementationClass().isAssignableFrom(classDescriptor.getClazz()))
            throw new CodeGenerationException(classDescriptor.getName() + ": "
                    + MUST_EXTEND_TEMPLATE.replace("$1", clazz.getSimpleName()));

        String declClassName = classDescriptor.getClazz().getCanonicalName();
        String implClassName = declClassName + "Impl";

        ClassHolder implClass = CodeGenUtil4Javassist.createNewClassHolder(implClassName);
        ClassHolder declClass = new ClassHolder(declClassName);

        // Declare inheritance to declaration class (important this approach for debug purposes)
        implClass.setSuperClass(declClassName);

        // Search for the context parameters
        // Search all fields. Including inherited ones that declare the ContextParameter annotation
        Map<String, Field> classContextVars = new HashMap<String, Field>();
        Class<?> tempClass = classDescriptor.getClazz();

        while (tempClass != null)
        {
            for (Field field: tempClass.getDeclaredFields())
            {
                if (field.getAnnotation(ContextParameter.class) != null)
                {
                    // Has the annotation
                    if (Modifier.isPrivate(field.getModifiers()))
                    {
                        // Is private, hence, non-usable
                        StringBuffer message = new StringBuffer();
                        message.append("   - " + classDescriptor.getClazz().getSimpleName());
                        message.append(": Field \"" + field.getName() + "\"");

                        if (tempClass != classDescriptor.getClazz())
                            message.append(" (inherited from class " + tempClass.getSimpleName() + ")");

                        message.append(" is private and since cannot be used!\n");

                        DIFLogger.getLogger().warn(message);
                    }
                    else
                        // All ok, will use it...
                        classContextVars.put(field.getName(), field);
                }
            }
            tempClass = tempClass.getSuperclass();
        }

        classDescriptor.setContextParameters(new ArrayList<String>(classContextVars.keySet()));

        // Has context parameters, add the initialization method and the getParameterValues

        // Add the initialization method
        StringBuffer methodSource = new StringBuffer();
        methodSource.append("public void initializeContext(java.util.Map params){\n");

        for (Entry<String, Field> param: classContextVars.entrySet())
        {
            methodSource.append("    " + param.getKey() + " = (" + param.getValue().getType().getCanonicalName()
                    + ")params.get(\"" + param.getKey().toLowerCase() + "\");\n");
            methodSource.append("    if (" + param.getKey() + " == null) getThrower().throwMissingContextException(\""
                    + classDescriptor.getName() + "\", \"" + param.getKey() + "\");\n");
        }

        methodSource.append("}\n");

        DIFLogger.getLogger().debug(
                "[" + classDescriptor.getClazz().getSimpleName() + ": " + classDescriptor.getName()
                        + "] initialization method\n\n" + methodSource.toString() + "\n");

        implClass.addInterface(new ClassHolder(IContextParameters.class.getCanonicalName()));
        implClass.addMethod(methodSource.toString());

        // Add the getParameterValues method
        methodSource = new StringBuffer();
        methodSource.append("protected java.util.Map getParameterValues(){\n");
        methodSource.append("    java.util.Map results = new java.util.HashMap();\n");

        for (Entry<String, Field> param: classContextVars.entrySet())
            methodSource.append("    results.put(\"" + param.getKey().toLowerCase() + "\", " + param.getKey() + ");\n");

        methodSource.append("    return results;\n}\n");

        DIFLogger.getLogger().debug(
                "[" + classDescriptor.getClazz().getSimpleName() + ": " + classDescriptor.getName()
                        + "] getParameterValues method\n\n" + methodSource.toString() + "\n");

        implClass.addMethod(methodSource.toString());

        // For all child create the corresponding overwritten methods
        for (AbstractMethodDescriptor child: tempChildItems)
        {
            // Copy the original method signature and body (this will facilitate the signature generation since generics
            // and other features are not supported by ByteCodeUtils and Java Compilers translated them into the byte
            // code in a way that may change in future versions)
            try
            {
                // CtMethod javaAssistMethod = CtMethod.g
                implClass.copyMethodFromClass(declClass, child.getMethodToImplementation().getName());

                // Create the new method source
                methodSource = new StringBuffer();
                StringBuffer argumentString = new StringBuffer();
                List<String> argumentList = ReflectionUtils.getMethodParameterNames(child.getDeclaringMethod());
                child.setParameters(argumentList);
                methodSource.append("{\n");

                // Parse method parameters to build the named parameter list
                for (String argName: argumentList)
                {
                    if (argName == null)
                        throw new CodeGenerationException(child.getName() + "["
                                + classDescriptor.getClazz().getSimpleName() + "."
                                + child.getMethodToImplementation().getName() + "]: " + PARAMETER_NOT_NAMED);
                    else if (argumentString.length() == 0)
                        argumentString.append(argName);
                    else
                        argumentString.append("," + argName);
                }

                // Determine result type
                if (child.getMethodToImplementation().getReturnType().isAssignableFrom(Boolean.class))
                    child.setBooleanResult(true);
                else if (child.getMethodToImplementation().getReturnType().isAssignableFrom(boolean.class))
                    child.setBooleanResult(true);

                // if the rule has an execution condition rule, execute it before
                if (child.getConditionRuleName() != null && !AnnotationTags.NONE.equals(child.getConditionRuleName()))
                {
                    // Is the method being generated is a rule in side a RuleGroup
                    boolean isCalledFromRule = classDescriptor instanceof RuleGroupDescriptor;

                    // Will determine if the operation should be validate the condition as true or false
                    String operatorPrefix = "!";
                    String ruleID = child.getConditionRuleName();

                    if (child.getConditionRuleName().startsWith("!"))
                    {
                        operatorPrefix = "";
                        ruleID = child.getConditionRuleName().substring(1);
                    }

                    // Local reference to a rule
                    if (!ruleID.contains("."))
                    {
                        if (isCalledFromRule)
                            ruleID = ((RuleGroupDescriptor) classDescriptor).getUniqueName() + "." + ruleID;
                        else
                            // local references only allowed inside Rule Groups
                            throw new RuleDoesNotExistException(ruleID);
                    }

                    String ruleInstanceCode = "this";

                    if (!isCalledFromRule)
                    {
                        int pos = ruleID.lastIndexOf(".");
                        ruleInstanceCode = "getRuleGroupCachedInstance(\"" + ruleID.substring(0, pos) + "\")";
                    }

                    methodSource.append("    boolean ruleEvaluationResult = false;\n");
                    methodSource.append("    String ruleExceptionDescritpion = null;\n");
                    methodSource.append("    if( getRuleManager().getRule(\"" + ruleID + "\").isEvaluation()){\n");
                    methodSource.append("      ruleEvaluationResult = " + "getRuleManager().evaluateRule("
                            + ruleInstanceCode + ", \"" + ruleID + "\", buildParameterMap(\"" + argumentString
                            + "\",$args));\n");

                    methodSource.append("    } else {\n");
                    methodSource.append("      " + RuleResult.class.getCanonicalName() + " ruleResult = "
                            + "getRuleManager().executeRule(" + ruleInstanceCode + ", \"" + ruleID
                            + "\", buildParameterMap(\"" + argumentString + "\",$args));\n");
                    methodSource.append("      ruleEvaluationResult = ruleResult.isSuccess();\n");
                    methodSource.append("      if( ruleResult.getException() != null){\n");
                    methodSource.append("        ruleExceptionDescritpion = ruleResult.getException().getMessage();\n");
                    methodSource.append("      }\n");
                    methodSource.append("    }\n");

                    methodSource.append("    if(" + operatorPrefix + "ruleEvaluationResult){\n");

                    if (child.isBooleanResult())
                        // Evaluation rules return false, all others must use the defined method result type
                        methodSource.append("        return false;\n");
                    else if (isCalledFromRule)
                        // Execution rules must return an exception as a result
                        methodSource.append("        return new"
                                + (child.getMethodToImplementation().getReturnType().getSimpleName()
                                        .equalsIgnoreCase("void") ? RuleResult.class.getSimpleName() : child
                                        .getMethodToImplementation().getReturnType().getSimpleName())
                                + "(getThrower().getConditionRuleInvalid(getRuleManager().getRule(\"" + ruleID
                                + "\"), ruleExceptionDescritpion));\n");
                    else
                        // Flows must return CONDITION_FAILED with the exception inside
                        methodSource
                                .append("        return new"
                                        + (child.getMethodToImplementation().getReturnType().getSimpleName()
                                                .equalsIgnoreCase("void") ? FlowActionResult.class.getSimpleName()
                                                : child.getMethodToImplementation().getReturnType().getSimpleName())
                                        + "ConditionFailed(null, getThrower().getConditionRuleInvalid(getRuleManager().getRule(\""
                                        + ruleID + "\"), ruleExceptionDescritpion));\n");

                    methodSource.append("    }\n");
                }

                // Call super so debug on the original class source code is available
                methodSource.append("    return super." + child.getMethodToImplementation().getName() + "($$);\n");
                methodSource.append("}");

                DIFLogger.getLogger().debug(
                        "[Rule: " + child.getName() + "] execution method\n\n" + methodSource.toString() + "\n");

                // Update with new body
                implClass.updateMethodSource(child.getMethodToImplementation().getName(), methodSource.toString());
            }
            catch (CodeGenerationException e)
            {
                classDescriptor.getChildIDs().remove(child.getName().toLowerCase());

                // Show exception but carry on process for other RuleGroups
                e.printStackTrace();
            }
            catch (ResourceNotFoundException e)
            {
                classDescriptor.getChildIDs().remove(child.getName().toLowerCase());

                // Show exception but carry on process for other Classes
                e.printStackTrace();
            }
            catch (RuleDoesNotExistException e)
            {
                // Show exception but carry on process for other Classes
                e.printStackTrace();
            }
        }

        // Write class to file
        Class<?> implGeneratedClass = implClass.writeClass();

        // After enhancement process, substitute the class methods and class instance type with the generated class
        // equivalents
        for (AbstractMethodDescriptor rule: tempChildItems)
            rule.setMethodToImplementation(implGeneratedClass.getMethod(rule.getMethodToImplementation().getName(),
                    rule.getMethodToImplementation().getParameterTypes()));
        classDescriptor.setClazz(implGeneratedClass);

        return classDescriptor;
    }

    /**
     * Receives a Flow definition and creates an implementation class for it. this implementation class will be the one
     * that will be used by the {@link IFlowManager} at all times.
     * 
     * @param flow
     * @param flowTempActions
     * @return the flow implementation class
     * @throws ResourceNotFoundException
     * @throws CodeGenerationException
     * @throws CannotCompileException
     * @throws ClassNotFoundException
     * @throws NoSuchMethodException
     * @throws SecurityException
     */
    public static FlowDescriptor enhanceFlow(FlowDescriptor flow, List<FlowActionDescriptor> flowTempActions)
            throws CodeGenerationException, ResourceNotFoundException, CannotCompileException, ClassNotFoundException,
            SecurityException, NoSuchMethodException
    {
        return enhanceClassInstance(FlowDescriptor.class, flow, flowTempActions);
    }

    /**
     * Receives a RuleGroup definition and creates an implementation class for it. this implementation class will be the
     * one that will be used by the {@link IRulesManager} at all times.
     * 
     * @param ruleGroup
     * @param groupTempRules
     * @return the rule group implementation class
     * @throws ResourceNotFoundException
     * @throws CodeGenerationException
     * @throws CannotCompileException
     * @throws ClassNotFoundException
     * @throws NoSuchMethodException
     * @throws SecurityException
     */
    public static RuleGroupDescriptor enhanceRuleGroup(RuleGroupDescriptor ruleGroup,
            List<RuleDescriptor> groupTempRules) throws CodeGenerationException, ResourceNotFoundException,
            CannotCompileException, ClassNotFoundException, SecurityException, NoSuchMethodException

    {
        return enhanceClassInstance(RuleGroupDescriptor.class, ruleGroup, groupTempRules);
    }
}
