/**
 * 2010, Digitalis Informatica. All rights reserved. Distribuicao e Gestao de Informatica, Lda. Estrada de Paco de Arcos
 * num.9 - Piso -1 2780-666 Paco de Arcos Telefone: (351) 21 4408990 Fax: (351) 21 4408999 http://www.digitalis.pt
 */

package pt.digitalis.dif.rules;

import java.lang.annotation.Annotation;
import java.lang.reflect.Method;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;

import pt.digitalis.dif.rules.condegen.IContextParameters;
import pt.digitalis.dif.rules.exceptions.MissingContextException;
import pt.digitalis.dif.rules.exceptions.TooManyContextParamsException;
import pt.digitalis.dif.rules.objects.AbstractClassDescriptor;
import pt.digitalis.dif.rules.objects.AbstractMethodDescriptor;

/**
 * Base class for the rules project managers
 * 
 * @author Pedro Viegas <a href="mailto:pviegas@digitalis.pt">pviegas@digitalis.pt</a><br/>
 * @created 2010/07/29
 */
public abstract class AbstractManager {

    /**
     * @param descriptor
     *            the descriptor with the method to call
     * @param parameters
     *            the received parameters
     * @return the argument array
     */
    protected Object[] buildExecutionArguments(AbstractMethodDescriptor descriptor, Map<String, Object> parameters)
    {
        if (parameters == null)
            parameters = new HashMap<String, Object>();

        // Set the array of parameters need to the execution. Will get the values from the parameter map received,
        // and compile them in the correct method call order.
        Object[] args = new Object[descriptor.getParameters().size()];
        int index = 0;

        for (String paramName: descriptor.getParameters())
            // Warning: Non present arguments will be passed as null.
            args[index++] = parameters.get(paramName);

        return args;
    }

    /**
     * @param classDescriptor
     *            the {@link AbstractClassDescriptor} instance
     * @param parameterNames
     *            the mandatory parameter names
     * @param parameterValues
     *            the parameter values received
     * @return a map with the bindings
     * @throws TooManyContextParamsException
     */
    protected Map<String, Object> buildParameterMap(AbstractClassDescriptor classDescriptor,
            List<String> parameterNames, Object[] parameterValues) throws TooManyContextParamsException
    {
        if (parameterValues.length > parameterNames.size())
            throw new TooManyContextParamsException(classDescriptor);

        Map<String, Object> parameterMap = new HashMap<String, Object>();

        for (int i = 0; i < parameterValues.length; i++)
            parameterMap.put(parameterNames.get(i), parameterValues[i]);

        return parameterMap;
    }

    /**
     * Dumps the current registry to log. Useful for debug or initialization feedback purposes
     */
    abstract protected void dumpRegistryToLog();

    /**
     * Searches for the parent overridden method of the given one, that must be annotated by the given annotation type.
     * 
     * @param <A>
     *            the annotation type that must be associated to the parent method
     * @param method
     *            the original method
     * @param annotation
     *            the annotation instance from the associated method
     * @return the found method annotation instance
     */
    protected <A extends Annotation> Method findParentMethodAnnotededBy(Method method, Class<A> annotation)
    {
        Method parentMethod = method;
        A result = parentMethod.getAnnotation(annotation);

        // Is an overridden Flow Action. Get parent method FlowAction definition
        while (result == null && parentMethod != null)
        {
            Class<?> parentClass = parentMethod.getDeclaringClass().getSuperclass();

            // No more classes to search in the hierarchy. Give up!
            if (parentClass == null)
                break;

            try
            {
                parentMethod = parentClass.getMethod(method.getName(), method.getParameterTypes());
            }
            catch (SecurityException e)
            {
                // Will force exit of search loop
                parentMethod = null;
                e.printStackTrace();
            }
            catch (NoSuchMethodException e)
            {
                // Will force exit of search loop
                parentMethod = null;
                e.printStackTrace();
            }

            if (parentMethod != null)
                result = parentMethod.getAnnotation(annotation);
        }

        return parentMethod;
    }

    /**
     * Initializes the rules engine. Will read all registered rules and build the Manager instance.
     * 
     * @throws Exception
     */
    // TODO: Should add this to the DIF initialization process. Add a DIFCore initialization contributions feature
    abstract public void initialize() throws Exception;

    /**
     * Initializes the given {@link IContextParameters} instance with the given parameter according the the declared
     * parameters in the {@link AbstractMethodDescriptor}
     * 
     * @param instance
     *            the instance to set the parameters
     * @param instanceDescriptor
     *            the descriptor that describes the instance element
     * @param parameters
     *            the parameters received
     * @throws MissingContextException
     */
    protected void initializeInstanceContext(Object instance, AbstractClassDescriptor instanceDescriptor,
            Map<String, Object> parameters) throws MissingContextException
    {
        // If the RuleGroup has context parameters, validate and initialize them
        if (!instanceDescriptor.getContextParameters().isEmpty())
        {
            HashMap<String, Object> parsedParameters = new HashMap<String, Object>();

            // LowerCase them to match
            if (parameters != null)
                for (Entry<String, Object> param: parameters.entrySet())
                    parsedParameters.put(param.getKey().toLowerCase(), param.getValue());

            // No parameters, passed! Cannot continue.
            if (parameters == null)
                throw new MissingContextException(instanceDescriptor);

            // Call initialization method with parameters
            ((IContextParameters) instance).initializeContext(parsedParameters);
        }
    }
}
