/**
 * 2010, Digitalis Informatica. All rights reserved. Distribuicao e Gestao de Informatica, Lda. Estrada de Paco de Arcos
 * num.9 - Piso -1 2780-666 Paco de Arcos Telefone: (351) 21 4408990 Fax: (351) 21 4408999 http://www.digitalis.pt
 */

package pt.digitalis.dif.rules;

import java.util.List;
import java.util.Map;

import pt.digitalis.dif.rules.annotations.Flow;
import pt.digitalis.dif.rules.annotations.FlowAction;
import pt.digitalis.dif.rules.exceptions.MissingContextException;
import pt.digitalis.dif.rules.exceptions.TooManyContextParamsException;
import pt.digitalis.dif.rules.exceptions.flow.FlowActionException;
import pt.digitalis.dif.rules.exceptions.flow.FlowException;
import pt.digitalis.dif.rules.exceptions.flow.InvalidFlow;
import pt.digitalis.dif.rules.objects.flow.AbstractFlow;
import pt.digitalis.dif.rules.objects.flow.FlowActionDescriptor;
import pt.digitalis.dif.rules.objects.flow.FlowActionResult;
import pt.digitalis.dif.rules.objects.flow.FlowDescriptor;
import pt.digitalis.dif.rules.objects.rules.RuleGroupDescriptor;

/**
 * Defines a Rule Manager functionalities
 * 
 * @author Pedro Viegas <a href="mailto:pviegas@digitalis.pt">pviegas@digitalis.pt</a><br/>
 * @created 2010/06/30
 */
public interface IFlowManager {

    /**
     * Runs a {@link FlowAction} and wraps the result in a {@link FlowActionResult} object. If any exception occurs in
     * the action execution it will be saved and set in the exception attribute of the {@link FlowActionResult} object.
     * 
     * @param uniqueActionID
     *            the unique action to execute including the flow ID. Ex. (registrationFlow.register)
     * @param parameters
     *            parameters to pass to the rule
     * @return the flow action result
     * @throws FlowException
     * @throws FlowActionException
     * @throws MissingContextException
     */
    public FlowActionResult<?> flowExecute(String uniqueActionID, Map<String, Object> parameters) throws FlowException,
            FlowActionException, MissingContextException;

    /**
     * Runs a {@link FlowAction} and wraps the result in a {@link FlowActionResult} object. If any exception occurs in
     * the action execution it will be saved and set in the exception attribute of the {@link FlowActionResult} object.
     * 
     * @param flowID
     *            the flow ID
     * @param actionID
     *            the action to execute
     * @param parameters
     *            parameters to pass to the rule
     * @return the flow action result
     * @throws FlowException
     * @throws FlowActionException
     * @throws MissingContextException
     */
    public FlowActionResult<?> flowExecute(String flowID, String actionID, Map<String, Object> parameters)
            throws FlowException, FlowActionException, MissingContextException;

    /**
     * Finds and retrieves a Flow by it's declaring class
     * 
     * @param <T>
     * @param clazz
     *            the class that declares the {@link Flow}
     * @return the rule group instance
     * @throws InvalidFlow
     *             if the class does not define a valid {@link Flow}
     */
    public <T extends AbstractFlow> FlowDescriptor getFlow(Class<T> clazz) throws InvalidFlow;

    /**
     * Finds and retrieves a Flow by it's name
     * 
     * @param flowID
     *            the flow ID
     * @return the {@link FlowDescriptor}
     */
    public FlowDescriptor getFlow(String flowID);

    /**
     * Finds and retrieves a given Action by it's unique ID
     * 
     * @param uniqueActionID
     *            the action unique ID (i.e. "flowName.actionName")
     * @return the {@link FlowAction} Descriptor
     */
    public FlowActionDescriptor getFlowAction(String uniqueActionID);

    /**
     * Finds and retrieves a given flow action by it's flow and action ID pair
     * 
     * @param flowID
     *            the flow ID
     * @param actionID
     *            the action ID
     * @return the Rule Descriptor
     */
    public FlowActionDescriptor getFlowAction(String flowID, String actionID);

    /**
     * Inspector for the 'flowActions' attribute.
     * 
     * @return the flowActions value
     */
    public Map<String, FlowActionDescriptor> getFlowActions();

    /**
     * Instantiates a {@link Flow} class with all context linked. If the {@link Flow} declares context parameters the
     * initialization will fail <br/>
     * Parameter types must match!
     * 
     * @param <T>
     * @param clazz
     *            the class to create
     * @return the Flow instance business class
     * @throws FlowException
     * @throws MissingContextException
     */
    public <T extends AbstractFlow> T getFlowInstance(Class<T> clazz) throws FlowException, MissingContextException;

    /**
     * Instantiates a {@link Flow} class with all context linked. If the {@link Flow} declares context parameters not
     * present by name in parameters the initialization will fail <br/>
     * Parameter types must match!
     * 
     * @param <T>
     * @param clazz
     *            the class to create
     * @param parameters
     *            the context parameters
     * @return the Flow instance business class
     * @throws FlowException
     * @throws MissingContextException
     */
    public <T extends AbstractFlow> T getFlowInstance(Class<T> clazz, Map<String, Object> parameters)
            throws FlowException, MissingContextException;

    /**
     * Instantiates a {@link Flow} class with all context linked. If the {@link Flow} declares context parameters not
     * passed by position in parameters the initialization will fail. <br/>
     * Parameter types must match!
     * 
     * @param <T>
     * @param clazz
     *            the class to create
     * @param parameters
     *            the context parameters, passed by declaration order
     * @return the Flow instance business class
     * @throws FlowException
     * @throws TooManyContextParamsException
     * @throws MissingContextException
     */
    public <T extends AbstractFlow> T getFlowInstance(Class<T> clazz, Object... parameters) throws FlowException,
            TooManyContextParamsException, MissingContextException;

    /**
     * Retrieves all Flows
     * 
     * @return all flows
     */
    public List<FlowDescriptor> getFlows();

    /**
     * Retrieves all first level Rule Groups
     * 
     * @return all root rule groups
     */
    public List<FlowDescriptor> getRootFlowGroups();

    /**
     * Finds and retrieves a Group's inner groups by it's unique name (supports group hierarchy IDs)
     * 
     * @param groupUniqueID
     *            the flow group unique ID (i.e. "parentFlowGroupName.flowGroupName")
     * @return the list of {@link RuleGroupDescriptor}
     */
    public List<String> getRuleGroupChildrenGroups(String groupUniqueID);

    /**
     * Initializes the flow engine. Will read all registered flows and build the Manager instance
     * 
     * @throws Exception
     */
    public void initialize() throws Exception;
}
