/**
 * 2010, Digitalis Informatica. All rights reserved. Distribuicao e Gestao de Informatica, Lda. Estrada de Paco de Arcos
 * num.9 - Piso -1 2780-666 Paco de Arcos Telefone: (351) 21 4408990 Fax: (351) 21 4408999 http://www.digitalis.pt
 */

package pt.digitalis.dif.rules;

import java.util.List;
import java.util.Map;

import pt.digitalis.dif.rules.annotations.RuleGroup;
import pt.digitalis.dif.rules.exceptions.MissingContextException;
import pt.digitalis.dif.rules.exceptions.TooManyContextParamsException;
import pt.digitalis.dif.rules.exceptions.rules.InvalidRuleGroup;
import pt.digitalis.dif.rules.exceptions.rules.RuleException;
import pt.digitalis.dif.rules.exceptions.rules.RuleGroupException;
import pt.digitalis.dif.rules.objects.rules.AbstractRuleGroup;
import pt.digitalis.dif.rules.objects.rules.RuleDescriptor;
import pt.digitalis.dif.rules.objects.rules.RuleGroupDescriptor;
import pt.digitalis.dif.rules.objects.rules.RuleResult;

/**
 * Defines a Rule Manager functionalities
 * 
 * @author Pedro Viegas <a href="mailto:pviegas@digitalis.pt">pviegas@digitalis.pt</a><br/>
 * @created 2010/06/30
 */
public interface IRulesManager {

    /**
     * Runs a Rule of type evaluation
     * 
     * @param uniqueRuleID
     *            the rule to execute, identified by it's unique name, including the rule group name. (i.e.
     *            "ruleGroupName.ruleName")
     * @param parameters
     *            parameters to pass to the rule
     * @return the rule evaluation result
     * @throws RuleException
     * @throws RuleGroupException
     * @throws MissingContextException
     * @throws Throwable
     */
    public boolean evaluateRule(String uniqueRuleID, Map<String, Object> parameters) throws RuleException,
            RuleGroupException, MissingContextException, Throwable;

    /**
     * Runs a Rule of type evaluation
     * 
     * @param groupID
     *            the rule group ID
     * @param ruleID
     *            the rule to execute
     * @param parameters
     *            parameters to pass to the rule
     * @return the rule evaluation result
     * @throws RuleException
     * @throws RuleGroupException
     * @throws MissingContextException
     * @throws Throwable
     */
    public boolean evaluateRule(String groupID, String ruleID, Map<String, Object> parameters) throws RuleException,
            RuleGroupException, MissingContextException, Throwable;

    /**
     * Runs a Rule and wraps the result in a {@link RuleResult} object. If any exception occurs in the rule execution it
     * will be saved and set in the exception attribute of the {@link RuleResult} object.
     * 
     * @param uniqueRuleID
     *            the rule to execute, identified by it's unique name, including the rule group name. (i.e.
     *            "ruleGroupName.ruleName")
     * @param parameters
     *            parameters to pass to the rule
     * @return the rule evaluation result
     * @throws RuleException
     * @throws RuleGroupException
     * @throws MissingContextException
     */
    public RuleResult<?> executeRule(String uniqueRuleID, Map<String, Object> parameters) throws RuleException,
            RuleGroupException, MissingContextException;

    /**
     * Runs a Rule and wraps the result in a {@link RuleResult} object. If any exception occurs in the rule execution it
     * will be saved and set in the exception attribute of the {@link RuleResult} object.
     * 
     * @param groupID
     *            the rule group ID
     * @param ruleID
     *            the rule to execute
     * @param parameters
     *            parameters to pass to the rule
     * @return the rule evaluation result
     * @throws RuleException
     * @throws RuleGroupException
     * @throws MissingContextException
     */
    public RuleResult<?> executeRule(String groupID, String ruleID, Map<String, Object> parameters)
            throws RuleException, RuleGroupException, MissingContextException;

    /**
     * Retrieves all first level Rule Groups
     * 
     * @return all root rule groups
     */
    public List<RuleGroupDescriptor> getRootRuleGroups();

    /**
     * Finds and retrieves a given Rule by it's unique ID
     * 
     * @param uniqueRuleID
     *            the rule unique ID (i.e. "ruleGroupName.ruleName")
     * @return the Rule Descriptor
     */
    public RuleDescriptor getRule(String uniqueRuleID);

    /**
     * Finds and retrieves a given Rule by it's unique ID
     * 
     * @param groupID
     *            the rule group ID
     * @param ruleID
     *            the rule ID
     * @return the Rule Descriptor
     */
    public RuleDescriptor getRule(String groupID, String ruleID);

    /**
     * Finds and retrieves a Group by it's dleclarating class
     * 
     * @param <T>
     * @param clazz
     *            the class that declares the {@link RuleGroup}
     * @return the rule group instance
     * @throws InvalidRuleGroup
     *             if the class does not define a valid {@link RuleGroup}
     */
    public <T extends AbstractRuleGroup> RuleGroupDescriptor getRuleGroup(Class<T> clazz) throws InvalidRuleGroup;

    /**
     * Finds and retrieves a Group by it's unique name (supports group hierarchy IDs, i.e. "dif.aclrules")
     * 
     * @param groupUniqueID
     *            the rule group unique ID (i.e. "parentGuleGroupName.ruleGroupName")
     * @return the {@link RuleGroupDescriptor}
     */
    public RuleGroupDescriptor getRuleGroup(String groupUniqueID);

    /**
     * Finds and retrieves a Group's inner groups by it's unique name (supports group hierarchy IDs, i.e.
     * "dif.aclrules")
     * 
     * @param groupUniqueID
     *            the rule group unique ID (i.e. "parentGuleGroupName.ruleGroupName")
     * @return the list of {@link RuleGroupDescriptor}
     */
    public List<String> getRuleGroupChildrenGroups(String groupUniqueID);

    /**
     * Instantiates a {@link RuleGroup} class with all context linked. If the {@link RuleGroup} declares context
     * parameters the initialization will fail <br/>
     * Parameter types must match!
     * 
     * @param <T>
     * @param clazz
     *            the class to create
     * @return the RuleGroup instance business class
     * @throws RuleGroupException
     * @throws MissingContextException
     */
    public <T extends AbstractRuleGroup> T getRuleGroupInstance(Class<T> clazz) throws RuleGroupException,
            MissingContextException;

    /**
     * Instantiates a {@link RuleGroup} class with all context linked. If the {@link RuleGroup} declares context
     * parameters not present by name in parameters the initialization will fail <br/>
     * Parameter types must match!
     * 
     * @param <T>
     * @param clazz
     *            the class to create
     * @param parameters
     *            the context parameters
     * @return the RuleGroup instance business class
     * @throws RuleGroupException
     * @throws MissingContextException
     */
    public <T extends AbstractRuleGroup> T getRuleGroupInstance(Class<T> clazz, Map<String, Object> parameters)
            throws RuleGroupException, MissingContextException;

    /**
     * Instantiates a {@link RuleGroup} class with all context linked. If the {@link RuleGroup} declares context
     * parameters not passed by position in parameters the initialization will fail. <br/>
     * Parameter types must match!
     * 
     * @param <T>
     * @param clazz
     *            the class to create
     * @param parameters
     *            the context parameters, passed by declaration order
     * @return the RuleGroup instance business class
     * @throws RuleGroupException
     * @throws TooManyContextParamsException
     * @throws MissingContextException
     */
    public <T extends AbstractRuleGroup> T getRuleGroupInstance(Class<T> clazz, Object... parameters)
            throws RuleGroupException, TooManyContextParamsException, MissingContextException;

    /**
     * Inspector for the 'rules' attribute.
     * 
     * @return the rules value
     */
    public Map<String, RuleDescriptor> getRules();

    /**
     * Initializes the rules engine. Will read all registered rules and build the Manager instance
     * 
     * @throws Exception
     */
    public void initialize() throws Exception;
}
