package pt.digitalis.dif.rules;

import java.lang.reflect.Method;
import java.lang.reflect.Modifier;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import pt.digitalis.dif.ioc.DIFIoCRegistry;
import pt.digitalis.dif.rules.annotations.RuleEvaluation;
import pt.digitalis.dif.rules.annotations.RuleExecution;
import pt.digitalis.dif.rules.annotations.RuleGroup;
import pt.digitalis.dif.rules.annotations.RuleOverride;
import pt.digitalis.dif.rules.condegen.RuleClassEnhancer;
import pt.digitalis.dif.rules.objects.rules.RuleDescriptor;
import pt.digitalis.dif.rules.objects.rules.RuleGroupDescriptor;
import pt.digitalis.dif.rules.objects.rules.RuleType;
import pt.digitalis.dif.rules.registration.ClassesRegistry;
import pt.digitalis.dif.rules.registration.IRulesRegistrator;
import pt.digitalis.dif.utils.logging.DIFLogger;
import pt.digitalis.utils.bytecode.holders.HolderRepository;

/**
 * 2010, Digitalis Informatica. All rights reserved. Distribuicao e Gestao de Informatica, Lda. Estrada de Paco de Arcos
 * num.9 - Piso -1 2780-666 Paco de Arcos Telefone: (351) 21 4408990 Fax: (351) 21 4408999 http://www.digitalis.pt
 */

/**
 * Simple {@link IRulesManager} implementation. Will use Rules defined by the statically declared rules in application
 * code
 * 
 * @author Pedro Viegas <a href="mailto:pviegas@digitalis.pt">pviegas@digitalis.pt</a><br/>
 * @created 2010/06/30
 */
public class RulesManager extends AbstractRulesManager {

    /** The rules classes registry */
    static private ClassesRegistry classesRegistry = new ClassesRegistry();

    /** T if initialization has took place */
    static private boolean isInitialized;

    /** The rule directory */
    static private List<String> rootRuleGroups;

    /** The group hierarchy */
    static private Map<String, List<String>> ruleGroupChildGroups;

    /** The rule directory */
    static private Map<String, RuleGroupDescriptor> ruleGroups;

    /** The rule directory */
    static private Map<String, RuleDescriptor> rules;

    /**
     * Parses a given class for Rule definitions and adds them to the Rule Manager. Will add both RuleGroups and all
     * inner Rules
     * 
     * @param ruleGroupClass
     *            the class to infer the rules from
     * @return the {@link RuleGroupDescriptor} added
     * @throws Exception
     */
    protected RuleGroupDescriptor addGroupRulesFromClass(Class<?> ruleGroupClass) throws Exception
    {
        RuleGroupDescriptor group = null;

        // Gets the class RuleGroup declaration annotation
        RuleGroup groupDef = ruleGroupClass.getAnnotation(RuleGroup.class);

        if (groupDef == null)
            // Was added a class to the RuleGroup registry that is not a RuleGroup declaration
            DIFLogger.getLogger().warn("   - " + ruleGroupClass.getSimpleName() + ": Does not declare a RuleGroup!");

        else if (!Modifier.isAbstract(ruleGroupClass.getModifiers()))
            // Was added a class to the RuleGroup registry that is not abstract
            DIFLogger.getLogger()
                    .warn("   - " + ruleGroupClass.getSimpleName()
                            + ": All RuleGroup declarations classes must be abstract!");
        else
        {

            // Creates a new instance
            group = new RuleGroupDescriptor(this, ruleGroupClass, groupDef.name(), groupDef.parentGroup());

            if (!ruleGroups.containsKey(group.getUniqueName()))
            {

                // The rule group rule IDs list
                List<String> innerRuleList = new ArrayList<String>();
                List<RuleDescriptor> groupTempRules = new ArrayList<RuleDescriptor>();

                // Parse the inner Rules
                for (Method method: ruleGroupClass.getMethods())
                {
                    RuleEvaluation evaluation = method.getAnnotation(RuleEvaluation.class);
                    RuleExecution execution = method.getAnnotation(RuleExecution.class);
                    Method declaringMethod = method;

                    // Is an overridden Rule. Get parent method Rule definition
                    if (evaluation == null && execution == null && method.getAnnotation(RuleOverride.class) != null)
                    {
                        // Search for parent RuleEvaluation declaration
                        declaringMethod = this.findParentMethodAnnotededBy(method, RuleEvaluation.class);

                        if (declaringMethod != null)
                            evaluation = declaringMethod.getAnnotation(RuleEvaluation.class);
                        else
                        {
                            // Search for parent RuleEvaluation declaration
                            declaringMethod = this.findParentMethodAnnotededBy(method, RuleExecution.class);

                            if (declaringMethod != null)
                                execution = declaringMethod.getAnnotation(RuleExecution.class);
                        }
                    }

                    RuleDescriptor rule = null;

                    if (execution != null)
                    {
                        // RuleExecution
                        rule = new RuleDescriptor(this, RuleType.EXECUTION, execution.name(), execution.description(),
                                group.getUniqueName(), method, declaringMethod, execution.conditionRule());
                    }
                    else if (evaluation != null)
                    {
                        // RuleEvaluation
                        rule = new RuleDescriptor(this, RuleType.EVALUATION, evaluation.name(),
                                evaluation.description(), group.getUniqueName(), method, declaringMethod,
                                evaluation.conditionRule());
                    }

                    // Add method parameters will be added in enhancement step ahead
                    if (rule != null)
                    {
                        innerRuleList.add(rule.getName().toLowerCase());
                        groupTempRules.add(rule);
                    }
                }

                group.setRuleIDs(innerRuleList);

                group = RuleClassEnhancer.enhanceRuleGroup(group, groupTempRules);

                // Since we want that a RuleGroup can be partially imported, we will use a logic of clean up
                // erroneous
                // rules
                // instead of error aborting the RuleGroup enhance process
                // Adds all non-invalid rules to the manager
                for (RuleDescriptor rule: groupTempRules)
                    if (group.getRuleIDs().contains(rule.getName().toLowerCase()))
                        rules.put(rule.getUniqueName().toLowerCase(), rule);

                // Adds the group to the directory indexed by the unique name
                ruleGroups.put(group.getUniqueName().toLowerCase(), group);

                // Handle the RuleGroup hierarchy
                if (group.getParentGroupName() != null)
                {
                    RuleGroupDescriptor parentGroupDesc = this.getRuleGroup(group.getParentGroupName());

                    // The Rule may not be register
                    if (parentGroupDesc == null)
                    {
                        String[] parentGroup = group.getParentGroupName().replace(".", "###").split("###");

                        Class<?> clazz = classesRegistry.getClassesByName().get(parentGroup[parentGroup.length - 1]);
                        if (clazz != null)
                        {
                            parentGroupDesc = addGroupRulesFromClass(clazz);
                        }
                        else
                        {
                            throw new Exception("The Rule Group with the name '" + group.getParentGroupName()
                                    + "' wasn't found.");
                        }
                    }

                    // If it is an inner group adds it to the parent group child list
                    if (ruleGroupChildGroups.get(parentGroupDesc.getUniqueName()) == null)
                    {
                        ruleGroupChildGroups.put(parentGroupDesc.getUniqueName(), new ArrayList<String>());
                    }

                    ruleGroupChildGroups.get(parentGroupDesc.getUniqueName()).add(group.getName().toLowerCase());
                }
                else
                    // If it is a root group add it to the root list
                    rootRuleGroups.add(group.getName().toLowerCase());

            }
            else
            {
                group = ruleGroups.get(group.getUniqueName());
            }
        }
        return group;
    }

    /**
     * @see pt.digitalis.dif.rules.IRulesManager#getRootRuleGroups()
     */
    public List<RuleGroupDescriptor> getRootRuleGroups()
    {
        List<RuleGroupDescriptor> rootGroupList = new ArrayList<RuleGroupDescriptor>();

        for (String groupID: rootRuleGroups)
            rootGroupList.add(ruleGroups.get(groupID));

        return rootGroupList;
    }

    /**
     * @see pt.digitalis.dif.rules.IRulesManager#getRule(java.lang.String)
     */
    public RuleDescriptor getRule(String uniqueRuleID)
    {
        return getRules().get(uniqueRuleID.toLowerCase());
    }

    /**
     * @see pt.digitalis.dif.rules.IRulesManager#getRuleGroup(java.lang.String)
     */
    public RuleGroupDescriptor getRuleGroup(String groupUniqueID)
    {
        return ruleGroups.get(groupUniqueID.toLowerCase());
    }

    /**
     * @see pt.digitalis.dif.rules.IRulesManager#getRuleGroupChildrenGroups(java.lang.String)
     */
    public List<String> getRuleGroupChildrenGroups(String groupUniqueID)
    {
        return ruleGroupChildGroups.get(groupUniqueID.toLowerCase());
    }

    /**
     * @see pt.digitalis.dif.rules.IRulesManager#getRules()
     */
    public Map<String, RuleDescriptor> getRules()
    {
        return rules;
    }

    /**
     * @throws Exception
     * @see pt.digitalis.dif.rules.AbstractManager#initialize()
     */
    @Override
    synchronized public void initialize() throws Exception
    {
        if (!isInitialized)
        {
            long startTime = System.currentTimeMillis();
            DIFLogger.getLogger().info("DIF Rules Manager Initialization started...");
            HolderRepository.initializeInternals();

            rootRuleGroups = new ArrayList<String>();
            ruleGroupChildGroups = new HashMap<String, List<String>>();
            ruleGroups = new HashMap<String, RuleGroupDescriptor>();
            rules = new HashMap<String, RuleDescriptor>();
            classesRegistry = new ClassesRegistry();

            // Read all rules registrators declare by the applications from the IoC bindings and add them to the
            // registry
            for (IRulesRegistrator registrator: DIFIoCRegistry.getRegistry()
                    .getImplementations(IRulesRegistrator.class))
                registrator.registerRules(classesRegistry);

            // Parse the registry and add all rule groups and their rules to the Manager rule directory
            for (Class<?> ruleGroupClass: classesRegistry.getClasses())
                addGroupRulesFromClass(ruleGroupClass);

            // CondtitionRules validation
            StringBuffer errors = new StringBuffer();
            for (RuleDescriptor rule: rules.values())
            {
                if (rule.getConditionRuleName() != null
                        && !rules.containsKey(rule.getConditionRuleName().replace("!", "")))
                {
                    errors.append("\nThe condition rule '" + rule.getConditionRuleName()
                            + "' isn't available for rule '" + rule.getUniqueName() + "'");
                }
            }

            HolderRepository.cleanUp();
            // Throw critical errors if they exist
            if (errors.length() > 0)
            {
                throw new Exception(errors.toString());
            }
            else
            {

                dumpRegistryToLog();
                DIFLogger.getLogger().info(
                        "DIF Rules Manager Initialized in " + (System.currentTimeMillis() - startTime) + " ms.");
            }

            isInitialized = true;
        }
    }
}
