/** 2007, Digitalis Informatica. All rights reserved.
 * 
 * Distribuicao e Gestao de Informatica, Lda.
 * Estrada de Paco de Arcos num.9 - Piso -1
 * 2780-666 Paco de Arcos
 * Telefone: (351) 21 4408990
 * Fax: (351) 21 4408999
 * http://www.digitalis.pt 
 */
package pt.digitalis.dif.test.objects;

/**
 * Server information data type.
 * 
 * @author Pedro Viegas <a href="mailto:pviegas@digitalis.pt">pviegas@digitalis.pt</a>
 * @author Rodrigo Gonalves <a href="mailto:rgoncalves@digitalis.pt">rgoncalves@digitalis.pt</a>
 * @created Jan 23, 2008
 */
public class ServerInfo {

    /** The first TCP/UDP port (0). */
    final static private int FIRST_PORT = 0;

    /** The last TCP/UDP port (65535). */
    final static private int LAST_PORT = 65535;
    
    /** The 'localhost' name. */
    final static private String LOCALHOST = "localhost";
    
    /** The default port number. */
    final static private int DEFAULT_PORT_NUMBER = 8080;

    /** The name of the server to access. */
    private String serverName;

    /** The server port. */
    private int port;

    /** The application. */
    private String application;

    /**
     * Constructor.
     * 
     * @param serverName the server name
     * @param portNumber the port number
     * @param application the application name 
     * @throws ServerInfoBadPortException if the port number is invalid */
    public ServerInfo(String serverName, int portNumber, String application) throws ServerInfoBadPortException
    {
        this.serverName = serverName;
        this.application = application;
        
        if (isPortValid(portNumber))
            this.port = portNumber;
        else
            throw new ServerInfoBadPortException(portNumber);
    }
    
    /**
     * Constructor.
     * 
     * @param application the application name
     */
    public ServerInfo(String application)
    {
        this.serverName = LOCALHOST;
        this.port = DEFAULT_PORT_NUMBER;
        this.application = application;
    }

    
    
    /**
     * Inspector for the deployment context.
     * 
     * @return the server's deployment context
     */
    public String getApplication() {
        return this.application;
    }

    /**
     * Modifier for the deployment context.
     * 
     * @param context
     *            the deployment context to set
     */
    public void setApplication(String context) {
        this.application = context;
    }

    /**
     * Inspector for the port.
     * 
     * @return the port
     */
    public int getPort() {
        return this.port;
    }

    /**
     * Modifier for the port.
     * 
     * @param port
     *            the port to set
     * @throws ServerInfoBadPortException
     *             if the port number is invalid
     */
    public void setPort(int port) throws ServerInfoBadPortException {
        if (!isPortValid(port))
            throw new ServerInfoBadPortException(port);
        this.port = port;
    }

    /**
     * Inspector for the server name.
     * 
     * @return the server name to set
     */
    public String getServerName() {
        return this.serverName;
    }

    /**
     * Modifier for the server name.
     * 
     * @param serverName
     *            the serverName to set
     */
    public void setServerName(String serverName) {
        this.serverName = serverName;
    }

    /**
     * @param port
     * @return T if FIRST_PORT <= port <= LAST_PORT, F otherwise
     */
    private boolean isPortValid(int port) {
        return ((FIRST_PORT <= port) && (port <= LAST_PORT));
    }

    /** Exception to throw if the port number is not valid. */
    final static private class ServerInfoBadPortException extends Exception {
        /** The serial id. */
        private static final long serialVersionUID = 1314615534696372726L;

        /**
         * Default constructor.
         * 
         * @param badPort
         *            the number of the bad port
         */
        ServerInfoBadPortException(int badPort)
        {
            super("Invalid port number specified! Port number must be between " + FIRST_PORT + " and " + LAST_PORT + " but the supplied port number was: " + badPort);
        }
    }

}
