/**
 * 2007, Digitalis Informatica. All rights reserved. Distribuicao e Gestao de Informatica, Lda. Estrada de Paco de Arcos
 * num.9 - Piso -1 2780-666 Paco de Arcos Telefone: (351) 21 4408990 Fax: (351) 21 4408999 http://www.digitalis.pt
 */

package pt.digitalis.dif.test.impl;

import java.util.Collections;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Vector;

import com.meterware.httpunit.WebResponse;

import pt.digitalis.dif.test.IConcurrentAccessPageTester;
import pt.digitalis.dif.test.IPageTester;
import pt.digitalis.dif.test.exception.PageTestException;
import pt.digitalis.dif.test.objects.ResponseData;
import pt.digitalis.dif.test.objects.URLCall;

/**
 * Implements a test object that performs concurrent access to a given page.
 *
 * @author Rodrigo Gonalves <a href="mailto:rgoncalves@digitalis.pt">rgoncalves@digitalis.pt</a><br/>
 * @created 2008/01/22
 */
public class ConcurrentAccessPageTesterImpl implements IConcurrentAccessPageTester {

    /** Emulates a parallelizable access to a web page. */
    final static private class ParallelAccess extends Thread {

        /** The access ID. */
        String accessID;

        /** Reference to the caller to return results. */
        ConcurrentAccessPageTesterImpl concurrentAccesPageTester;

        /** The page object. */
        URLCall page;

        /**
         * Builds a new parallel access thread.
         *
         * @param concurrentAccesPageTester
         *            the caller for notifying
         * @param page
         *            the page object
         * @param accessID
         *            the access id
         */
        public ParallelAccess(ConcurrentAccessPageTesterImpl concurrentAccesPageTester, URLCall page, String accessID)
        {
            this.concurrentAccesPageTester = concurrentAccesPageTester;
            this.page = page;

            // If accessID was supplied set it
            if (accessID != null)
                this.accessID = accessID;
            // ...else generate an ID from thread's name
            else
                this.accessID = Thread.currentThread().getName();
        }

        /**
         * Makes an access to the defined URL.
         */
        @Override
        public void run()
        {

            try
            {
                // Build the response data element
                ResponseData responseData = new ResponseData();
                responseData.setCallerID(this.accessID);

                // Build page tester
                IPageTester pageTester = new PageTesterImpl();

                // Record start time
                responseData.setStartTime(System.currentTimeMillis());

                // Access the page
                WebResponse response = pageTester.testPageGET(this.page);

                // Record end time
                responseData.setEndTime(System.currentTimeMillis());

                // Set web response
                responseData.setResponse(response);

                // If response was not null pass it back to the caller
                if (response != null)
                    concurrentAccesPageTester.addResponse(responseData);
                else
                    System.err.println("Response was null for Thread " + Thread.currentThread().getId() + " accessing "
                            + this.page.getURL());

            }
            catch (PageTestException pageTestException)
            {
                pageTestException.printStackTrace();
            }
        }
    } // END ParallelAccess CLASS

    /** The response set. */
    List<ResponseData> responseSet = new Vector<ResponseData>();

    /**
     * Adds a new response to the response set.
     *
     * @param response
     *            the response to add
     */
    public void addResponse(ResponseData response)
    {
        // Store result
        this.responseSet.add(response);
        // Notify waiting thread (this!!)
        synchronized (this)
        {
            notify();
        }
    }

    /**
     * @see pt.digitalis.dif.test.IConcurrentAccessPageTester#doConcurrentAccessToPage(java.util.Map, long)
     */
    public Map<String, ResponseData> doConcurrentAccessToPage(Map<String, URLCall> pages, long delay)
    {

        for (String accessID: pages.keySet())
        {
            lauchAccess(pages.get(accessID), accessID);
            // Wait for the specified time
            waitFor(delay);
        }

        // Wait for all responses
        waitForAllResponses(pages.size());

        Collections.sort(this.responseSet);

        Map<String, ResponseData> result = new HashMap<String, ResponseData>();

        for (Iterator<ResponseData> iterator = this.responseSet.iterator(); iterator.hasNext();)
        {
            ResponseData responseData = iterator.next();
            result.put(responseData.getCallerID(), responseData);

        }

        return result;
    }

    /**
     * @see pt.digitalis.dif.test.IConcurrentAccessPageTester#doConcurrentAccessToPage(pt.digitalis.dif.test.objects.URLCall,
     *      int, long)
     */
    public List<ResponseData> doConcurrentAccessToPage(URLCall page, int numberOfSessions, long delay)
    {

        // Launch as many ParallellAccesses as the defined number of sessions
        for (int launchedSession = 0; launchedSession != numberOfSessions; ++launchedSession)
        {
            lauchAccess(page, null);
            // Wait for the specified time
            waitFor(delay);
        }

        // Wait for all responses
        waitForAllResponses(numberOfSessions);

        return this.responseSet;
    }

    /**
     * Launch a page access to the given URL. Allows the pass of an access id for further response inspection.
     *
     * @param page
     *            the page object
     * @param accessID
     *            the accessID
     */
    private void lauchAccess(URLCall page, String accessID)
    {
        // Create a new thread reference
        ParallelAccess parallelAccess = new ParallelAccess(this, page, accessID);
        parallelAccess.start();
    }

    /**
     * Wait for a specified time.
     *
     * @param delay
     *            the time delay between the requests (in ms).
     */
    private void waitFor(long delay)
    {
        // If a delay between sessions was set, pause for the adequate time
        if (delay != 0)
        {
            try
            {
                synchronized (this)
                {
                    Thread.sleep(delay);
                }
            }
            catch (InterruptedException interruptedException)
            {
                interruptedException.printStackTrace();
            }
        }
    }

    /**
     * Wait for all responses to be received.
     *
     * @param numberOfResponses
     *            the number of responses to wait for
     */
    private void waitForAllResponses(int numberOfResponses)
    {
        // Wait for all sessions to finish and return their results.
        while (responseSet.size() != numberOfResponses)
            try
            {
                synchronized (this)
                {
                    wait();
                }
            }
            catch (InterruptedException interruptedException)
            {
                interruptedException.printStackTrace();
            }
    }

} // END ConcurrentAccessPageImpl CLASS
