/**
 * 2007, Digitalis Informatica. All rights reserved. Distribuicao e Gestao de Informatica, Lda. Estrada de Paco de Arcos
 * num.9 - Piso -1 2780-666 Paco de Arcos Telefone: (351) 21 4408990 Fax: (351) 21 4408999 http://www.digitalis.pt
 */
package pt.digitalis.dif.test.objects;

import java.util.HashMap;
import java.util.Map;

/**
 * Holds information about a server call.
 *
 * @author Rodrigo Goncalves <a href="mailto:rgoncalves@digitalis.pt">rgoncalves@digitalis.pt</a>
 * @author Pedro Viegas <a href="mailto:pviegas@digitalis.pt">pviegas@digitalis.pt</a>
 * @created Jul 1, 2008
 */
public class URLCall
{

    /** The '&' character as a String. */
    final static protected String AMPERSAND = "&";

    /** The ':' character as a String. */
    final static protected String COLON = ":";

    /** The '=' character as a String. */
    final static protected String EQUALS = "=";

    /** The '/' character as a String. */
    final static protected String SLASH = "/";

    /** The 'http://' sequence. */
    final static protected String THE_HTTP_PREFIX = "http:" + SLASH + SLASH;

    /** The url parameters. <K = parameter name; V = parameter value> */
    protected Map<String, String> parameters;

    /** The relative URL to run */
    protected String relativeURL;

    /** T if the scripting is enabled, F otherwise. */
    protected boolean scriptingEnabled;

    /** Information about the server where the stage will run. */
    protected ServerInfo serverInfo;

    /** the session in witch to run call this URL */
    protected JavaSession session = null;

    /**
     * Constructor.
     *
     * @param serverInfo       app server info
     * @param relativeURL      the url of the stage to run
     * @param parameters       the stage parameters
     * @param scriptingEnabled T if the scripting analysis should be done, F otherwise
     */
    public URLCall(ServerInfo serverInfo, String relativeURL, Map<String, String> parameters, boolean scriptingEnabled)
    {
        this.serverInfo = serverInfo;
        this.relativeURL = relativeURL;
        this.scriptingEnabled = scriptingEnabled;

        if (parameters == null)
            this.parameters = new HashMap<String, String>();
        else
            this.parameters = parameters;
    }

    /**
     * Adds a new parameter to the stage parameters.
     *
     * @param name  the parameter name
     * @param value the parameter value
     */
    public void addParameter(String name, String value)
    {
        this.parameters.put(name, value);
    }

    /**
     * Gets the value of a given parameter.
     *
     * @param name the parameter name
     *
     * @return the value of the parameter with the given name if it exists, null otherwise
     */
    public String getParameterValue(String name)
    {
        return this.parameters.get(name);
    }

    /**
     * Parameters inspector.
     *
     * @return the stage's parameters
     */
    public Map<String, String> getParameters()
    {
        return this.parameters;
    }

    /**
     * Parameters modifier.
     *
     * @param parameters the parameters to set
     */
    public void setParameters(Map<String, String> parameters)
    {
        this.parameters = parameters;
    }

    /**
     * URL inspector.
     *
     * @return the url
     */
    public String getRelativeURL()
    {
        return this.relativeURL;
    }

    /**
     * relativeURL modifier.
     *
     * @param relativeURL the url to set
     */
    public void setRelativeURL(String relativeURL)
    {
        this.relativeURL = relativeURL;
    }

    /**
     * Server info inspector.
     *
     * @return the serverInfo
     */
    public ServerInfo getServerInfo()
    {

        if (getSession() == null)
            return this.serverInfo;
        else
            return this.session.getServerInfo();
    }

    /**
     * Server info modifier.
     *
     * @param serverInfo the serverInfo to set
     */
    public void setServerInfo(ServerInfo serverInfo)
    {
        this.serverInfo = serverInfo;
    }

    /**
     * Inspector for the 'session' attribute.
     *
     * @return the session value
     */
    public JavaSession getSession()
    {
        return session;
    }

    /**
     * Modifier for the 'session' attribute.
     *
     * @param session the new session value to set
     */
    public void setSession(JavaSession session)
    {
        this.session = session;
    }

    /**
     * Builds the URL of this stage call.
     *
     * @return the URL for the stage call
     */
    public String getURL()
    {
        StringBuffer url = new StringBuffer(THE_HTTP_PREFIX);

        url.append(this.getServerInfo().getServerName());
        url.append(COLON);
        url.append(this.getServerInfo().getPort());
        url.append(SLASH);
        url.append(this.getServerInfo().getApplication());
        url.append(SLASH);
        url.append(this.relativeURL);

        // Append parameters if they exist
        if (parameters.keySet().size() > 0)
        {
            for (String parameterName : this.parameters.keySet())
            {
                url.append(AMPERSAND);
                url.append(parameterName);
                url.append(EQUALS);
                url.append(this.parameters.get(parameterName));
            }
        }
        return url.toString();
    }

    /**
     * Inspector for the 'scripting enabled' property.
     *
     * @return the scriptingEnabled
     */
    public boolean isScriptingEnabled()
    {

        if (getSession() == null)
            return this.scriptingEnabled;
        else
            return this.session.isScriptingEnabled();
    }

    /**
     * Modifier for the 'scripting enabled' property.
     *
     * @param scriptingEnabled the scriptingEnabled to set
     */
    public void setScriptingEnabled(boolean scriptingEnabled)
    {
        this.scriptingEnabled = scriptingEnabled;
    }

    /**
     * Removes an existing parameter from the stage parameters.
     *
     * @param name the parameter name
     */
    public void removeParameter(String name)
    {
        this.parameters.remove(name);
    }
}
