package pt.digitalis.dif.utils.pdf;

import com.lowagie.text.Document;
import com.lowagie.text.DocumentException;
import com.lowagie.text.Paragraph;
import com.lowagie.text.Rectangle;
import com.lowagie.text.pdf.AcroFields;
import com.lowagie.text.pdf.PdfPKCS7;
import com.lowagie.text.pdf.PdfReader;
import com.lowagie.text.pdf.PdfSignatureAppearance;
import com.lowagie.text.pdf.PdfStamper;
import com.lowagie.text.pdf.PdfWriter;
import pt.digitalis.log.LogLevel;
import pt.digitalis.log.Logger;
import pt.digitalis.utils.config.IConfigurations;
import pt.digitalis.utils.ioc.guice.IoCRegistryGuiceImpl;

import javax.security.auth.x500.X500Principal;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.security.KeyStore;
import java.security.PrivateKey;
import java.security.cert.Certificate;
import java.security.cert.X509Certificate;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Date;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.Map;

/**
 * The Class CertificateManager.
 */
public class CertificateManager
{

    /** The Constant SIGNATURE_NAME. */
    public static final String SIGNATURE_NAME = "SignatureAddedByDigitalis";

    /** The config id. */
    public static String CONFIG_ID = "dif2";

    /** The section id. */
    public static String SECTION_ID = "Security/DigitalCertificate";

    /** The instance. */
    private static CertificateManager instance = null;

    /** The certificate cache. */
    private Map<String, DigitalCertificateConfiguration> certificateCache =
            new HashMap<String, DigitalCertificateConfiguration>();

    /**
     * Creates the dummy pdf.
     *
     * @return the {@link ByteArrayOutputStream} PDF
     *
     * @exception DocumentException the document exception
     * @exception IOException       Signals that an I/O exception has occurred.
     */
    public static ByteArrayOutputStream createDummyPDF() throws DocumentException, IOException
    {
        ByteArrayOutputStream result = new ByteArrayOutputStream();

        Document document = new Document();
        PdfWriter.getInstance(document, result);
        document.open();
        document.add(new Paragraph("Create by Digitalis Development Team (For Tests Purpose)"));
        document.add(new Paragraph(new Date().toString()));

        document.close();

        return result;
    }

    /**
     * Gets the single instance of CertificateManager.
     *
     * @return single instance of CertificateManager
     *
     * @exception Exception the exception
     */
    static public CertificateManager getInstance() throws Exception
    {
        if (instance == null)
        {
            instance = new CertificateManager();

            instance.certificateCache = IoCRegistryGuiceImpl.getRegistry().getImplementation(IConfigurations.class)
                    .readAllConfigurations(CONFIG_ID, SECTION_ID, DigitalCertificateConfiguration.class);

            if (instance.certificateCache != null)
            {
                for (DigitalCertificateConfiguration conf : instance.certificateCache.values())
                {
                    conf.initialize();
                }
            }
        }
        return instance;
    }

    /**
     * Gets the all certificates.
     *
     * @return the all certificates
     */
    public Map<String, DigitalCertificateConfiguration> getAllCertificates()
    {
        return certificateCache;
    }

    /**
     * Gets the certificate available.
     *
     * @param certificateID the certificate ID
     *
     * @return the certificate available
     *
     * @exception Exception the exception
     */
    public boolean getCertificateAvailable(String certificateID) throws Exception
    {
        DigitalCertificateConfiguration conf;
        if (certificateID == null)
        {
            conf = DigitalCertificateConfiguration.getInstance();
        }
        else
        {
            conf = this.getCertificateConfiguration(certificateID);
        }

        return conf.isCertificateValid();
    }

    /**
     * Gets the certificate configuration.
     *
     * @param certificateID the certificate ID
     *
     * @return the certificate configuration
     *
     * @exception Exception the exception
     */
    public DigitalCertificateConfiguration getCertificateConfiguration(String certificateID) throws Exception
    {

        if (!certificateCache.containsKey(certificateID))
        {

            if (certificateID == null)
            {
                certificateCache.put(certificateID, DigitalCertificateConfiguration.getInstance());
            }
            else
            {
                DigitalCertificateConfiguration certificateConfiguration =
                        IoCRegistryGuiceImpl.getRegistry().getImplementation(IConfigurations.class)
                                .readConfiguration(CONFIG_ID, SECTION_ID + "/" + certificateID,
                                        DigitalCertificateConfiguration.class);
                if (certificateConfiguration != null)
                {
                    certificateConfiguration.setId(certificateID);
                    certificateConfiguration.initialize();
                    certificateCache.put(certificateID, certificateConfiguration);
                }
            }
        }

        return certificateCache.get(certificateID);
    }

    /**
     * Gets the certificate data.
     *
     * @param certificateID the certificate ID
     *
     * @return the certificate data
     *
     * @exception Exception the exception
     */
    public Map<String, String> getCertificateData(String certificateID) throws Exception
    {
        Map<String, String> result = new LinkedHashMap<String, String>();

        DigitalCertificateConfiguration conf;
        if (certificateID == null)
        {
            conf = DigitalCertificateConfiguration.getInstance();
        }
        else
        {
            conf = this.getCertificateConfiguration(certificateID);
        }

        KeyStore keystore = conf.getKeystore();
        String alias = conf.getAlias();

        X509Certificate cert;

        cert = (X509Certificate) keystore.getCertificate(alias);

        for (String value : cert.getSubjectX500Principal().getName(X500Principal.RFC1779).split(","))
        {
            String[] values = value.split("=");
            if (values.length == 2)
            {
                result.put("Issue To \"" + values[0] + "\"", values[1]);
            }
        }

        result.put("Issuer", cert.getIssuerX500Principal().getName());
        if (cert.getNotBefore() != null)
        {
            result.put("Validity From", cert.getNotBefore().toString());
        }
        if (cert.getNotAfter() != null)
        {
            result.put("Validity To", cert.getNotAfter().toString());
        }
        result.put("SerialNumber", cert.getSerialNumber().toString());
        result.put("Type", cert.getType());

        return result;
    }

    /**
     * Gets if certificate available
     *
     * @return if certificate available
     *
     * @exception Exception the exception
     */
    public boolean getDefaultCertificateAvailable() throws Exception
    {
        return this.getCertificateAvailable(null);
    }

    /**
     * Gets the certificate configuration.
     *
     * @return the certificate configuration
     *
     * @exception Exception the exception
     */
    public DigitalCertificateConfiguration getDefaultCertificateConfiguration() throws Exception
    {
        return this.getCertificateConfiguration(null);
    }

    /**
     * Gets the certificate data.
     *
     * @return the certificate data
     *
     * @exception Exception the exception
     */
    public Map<String, String> getDefaultCertificateData() throws Exception
    {
        return this.getCertificateData(null);
    }

    /**
     * verify if the pdf has the signature.
     *
     * @param pdfToValidateSingature the pdf to validate Signature
     * @param signatureName          the signature name
     * @param certificateID          the certificate ID
     *
     * @return true, if successful
     *
     * @exception Exception the exception
     */
    public boolean hasSignature(byte[] pdfToValidateSingature, String signatureName, String certificateID)
            throws Exception
    {
        Boolean result = null;

        DigitalCertificateConfiguration conf;
        if (certificateID == null)
        {
            conf = DigitalCertificateConfiguration.getInstance();
        }
        else
        {
            conf = this.getCertificateConfiguration(certificateID);
        }

        PdfReader reader = new PdfReader(pdfToValidateSingature);
        AcroFields af = reader.getAcroFields();
        ArrayList names = af.getSignatureNames();
        for (int k = 0; k < names.size(); ++k)
        {
            String name = (String) names.get(k);

            if (signatureName.equals(name))
            {
                return true;
            }
        }

        if (result == null)
        {
            result = false;
        }
        return result;
    }

    /**
     * verify if the pdf has the signature.
     *
     * @param pdfToValidateSingature the pdf to verify Signature
     * @param signatureName          the signature name
     *
     * @return true, if successful
     *
     * @exception Exception the exception
     */
    public boolean hasSignature(byte[] pdfToValidateSingature, String signatureName) throws Exception
    {
        return this.hasSignature(pdfToValidateSingature, signatureName, null);
    }

    /**
     * Internal sign PDF.
     *
     * @param pdfToSign                the pdf to sign
     * @param conf                     the conf
     * @param forceAllowFurtherChanges the force allow further changes
     *
     * @return the byte array output stream
     *
     * @exception Exception the exception
     */
    private ByteArrayOutputStream internalSignPDF(byte[] pdfToSign, DigitalCertificateConfiguration conf,
            Boolean forceAllowFurtherChanges) throws Exception
    {
        if (conf == null)
        {
            conf = DigitalCertificateConfiguration.getInstance();
        }

        OutputStream fout = null;

        KeyStore keystore = conf.getKeystore();
        String alias = conf.getAlias();
        PrivateKey key = (PrivateKey) keystore.getKey(alias, conf.getPassword().toCharArray());
        Certificate[] chain = keystore.getCertificateChain(alias);

        PdfReader reader = new PdfReader(pdfToSign);
        fout = new ByteArrayOutputStream();

        PdfStamper stp = PdfStamper.createSignature(reader, fout, '\0');
        PdfSignatureAppearance sap = stp.getSignatureAppearance();
        sap.setCrypto(key, chain, null, PdfSignatureAppearance.WINCER_SIGNED);
        if (conf.getReason() != null)
        {
            sap.setReason(conf.getReason());
        }
        if (conf.getLocation() != null)
        {
            sap.setLocation(conf.getLocation());
        }
        if (conf.getContact() != null)
        {
            sap.setContact(conf.getContact());
        }

        // comment next line to have an invisible signature
        if (conf.getShowSignature())
        {
            // If some of the coordinates aren't passed, default values come into action
            Float lowerLeftXAux = (conf.getLowerLeftX() == null ? 100 : conf.getLowerLeftX().floatValue());
            Float lowerLeftYAux = (conf.getLowerLeftY() == null ? 100 : conf.getLowerLeftY().floatValue());
            Float upperRightXAux = (conf.getUpperRightX() == null ? 200 : conf.getUpperRightX().floatValue());
            Float upperRightYAux = (conf.getUpperRightY() == null ? 200 : conf.getUpperRightY().floatValue());
            //Calendar calendar = Calendar.getInstance();

            int page = 1;
            if ("L".equalsIgnoreCase(conf.getPage()))
            {
                page = reader.getNumberOfPages();
            }

            sap.setVisibleSignature(new Rectangle(lowerLeftXAux, lowerLeftYAux, upperRightXAux, upperRightYAux), page,
                    SIGNATURE_NAME /*+ calendar.getTimeInMillis()*/);
        }

        if (conf.getCerfificationMode() != null && conf.getCerfificationMode().equals("AFC") ||
            forceAllowFurtherChanges)
        {
            // Allow form filling and annotations
            sap.setCertificationLevel(PdfSignatureAppearance.CERTIFIED_FORM_FILLING_AND_ANNOTATIONS);
        }
        else if (conf.getCerfificationMode() != null && conf.getCerfificationMode().equals("NFC"))
        {
            // Now changes allowed
            sap.setCertificationLevel(PdfSignatureAppearance.CERTIFIED_NO_CHANGES_ALLOWED);
        }
        else
        {
            // Now changes allowed
            sap.setCertificationLevel(PdfSignatureAppearance.CERTIFIED_NO_CHANGES_ALLOWED);
        }

        stp.close();

        return (ByteArrayOutputStream) fout;
    }

    /**
     * Save configuration.
     *
     * @param certificateID the certificate ID
     * @param path          the path
     * @param password      the password
     * @param showSignature the show signature
     * @param lowerLeftX    the lower left x
     * @param lowerLeftY    the lower left y
     * @param upperRightX   the upper right x
     * @param upperRightY   the upper right y
     * @param contact       the contact
     * @param reason        the reason
     * @param location      the location
     * @param page          the page
     *
     * @exception Exception the exception
     */
    public void saveConfiguration(String certificateID, String path, String password, Boolean showSignature,
            Double lowerLeftX, Double lowerLeftY, Double upperRightX, Double upperRightY, String contact, String reason,
            String location, String page) throws Exception
    {
        DigitalCertificateConfiguration certificateConfiguration;
        if (certificateID == null)
        {
            certificateConfiguration = DigitalCertificateConfiguration.getInstance();
        }
        else
        {
            certificateConfiguration = this.getCertificateConfiguration(certificateID);
        }

        if (path != null && !"".equals(path))
        {
            certificateConfiguration.setPath(path);
        }

        if (password != null && !"".equals(password))
        {
            certificateConfiguration.setPassword(password);
        }

        certificateConfiguration.setShowSignature(showSignature);

        if (lowerLeftX != null)
        {
            certificateConfiguration.setLowerLeftX(lowerLeftX.toString());
        }
        if (lowerLeftY != null)
        {
            certificateConfiguration.setLowerLeftY(lowerLeftY.toString());
        }
        if (upperRightX != null)
        {
            certificateConfiguration.setUpperRightX(upperRightX.toString());
        }
        if (upperRightY != null)
        {
            certificateConfiguration.setUpperRightY(upperRightY.toString());
        }

        if (location != null)
        {
            certificateConfiguration.setLocation(location);
        }
        if (reason != null)
        {
            certificateConfiguration.setReason(reason);
        }
        if (contact != null)
        {
            certificateConfiguration.setContact(contact);
        }

        if (page != null)
        {
            certificateConfiguration.setPage(page);
        }

        String confPath = SECTION_ID;
        if (certificateID != null)
        {
            confPath += "/" + certificateID;
        }

        IoCRegistryGuiceImpl.getRegistry().getImplementation(IConfigurations.class)
                .writeConfiguration(CONFIG_ID, confPath, certificateConfiguration);

        certificateConfiguration.cleanCache();

        if (certificateID != null)
        {
            this.certificateCache.put(certificateID, certificateConfiguration);
        }
    }

    /**
     * Save default configuration.
     *
     * @param path          the path
     * @param password      the password
     * @param showSignature the show signature
     * @param lowerLeftX    the lower left X
     * @param lowerLeftY    the lower left Y
     * @param upperRightX   the upper right X
     * @param upperRightY   the upper right Y
     * @param contact       the contact
     * @param reason        the reason
     * @param location      the location
     * @param page          the page
     *
     * @exception Exception the exception
     */
    public void saveDefaultConfiguration(String path, String password, Boolean showSignature, Double lowerLeftX,
            Double lowerLeftY, Double upperRightX, Double upperRightY, String contact, String reason, String location,
            String page) throws Exception
    {
        this.saveConfiguration(null, path, password, showSignature, lowerLeftX, lowerLeftY, upperRightX, upperRightY,
                contact, reason, location, page);
    }

    /**
     * Sign PDF.
     *
     * @param pdfToSign The PDF to Sign
     *
     * @return Then PDF Signed in
     *
     * @exception Exception the exception
     */
    public ByteArrayOutputStream signPDF(byte[] pdfToSign) throws Exception
    {
        return this.signPDF(pdfToSign, null, false);
    }

    /**
     * Sign PDF.
     *
     * @param pdfToSign                The PDF to Sign
     * @param forceAllowFurtherChanges the force allow further changes
     *
     * @return Then PDF Signed in
     *
     * @exception Exception the exception
     */
    public ByteArrayOutputStream signPDF(byte[] pdfToSign, Boolean forceAllowFurtherChanges) throws Exception
    {
        return this.signPDF(pdfToSign, null, forceAllowFurtherChanges);
    }

    /**
     * Sign PDF.
     *
     * @param pdfToSign     The PDF to Sign
     * @param showSignature <code>false</code> to make the signature invisible
     * @param reason        the reason (signature property)
     * @param location      the location (signature property)
     * @param contact       the contact (signature property)
     * @param lowerLeftX    the lower left x
     * @param lowerLeftY    the lower left y
     * @param upperRightX   the upper right x
     * @param upperRightY   the upper right y
     *
     * @return Then PDF Signed
     *
     * @exception Exception the exception
     */
    public ByteArrayOutputStream signPDF(byte[] pdfToSign, boolean showSignature, String reason, String location,
            String contact, Double lowerLeftX, Double lowerLeftY, Double upperRightX, Double upperRightY)
            throws Exception
    {
        return internalSignPDF(pdfToSign, null, false);
    }

    /**
     * Sign PDF.
     *
     * @param pdfToSign                The PDF to Sign
     * @param showSignature            <code>false</code> to make the signature invisible
     * @param reason                   the reason (signature property)
     * @param location                 the location (signature property)
     * @param contact                  the contact (signature property)
     * @param lowerLeftX               the lower left x
     * @param lowerLeftY               the lower left y
     * @param upperRightX              the upper right x
     * @param upperRightY              the upper right y
     * @param forceAllowFurtherChanges the force allow further changes
     *
     * @return Then PDF Signed
     *
     * @exception Exception the exception
     */
    public ByteArrayOutputStream signPDF(byte[] pdfToSign, boolean showSignature, String reason, String location,
            String contact, Double lowerLeftX, Double lowerLeftY, Double upperRightX, Double upperRightY,
            Boolean forceAllowFurtherChanges) throws Exception
    {
        return internalSignPDF(pdfToSign, null, forceAllowFurtherChanges);
    }

    /**
     * Sign PDF.
     *
     * @param pdfToSign     The PDF to Sign
     * @param certificateID the certificate ID
     *
     * @return Then PDF Signed in
     *
     * @exception Exception the exception
     */
    public ByteArrayOutputStream signPDF(byte[] pdfToSign, String certificateID) throws Exception
    {
        DigitalCertificateConfiguration conf;
        if (certificateID == null)
        {
            conf = DigitalCertificateConfiguration.getInstance();
        }
        else
        {
            conf = this.getCertificateConfiguration(certificateID);
        }

        return internalSignPDF(pdfToSign, conf, false);
    }

    /**
     * Sign PDF.
     *
     * @param pdfToSign                The PDF to Sign
     * @param certificateID            the certificate ID
     * @param forceAllowFurtherChanges the force allow further changes
     *
     * @return Then PDF Signed in
     *
     * @exception Exception the exception
     */
    public ByteArrayOutputStream signPDF(byte[] pdfToSign, String certificateID, Boolean forceAllowFurtherChanges)
            throws Exception
    {
        DigitalCertificateConfiguration conf;
        if (certificateID == null)
        {
            conf = DigitalCertificateConfiguration.getInstance();
        }
        else
        {
            conf = this.getCertificateConfiguration(certificateID);
        }

        return internalSignPDF(pdfToSign, conf, forceAllowFurtherChanges);
    }

    /**
     * Validate signature.
     *
     * @param pdfToValidateSingature the pdf to validate Signature
     *
     * @return true, if successful
     *
     * @exception Exception the exception
     */
    public boolean validateSignature(byte[] pdfToValidateSingature) throws Exception
    {
        return this.validateSignature(pdfToValidateSingature, null);
    }

    /**
     * Validate signature.
     *
     * @param pdfToValidateSingature the pdf to validate Signature
     * @param certificateID          the certificate ID
     *
     * @return true, if successful
     *
     * @exception Exception the exception
     */
    public boolean validateSignature(byte[] pdfToValidateSingature, String certificateID) throws Exception
    {
        Boolean result = null;

        DigitalCertificateConfiguration conf;
        if (certificateID == null)
        {
            conf = DigitalCertificateConfiguration.getInstance();
        }
        else
        {
            conf = this.getCertificateConfiguration(certificateID);
        }

        PdfReader reader = new PdfReader(pdfToValidateSingature);
        AcroFields af = reader.getAcroFields();
        ArrayList names = af.getSignatureNames();
        for (int k = 0; k < names.size(); ++k)
        {
            String name = (String) names.get(k);
            PdfPKCS7 pk = af.verifySignature(name);
            Calendar cal = pk.getSignDate();
            X509Certificate pkc[] = (X509Certificate[]) pk.getCertificates();

            Object fails[] = PdfPKCS7.verifyCertificates(pkc, conf.getKeyStoreAll(), null, cal);
            if (fails == null)
            {
                if (result == null)
                {
                    result = true;
                }
                else
                {
                    result = result && true;
                }
            }
        }

        if (result == null)
        {
            result = false;
        }
        return result;
    }

    /**
     * Verify all certificates.
     *
     * @param logLevel the log level
     *
     * @exception Exception the exception
     */
    public void verifyAllCertificates(LogLevel logLevel) throws Exception
    {

        // Verify the default certificate
        if (!this.getCertificateAvailable(null))
        {
            Logger.getLogger().log(logLevel, "The default digital certificate is not valid!");
        }

        for (String certificateID : this.certificateCache.keySet())
        {
            if (!this.getCertificateAvailable(certificateID))
            {
                Logger.getLogger().log(logLevel, "The '" + certificateID + "' digital certificate is not valid!");
            }
        }
    }
}
