/**
 * 2009, Digitalis Informatica. All rights reserved. Distribuicao e Gestao de Informatica, Lda. Estrada de Paco de Arcos
 * num.9 - Piso -1 2780-666 Paco de Arcos Telefone: (351) 21 4408990 Fax: (351) 21 4408999 http://www.digitalis.pt
 */

package pt.digitalis.dif.utils.pdf;

import com.lowagie.text.DocumentException;
import pt.digitalis.utils.common.StringUtils;
import pt.digitalis.utils.config.IConfigurations;
import pt.digitalis.utils.config.annotations.ConfigDefault;
import pt.digitalis.utils.config.annotations.ConfigID;
import pt.digitalis.utils.config.annotations.ConfigIgnore;
import pt.digitalis.utils.config.annotations.ConfigLOVValues;
import pt.digitalis.utils.config.annotations.ConfigSectionID;
import pt.digitalis.utils.ioc.guice.IoCRegistryGuiceImpl;

import java.io.FileInputStream;
import java.io.IOException;
import java.security.KeyStore;
import java.security.KeyStoreException;
import java.security.NoSuchAlgorithmException;
import java.security.cert.Certificate;
import java.security.cert.CertificateException;
import java.security.cert.X509Certificate;
import java.util.Enumeration;

/**
 * The Class Digital Certificates Security Configuration.
 *
 * @author Luis Pinto <a href="mailto:lpinto@digitalis.pt">lpinto@digitalis.pt</a><br/>
 * @created Jan 31, 2012
 */
@ConfigID("dif2")
@ConfigSectionID("Security/DigitalCertificate")
public class DigitalCertificateConfiguration
{

    /** The singleton instance property. */
    private static DigitalCertificateConfiguration instance;

    /** The alias. */
    private String alias = null;

    /** The contact. */
    private String contact;

    /** The certificate id. */
    private String id;

    /** The initialized. */
    private boolean initialized = false;

    /** The is certificate valid. */
    private Boolean isCertificateValid = null;

    /** The key store. */
    private KeyStore keystore = null;

    /** The key store. */
    private KeyStore keyStoreAll = null;

    /** The location. */
    private String location;

    /** The lower left x. */
    private String lowerLeftX;

    /** The lower left y. */
    private String lowerLeftY;

    /** The password. */
    private String password;

    /** The path. */
    private String path;

    /** The reason. */
    private String reason;

    /** The show signature. */
    private Boolean showSignature;

    /** The type. */
    private String type;

    /** The upper right x. */
    private String upperRightX;

    /** The upper right y. */
    private String upperRightY;

    /** The certification mode */
    private String cerfificationMode;

    /** The Page. */
    private String page;

    /**
     * Returns the active configuration object instance.
     *
     * @return the configuration instance
     *
     * @exception Exception the exception
     */
    @ConfigIgnore
    static public DigitalCertificateConfiguration getInstance() throws Exception
    {
        if (instance == null)
        {
            try
            {
                instance = IoCRegistryGuiceImpl.getRegistry().getImplementation(IConfigurations.class)
                        .readConfiguration(DigitalCertificateConfiguration.class);
            }
            catch (Exception e)
            {
                e.printStackTrace();
                instance = null;
            }
            if (instance != null)
            {
                instance.initialize();
            }
        }

        return instance;
    }

    /**
     * Clean cache.
     */
    public void cleanCache()
    {
        instance = null;
        this.initialized = false;
        this.isCertificateValid = null;
    }

    /**
     * Inspector for the 'alias' attribute.
     *
     * @return the alias value
     *
     * @exception KeyStoreException        the key store exception
     * @exception NoSuchAlgorithmException the no such algorithm exception
     * @exception CertificateException     the certificate exception
     * @exception IOException              the io exception
     * @exception KeyStoreException        the key store exception
     * @exception NoSuchAlgorithmException the no such algorithm exception
     * @exception CertificateException     the certificate exception
     * @exception IOException              the io exception
     */
    @ConfigIgnore
    public String getAlias() throws KeyStoreException, NoSuchAlgorithmException, CertificateException, IOException
    {
        if (!this.initialized)
        {
            this.initialize();
        }
        return this.alias;
    }

    /**
     * Gets cerfification mode.
     *
     * @return the cerfification mode
     */
    @ConfigDefault("CNCA")
    @ConfigLOVValues("NFC=No further changes allowed,AFC=Allow further changes")
    public String getCerfificationMode()
    {
        return this.cerfificationMode;
    }

    /**
     * Sets cerfification mode.
     *
     * @param cerfificationMode the cerfification mode
     */
    public void setCerfificationMode(String cerfificationMode)
    {
        this.cerfificationMode = cerfificationMode;
    }

    /**
     * Inspector for the 'contact' attribute.
     *
     * @return the contact value
     */
    @ConfigDefault("")
    public String getContact()
    {
        return this.contact;
    }

    /**
     * Modifier for the 'contact' attribute.
     *
     * @param contact the new contact value to set
     */
    public void setContact(String contact)
    {
        this.contact = contact;
    }

    /**
     * Inspector for the 'id' attribute.
     *
     * @return the id value
     */
    public String getId()
    {
        return this.id;
    }

    /**
     * Modifier for the 'id' attribute.
     *
     * @param id the new id value to set
     */
    public void setId(String id)
    {
        this.id = id;
    }

    /**
     * Inspector for the 'keyStoreAll' attribute.
     *
     * @return the keyStoreAll value
     *
     * @exception KeyStoreException        the key store exception
     * @exception NoSuchAlgorithmException the no such algorithm exception
     * @exception CertificateException     the certificate exception
     * @exception IOException              the io exception
     */
    @ConfigIgnore
    public KeyStore getKeyStoreAll()
            throws KeyStoreException, NoSuchAlgorithmException, CertificateException, IOException
    {
        if (!this.initialized)
        {
            this.initialize();
        }
        return this.keyStoreAll;
    }

    /**
     * Inspector for the 'keystore' attribute.
     *
     * @return the keystore value
     *
     * @exception KeyStoreException        the key store exception
     * @exception NoSuchAlgorithmException the no such algorithm exception
     * @exception CertificateException     the certificate exception
     * @exception IOException              the io exception
     */
    @ConfigIgnore
    public KeyStore getKeystore() throws KeyStoreException, NoSuchAlgorithmException, CertificateException, IOException
    {
        if (!this.initialized)
        {
            this.initialize();
        }
        return this.keystore;
    }

    /**
     * Inspector for the 'location' attribute.
     *
     * @return the location value
     */
    @ConfigDefault("")
    public String getLocation()
    {
        return this.location;
    }

    /**
     * Modifier for the 'location' attribute.
     *
     * @param location the new location value to set
     */
    public void setLocation(String location)
    {
        this.location = location;
    }

    /**
     * Inspector for the 'lowerLeftX' attribute.
     *
     * @return the lowerLeftX value
     */
    @ConfigDefault("")
    public Double getLowerLeftX()
    {
        if (this.lowerLeftX != null && !"".equals(this.lowerLeftX))
        {
            return Double.valueOf(this.lowerLeftX);
        }
        else
        {
            return null;
        }
    }

    /**
     * Modifier for the 'lowerLeftX' attribute.
     *
     * @param lowerLeftX the new lowerLeftX value to set
     */
    public void setLowerLeftX(String lowerLeftX)
    {
        this.lowerLeftX = lowerLeftX;
    }

    /**
     * Inspector for the 'lowerLeftY' attribute.
     *
     * @return the lowerLeftY value
     */
    @ConfigDefault("")
    public Double getLowerLeftY()
    {
        if (this.lowerLeftY != null && !"".equals(this.lowerLeftY))
        {
            return Double.valueOf(this.lowerLeftY);
        }
        else
        {
            return null;
        }
    }

    /**
     * Modifier for the 'lowerLeftY' attribute.
     *
     * @param lowerLeftY the new lowerLeftY value to set
     */
    public void setLowerLeftY(String lowerLeftY)
    {
        this.lowerLeftY = lowerLeftY;
    }

    /**
     * Gets page.
     *
     * @return the page
     */
    @ConfigDefault("F")
    public String getPage()
    {
        return this.page;
    }

    /**
     * Sets page.
     *
     * @param page the page
     */
    public void setPage(String page)
    {
        this.page = page;
    }

    /**
     * Inspector for the 'password' attribute.
     *
     * @return the password value
     */
    @ConfigDefault("")
    public String getPassword()
    {
        return this.password;
    }

    /**
     * Modifier for the 'password' attribute.
     *
     * @param password the new password value to set
     */
    public void setPassword(String password)
    {
        this.password = password;
    }

    /**
     * Inspector for the 'path' attribute.
     *
     * @return the path value
     */
    @ConfigDefault("")
    public String getPath()
    {
        return this.path;
    }

    /**
     * Modifier for the 'path' attribute.
     *
     * @param path the new path value to set
     */
    public void setPath(String path)
    {
        this.path = path;
    }

    /**
     * Inspector for the 'reason' attribute.
     *
     * @return the reason value
     */
    @ConfigDefault("")
    public String getReason()
    {
        return this.reason;
    }

    /**
     * Modifier for the 'reason' attribute.
     *
     * @param reason the new reason value to set
     */
    public void setReason(String reason)
    {
        this.reason = reason;
    }

    /**
     * Inspector for the 'showSignature' attribute.
     *
     * @return the showSignature value
     */
    @ConfigDefault("false")
    public Boolean getShowSignature()
    {
        return this.showSignature;
    }

    /**
     * Modifier for the 'showSignature' attribute.
     *
     * @param showSignature the new showSignature value to set
     */
    public void setShowSignature(Boolean showSignature)
    {
        this.showSignature = showSignature;
    }

    /**
     * Inspector for the 'type' attribute.
     *
     * @return the type value
     */
    @ConfigDefault("pkcs12")
    public String getType()
    {
        return this.type;
    }

    /**
     * Modifier for the 'type' attribute.
     *
     * @param type the new type value to set
     */
    public void setType(String type)
    {
        this.type = type;
    }

    /**
     * Inspector for the 'upperRightX' attribute.
     *
     * @return the upperRightX value
     */
    @ConfigDefault("")
    public Double getUpperRightX()
    {
        if (this.upperRightX != null && !"".equals(this.upperRightX))
        {
            return Double.valueOf(this.upperRightX);
        }
        else
        {
            return null;
        }
    }

    /**
     * Modifier for the 'upperRightX' attribute.
     *
     * @param upperRightX the new upperRightX value to set
     */
    public void setUpperRightX(String upperRightX)
    {
        this.upperRightX = upperRightX;
    }

    /**
     * Inspector for the 'upperRightY' attribute.
     *
     * @return the upperRightY value
     */
    @ConfigDefault("")
    public Double getUpperRightY()
    {
        if (this.upperRightY != null && !"".equals(this.upperRightY))
        {
            return Double.valueOf(this.upperRightY);
        }
        else
        {
            return null;
        }
    }

    /**
     * Modifier for the 'upperRightY' attribute.
     *
     * @param upperRightY the new upperRightY value to set
     */
    public void setUpperRightY(String upperRightY)
    {
        this.upperRightY = upperRightY;
    }

    /**
     * Initialize.
     *
     * @exception KeyStoreException        the key store exception
     * @exception NoSuchAlgorithmException the no such algorithm exception
     * @exception CertificateException     the certificate exception
     * @exception IOException              Signals that an I/O exception has occurred.
     */
    public void initialize() throws KeyStoreException, NoSuchAlgorithmException, CertificateException, IOException
    {
        if (!this.initialized && this.path != null && this.password != null && !"".equals(this.path) &&
            !"".equals(this.password))
        {

            this.keyStoreAll = KeyStore.getInstance("JKS");
            this.keyStoreAll.load(null, null);

            this.keystore = KeyStore.getInstance(this.getType());
            this.keystore.load(new FileInputStream(this.path), this.password.toCharArray());

            this.alias = null;
            //keystore.aliases().nextElement();

            Enumeration<String> e = this.keystore.aliases();

            /* Add the certificate chain to the store */
            for (; e.hasMoreElements(); )
            {
                String a = e.nextElement();
                Certificate mainCert = this.keystore.getCertificate(a);
                this.keyStoreAll.setCertificateEntry(a, mainCert);

                Certificate[] certificados = this.keystore.getCertificateChain(a);

                if (certificados != null)
                {
                    // Will get the first alias that has certificates.
                    if (StringUtils.isEmpty(this.alias))
                    {
                        this.alias = a;
                    }

                    for (Certificate cert : certificados)
                    {
                        X509Certificate x509cert = (X509Certificate) cert;
                        this.keyStoreAll.setCertificateEntry(x509cert.getIssuerDN().getName(), x509cert);
                    }
                }
            }
            this.initialized = true;
        }
    }

    /**
     * Test certificate.
     *
     * @return the boolean
     */
    Boolean isCertificateValid()
    {
        if (this.isCertificateValid == null)
        {
            try
            {
                this.isCertificateValid = StringUtils.isNotBlank(this.getPath());
                if (this.isCertificateValid)
                {
                    CertificateManager.getInstance()
                            .signPDF(CertificateManager.createDummyPDF().toByteArray(), this.id);
                    this.isCertificateValid = true;
                }
            }
            catch (DocumentException e)
            {
                e.printStackTrace();
            }
            catch (IOException e)
            {
                e.printStackTrace();
            }
            catch (Exception e)
            {
                e.printStackTrace();
            }
        }
        return this.isCertificateValid;
    }
}
