package pt.digitalis.utils.documents.excel;

import org.apache.poi.hssf.usermodel.HSSFRichTextString;
import org.apache.poi.hssf.usermodel.HSSFWorkbook;
import org.apache.poi.ss.formula.FormulaParser;
import org.apache.poi.ss.formula.FormulaRenderer;
import org.apache.poi.ss.formula.FormulaType;
import org.apache.poi.ss.formula.ptg.Ptg;
import org.apache.poi.ss.formula.ptg.RefPtgBase;
import org.apache.poi.ss.usermodel.Cell;
import org.apache.poi.ss.usermodel.CellStyle;
import org.apache.poi.ss.usermodel.Comment;
import org.apache.poi.ss.usermodel.RichTextString;
import org.apache.poi.ss.usermodel.Row;
import org.apache.poi.ss.usermodel.Sheet;
import org.apache.poi.ss.usermodel.Workbook;
import org.apache.poi.ss.util.CellRangeAddress;
import org.apache.poi.xssf.usermodel.XSSFEvaluationWorkbook;
import org.apache.poi.xssf.usermodel.XSSFRichTextString;
import org.apache.poi.xssf.usermodel.XSSFWorkbook;
import pt.digitalis.utils.common.BeanInspector;

import java.io.ByteArrayOutputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;

/**
 * @author Galaio da Silva <a href="mailto:jgalaio@digitalis.pt">jgalaio@digitalis.pt</a><br/>
 * @created Dec 16, 2009
 */
public class ExcelDirectDocumentImpl implements IExcelDirectDocument
{

    /** cell position to start writing */
    private Integer cellPosition = 0;

    /** Excel document name */
    private String name;

    /** row position to start writing */
    private Integer rowPosition = 0;

    /** the sheet number in excel document. */
    private Integer sheetNumber = 0;

    /** Template Row */
    private Row templateRow = null;

    /** Work book */
    private Workbook workBook;

    /**
     * @param inputStream the inputStream containing the ExcelDocument
     */
    public ExcelDirectDocumentImpl(InputStream inputStream)
    {
        try
        {
            this.workBook = this.getWorkbookInstance(inputStream, true);
        }
        catch (IOException e)
        {
            e.printStackTrace();
        }
    }

    /**
     * @param inputStream the inputStream containing the ExcelDocument
     * @param oldFormat   specify if use old format
     */
    public ExcelDirectDocumentImpl(InputStream inputStream, boolean oldFormat)
    {
        try
        {
            this.workBook = this.getWorkbookInstance(inputStream, oldFormat);
        }
        catch (IOException e)
        {
            e.printStackTrace();
        }
    }

    /**
     * Constructor for <code>ExcelDirectDocumentImpl</code>
     *
     * @param path the Excel document path
     */
    public ExcelDirectDocumentImpl(String path)
    {
        InputStream inputStream;
        try
        {
            inputStream = Thread.currentThread().getContextClassLoader().getResourceAsStream(path);
            this.workBook = this.getWorkbookInstance(inputStream, true);
        }
        catch (FileNotFoundException e)
        {
            e.printStackTrace();
        }
        catch (IOException e)
        {
            e.printStackTrace();
        }
    }

    /**
     * Constructor for <code>ExcelDirectDocumentImpl</code>
     *
     * @param path      the Excel document path
     * @param oldFormat specify if use old format
     */
    public ExcelDirectDocumentImpl(String path, boolean oldFormat)
    {
        InputStream inputStream;
        try
        {
            inputStream = Thread.currentThread().getContextClassLoader().getResourceAsStream(path);
            this.workBook = this.getWorkbookInstance(inputStream, oldFormat);
        }
        catch (FileNotFoundException e)
        {
            e.printStackTrace();
        }
        catch (IOException e)
        {
            e.printStackTrace();
        }
    }

    /**
     * @see pt.digitalis.utils.documents.excel.IExcelDirectDocument#addCellValue(java.lang.Integer,
     *         java.lang.Integer,
     *         java.lang.Integer, java.lang.String)
     */
    public void addCellValue(Integer sheetNumber, Integer rowPosition, Integer cellPosition, String value)
    {
        Row hssfRow = workBook.getSheetAt(sheetNumber).getRow(rowPosition);
        if (hssfRow == null)
            hssfRow = workBook.getSheetAt(sheetNumber).createRow(rowPosition);

        Cell cell = workBook.getSheetAt(sheetNumber).getRow(rowPosition).getCell(cellPosition);
        if (cell == null)
        {
            cell = hssfRow.createCell(cellPosition.shortValue());
            if (this.getTemplateRow() != null && this.getTemplateRow().getCell(cellPosition.shortValue()) != null)
            {
                CellStyle cellStyle = this.getTemplateRow().getCell(cellPosition.shortValue()).getCellStyle();
                int cellType = this.getTemplateRow().getCell(cellPosition.shortValue()).getCellType();
                Comment comment = this.getTemplateRow().getCell(cellPosition.shortValue()).getCellComment();

                if (cellStyle != null)
                    cell.setCellStyle(cellStyle);
                if (comment != null)
                    cell.setCellComment(comment);

                cell.setCellType(cellType);
            }
        }

        RichTextString richTextString = null;
        if (workBook instanceof HSSFWorkbook)
            richTextString = new HSSFRichTextString(value);
        else
            richTextString = new XSSFRichTextString(value);

        cell.setCellValue(richTextString);
    }

    /**
     * @see pt.digitalis.utils.documents.excel.IExcelDirectDocument#addData(java.util.List, java.util.LinkedHashMap)
     */
    public void addData(List<?> data, LinkedHashMap<String, Integer> fieldsMap)
    {

        Iterator<String> keys = null;

        for (Object obj : data)
        {
            keys = fieldsMap.keySet().iterator();
            while (keys.hasNext())
            {
                String key = keys.next();
                String value = BeanInspector.getValueAsString(obj, key);
                if (value != null)
                    this.addCellValue(this.sheetNumber, this.rowPosition, fieldsMap.get(key), value);
            }
            this.rowPosition++;
        }
    }

    /**
     * @see pt.digitalis.utils.documents.excel.IExcelDirectDocument#addData(java.util.List, String[])
     */
    public void addData(List<?> data, String[] fields)
    {
        Integer cellPos = this.cellPosition;
        for (Object obj : data)
        {
            for (String field : fields)
            {
                if (field != null)
                {
                    String value = BeanInspector.getValueAsString(obj, field);
                    if (value != null)
                        this.addCellValue(this.sheetNumber, this.rowPosition, cellPos, value);
                }
                cellPos++;
            }
            cellPos = this.cellPosition;
            this.rowPosition++;
        }
    }

    /**
     * @see pt.digitalis.utils.documents.excel.IExcelDirectDocument#addData(java.util.List, String[],
     *         java.lang.Integer,
     *         java.lang.Integer)
     */
    public void addData(List<?> data, String[] fields, Integer startRow, Integer startCell)
    {
        this.addData(data, fields, startRow, startCell);
    }

    /**
     * @see pt.digitalis.utils.documents.excel.IExcelDirectDocument#addData(java.util.List, String[],
     *         java.lang.Integer,
     *         java.lang.Integer, java.lang.Integer)
     */
    public void addData(List<?> data, String[] fields, Integer sheetNumber, Integer startRow, Integer startCell)
    {
        if (sheetNumber != null)
            this.sheetNumber = sheetNumber;
        if (startRow != null)
            this.rowPosition = startRow;
        if (startCell != null)
            this.cellPosition = startCell;
        this.addData(data, fields);
    }

    /**
     * @see pt.digitalis.utils.documents.excel.IExcelDirectDocument#addData(java.util.Map, java.util.Map)
     */
    public void addData(Map<String, String> data, Map<String, Integer> fieldsMap)
    {
        for (Entry<String, Integer> entry : fieldsMap.entrySet())
        {
            String value = data.get(entry.getKey());
            if (value != null)
                this.addCellValue(this.sheetNumber, this.rowPosition, entry.getValue(), value);
        }
        this.rowPosition++;
    }

    /**
     * @see pt.digitalis.utils.documents.excel.IExcelDirectDocument#copyBlockOfRows(int, int, int, int)
     */
    public void copyBlockOfRows(int sheetNumber, int startPosition, int endPosition, int targetRowPosition)
    {
        int targetPosition = targetRowPosition;

        for (int rowPos = startPosition; rowPos <= endPosition; rowPos++)
        {
            this.copyRow(sheetNumber, rowPos, targetPosition++);
        }
    }

    /**
     * Copy a row from on position to another in the Excel Documen. <br>
     *
     * @param sheetNumber            the sheet number
     * @param sourceRowPosition      the source row position
     * @param destinationRowPosition the destination row position
     */
    public void copyRow(int sheetNumber, int sourceRowPosition, int destinationRowPosition)
    {
        // Get the source / new row
        Sheet worksheet = this.getWorkBook().getSheetAt(sheetNumber);
        Row newRow = worksheet.getRow(destinationRowPosition);
        Row sourceRow = worksheet.getRow(sourceRowPosition);

        ArrayList<String> fArray = new ArrayList<String>();
        Row origRow = worksheet.getRow(sourceRowPosition);
        for (int i = 0; i < origRow.getLastCellNum(); i++)
        {
            // if (origRow.getCell(i) != null && origRow.getCell(i).getCellType() == Cell.CELL_TYPE_FORMULA)
            // fArray.add(origRow.getCell(i).getCellFormula());
            // else
            // fArray.add(null);

            if (origRow.getCell(i) != null && origRow.getCell(i).getCellType() == Cell.CELL_TYPE_FORMULA)
            {

                String formula = origRow.getCell(i).getCellFormula();
                XSSFEvaluationWorkbook workbookWrapper =
                        XSSFEvaluationWorkbook.create((XSSFWorkbook) this.getWorkBook());
                /* parse formula */
                Ptg[] ptgs = FormulaParser.parse(formula, workbookWrapper, FormulaType.CELL, 0 /* sheet index */);

                /* re-calculate cell references */
                for (Ptg ptg : ptgs)
                {
                    if (ptg instanceof RefPtgBase) // base class for cell reference "things"
                    {
                        RefPtgBase ref = (RefPtgBase) ptg;
                        if (ref.isColRelative())
                            ref.setColumn(ref.getColumn() + 0);
                        if (ref.isRowRelative())
                            ref.setRow(destinationRowPosition);
                    }
                }

                formula = FormulaRenderer.toFormulaString(workbookWrapper, ptgs);
                fArray.add(formula);
                // row.getCell(i).setCellFormula(formula);
            }
            else
            {
                fArray.add(null);
            }
        }

        // If the row exist in destination, push down all rows by 1 else create a new row
        if (newRow != null)
        {
            worksheet.shiftRows(destinationRowPosition, worksheet.getLastRowNum(), 1);
        }
        else
        {
            newRow = worksheet.createRow(destinationRowPosition);
        }

        // Loop through source columns to add to new row
        for (int i = 0; i < sourceRow.getLastCellNum(); i++)
        {
            // Grab a copy of the old/new cell
            Cell oldCell = sourceRow.getCell(i);
            Cell newCell = newRow.createCell(i);

            // If the old cell is null jump to next cell
            if (oldCell == null)
            {
                newCell = null;
                continue;
            }

            // Copy style from old cell and apply to new cell
            CellStyle newCellStyle = this.getWorkBook().createCellStyle();
            newCellStyle.cloneStyleFrom(oldCell.getCellStyle());
            ;
            newCell.setCellStyle(newCellStyle);

            // If there is a cell comment, copy
            if (oldCell.getCellComment() != null)
            {
                newCell.setCellComment(oldCell.getCellComment());
            }

            // If there is a cell hyperlink, copy
            if (oldCell.getHyperlink() != null)
            {
                newCell.setHyperlink(oldCell.getHyperlink());
            }

            // Set the cell data type
            newCell.setCellType(oldCell.getCellType());

            // Set the cell data value
            switch (oldCell.getCellType())
            {
                case Cell.CELL_TYPE_BLANK:
                    newCell.setCellValue(oldCell.getStringCellValue());
                    break;
                case Cell.CELL_TYPE_BOOLEAN:
                    newCell.setCellValue(oldCell.getBooleanCellValue());
                    break;
                case Cell.CELL_TYPE_ERROR:
                    newCell.setCellErrorValue(oldCell.getErrorCellValue());
                    break;
                case Cell.CELL_TYPE_FORMULA:
                    newCell.setCellFormula(fArray.get(i));
                    break;
                case Cell.CELL_TYPE_NUMERIC:
                    newCell.setCellValue(oldCell.getNumericCellValue());
                    break;
                case Cell.CELL_TYPE_STRING:
                    newCell.setCellValue(oldCell.getRichStringCellValue());
                    break;
            }
        }

        // If there are are any merged regions in the source row, copy to new row
        for (int i = 0; i < worksheet.getNumMergedRegions(); i++)
        {
            CellRangeAddress cellRangeAddress = worksheet.getMergedRegion(i);
            if (cellRangeAddress.getFirstRow() == sourceRow.getRowNum())
            {
                CellRangeAddress newCellRangeAddress = new CellRangeAddress(newRow.getRowNum(),
                        (newRow.getRowNum() + (cellRangeAddress.getLastRow() - cellRangeAddress.getFirstRow())),
                        cellRangeAddress.getFirstColumn(), cellRangeAddress.getLastColumn());
                worksheet.addMergedRegion(newCellRangeAddress);
            }
        }
    }

    /**
     * @see pt.digitalis.utils.documents.excel.IExcelDirectDocument#deleteCell(java.lang.Integer, java.lang.Integer,
     *         java.lang.Integer)
     */
    public void deleteCell(Integer sheetNumber, Integer rowPosition, Integer cellPosition)
    {
        Cell cell = workBook.getSheetAt(sheetNumber).getRow(rowPosition).getCell(cellPosition);
        if (cell != null)
        {
            this.workBook.getSheetAt(sheetNumber).getRow(rowPosition).removeCell(cell);
        }
    }

    /**
     * @see pt.digitalis.utils.documents.IDocument#exportAsOutputStream()
     */
    public ByteArrayOutputStream exportAsOutputStream()
    {
        ByteArrayOutputStream output = null;
        // Write to the output
        try
        {
            output = new ByteArrayOutputStream();
            this.workBook.write(output);
            output.close();
        }
        catch (IOException e)
        {
            e.printStackTrace();
        }

        return output;
    }

    /**
     * @see pt.digitalis.utils.documents.IDocument#getBytes()
     */
    public byte[] getBytes()
    {
        byte[] result;
        ByteArrayOutputStream out = new ByteArrayOutputStream();
        try
        {
            this.workBook.write(out);
            result = out.toByteArray();
            out.flush();
            out.close();
        }
        catch (IOException e)
        {
            e.printStackTrace();
            return null;
        }
        return result;
    }

    /**
     * Inspector for the 'cellPosition' attribute.
     *
     * @return the cellPosition value
     */
    public Integer getCellPosition()
    {
        return cellPosition;
    }

    /**
     * @see pt.digitalis.utils.documents.excel.IExcelDirectDocument#setCellPosition(java.lang.Integer)
     */
    public void setCellPosition(Integer cellPosition)
    {
        this.cellPosition = cellPosition;
    }

    /**
     * @see pt.digitalis.utils.documents.excel.IExcelDirectDocument#getData(java.lang.Class, String[],
     *         java.lang.Integer, java.lang.Integer, java.lang.Integer, java.lang.Integer)
     */
    public <T extends Object> List<T> getData(Class<T> theClass, String[] fields, Integer sheetNumber, Integer startRow,
            Integer endRow, Integer startCell)
    {
        List<T> result = new ArrayList<T>();
        T obj = null;

        for (int row = startRow; row <= endRow; row++)
        {
            int cellPos = startCell;
            try
            {
                obj = theClass.newInstance();
                for (String field : fields)
                {
                    Cell hssfCell = this.workBook.getSheetAt(sheetNumber).getRow(row).getCell(cellPos++);
                    if (hssfCell != null)
                    {
                        BeanInspector.setNestedAtributeValue(obj, field, hssfCell.getRichStringCellValue().toString());
                    }
                }
                result.add(obj);
            }
            catch (InstantiationException e)
            {
                e.printStackTrace();
            }
            catch (IllegalAccessException e)
            {
                e.printStackTrace();
            }
        }
        return result;
    }

    /**
     * @see pt.digitalis.utils.documents.excel.IExcelDirectDocument#getData(java.lang.Integer, java.lang.Integer,
     *         java.lang.Integer, java.lang.Integer)
     */
    public List<String> getData(Integer sheetNumber, Integer startRow, Integer endRow, Integer cellNumber)
    {
        List<String> result = new ArrayList<String>();
        for (int row = startRow; row <= endRow; row++)
        {
            Cell hssfCell = this.workBook.getSheetAt(sheetNumber).getRow(row).getCell(cellNumber);
            if (hssfCell != null)
            {
                result.add(hssfCell.getRichStringCellValue().toString());
            }
        }
        return result;
    }

    /**
     * @see pt.digitalis.utils.documents.excel.IExcelDirectDocument#getData(java.lang.Integer, java.lang.Integer,
     *         java.lang.Integer, java.lang.Integer, java.lang.Integer)
     */
    public List<String[]> getData(Integer sheetNumber, Integer startRow, Integer endRow, Integer startCell,
            Integer endCell)
    {
        Sheet sheet = this.workBook.getSheetAt(sheetNumber);

        List<String[]> result = new ArrayList<String[]>();
        int numberCells = (endCell + 1) - startCell;
        if (endRow > sheet.getLastRowNum())
            endRow = sheet.getLastRowNum();

        for (int row = startRow; row <= endRow; row++)
        {
            int cellPos = 0;
            String[] cellData = new String[numberCells];

            Row hssfRow = sheet.getRow(row);
            if (hssfRow == null)
            {

            }
            else
            {
                for (int cell = startCell; cell <= endCell; cell++)
                {
                    Cell hssfCell = hssfRow.getCell(cell);
                    if (hssfCell != null)
                    {
                        if (hssfCell.getCellType() == Cell.CELL_TYPE_NUMERIC)
                            cellData[cellPos++] = new Double(hssfCell.getNumericCellValue()).toString();
                        else
                            cellData[cellPos++] = hssfCell.getRichStringCellValue().toString();
                    }
                    else
                        cellPos++;
                }
                result.add(cellData);
            }
        }
        return result;
    }

    /**
     * @see pt.digitalis.utils.documents.IDocument#getName()
     */
    public String getName()
    {
        return this.name;
    }

    /**
     * @see pt.digitalis.utils.documents.IDocument#setName(java.lang.String)
     */
    public void setName(String name)
    {
        this.name = name;
    }

    /**
     * @see pt.digitalis.utils.documents.excel.IExcelDirectDocument#getRowPosition()
     */
    public Integer getRowPosition()
    {
        return rowPosition;
    }

    /**
     * @see pt.digitalis.utils.documents.excel.IExcelDirectDocument#setRowPosition(java.lang.Integer)
     */
    public void setRowPosition(Integer rowPosition)
    {
        this.rowPosition = rowPosition;
    }

    /**
     * @see pt.digitalis.utils.documents.excel.IExcelDirectDocument#getSheetNumber()
     */
    public Integer getSheetNumber()
    {
        return sheetNumber;
    }

    /**
     * @see pt.digitalis.utils.documents.excel.IExcelDirectDocument#setSheetNumber(java.lang.Integer)
     */
    public void setSheetNumber(Integer sheetNumber)
    {
        this.sheetNumber = sheetNumber;
    }

    /**
     * Inspector for the 'initialRow' attribute.
     *
     * @return the initialRow value
     */
    public Row getTemplateRow()
    {
        return templateRow;
    }

    /**
     * Modifier for the 'initialRow' attribute.
     *
     * @param initialRow the new initialRow value to set
     */
    public void setTemplateRow(Row initialRow)
    {
        this.templateRow = initialRow;
    }

    /**
     * Inspector for the 'workBook' attribute.
     *
     * @return the workBook value
     */
    public Workbook getWorkBook()
    {
        return workBook;
    }

    /**
     * Gets the workbook.
     *
     * @param inputStream the input stream
     * @param oldFormat   the old format
     *
     * @return the workbook
     *
     * @exception IOException
     */
    private Workbook getWorkbookInstance(InputStream inputStream, boolean oldFormat) throws IOException
    {
        if (oldFormat)
            return new HSSFWorkbook(inputStream);
        else
            return new XSSFWorkbook(inputStream);
    }

    /**
     * @see pt.digitalis.utils.documents.IDocument#saveToFile(java.lang.String)
     */
    public void saveToFile(String fileName) throws IOException
    {
        FileOutputStream fout = new FileOutputStream(fileName);
        this.workBook.write(fout);
        fout.flush();
        fout.close();
    }

    /**
     * @see pt.digitalis.utils.documents.excel.IExcelDirectDocument#setCellValue(java.lang.Integer,
     *         java.lang.Integer,
     *         java.lang.Integer, java.lang.String)
     */
    public void setCellValue(Integer sheetNumber, Integer rowPosition, Integer cellPosition, String value)
    {
        Cell cell = workBook.getSheetAt(sheetNumber).getRow(rowPosition).getCell(cellPosition);
        if (cell == null)
        {
            addCellValue(sheetNumber, rowPosition, cellPosition, value);
        }
        else
        {
            if (workBook instanceof XSSFWorkbook)
            {
                cell.setCellValue(new XSSFRichTextString(value));
            }
            else
            {
                cell.setCellValue(new HSSFRichTextString(value));
            }
        }
    }

    /**
     * @see pt.digitalis.utils.documents.excel.IExcelDirectDocument#setTemplateRow(java.lang.Integer,
     *         java.lang.Integer)
     */
    public void setTemplateRow(Integer sheetNumber, Integer rowNumber)
    {
        this.templateRow = workBook.getSheetAt(sheetNumber).getRow(rowNumber);
    }
}
