package pt.digitalis.utils.documents.excel;

import org.apache.commons.lang.math.NumberUtils;
import org.apache.poi.hssf.usermodel.HSSFCell;
import org.apache.poi.hssf.usermodel.HSSFCellStyle;
import org.apache.poi.hssf.usermodel.HSSFRichTextString;
import org.apache.poi.hssf.usermodel.HSSFRow;
import org.apache.poi.hssf.usermodel.HSSFSheet;
import org.apache.poi.hssf.usermodel.HSSFWorkbook;
import pt.digitalis.utils.common.StringUtils;
import pt.digitalis.utils.documents.IDocument;

import java.io.ByteArrayOutputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;

/**
 * Excel document wrapper class
 *
 * @author Antnio Silva <a href="mailto:asilva@digitalis.pt">asilva@digitalis.pt</a><br/>
 * @author Pedro Viegas <a href="mailto:pviegas@digitalis.pt">pviegas@digitalis.pt</a><br/>
 * @created 2008/08/01
 */
public class ExcelDocument implements IDocument
{

    /**
     * Document sheets
     */
    private final Map<String, ExcelSheet> sheets = new HashMap<String, ExcelSheet>();

    /** A simple index to maintain insertion order for sheets */
    List<String> sheetOrder = new ArrayList<String>();

    /**
     * The name of the Excel Document
     */
    private String name;

    /** The styles cache for reuse. */
    private Map<String, HSSFCellStyle> styles = new HashMap<String, HSSFCellStyle>();

    /** Use plain text */
    private Boolean usePlainText = false;

    /** the result native excel object */
    private HSSFWorkbook workbook = null;

    /**
     * Default constructor
     *
     * @param name
     */
    public ExcelDocument(String name)
    {
        this.name = name;
    }

    /**
     * Adds a set of data to the Excel
     *
     * @param data the data to insert
     *
     * @return the updated sheet
     */
    public ExcelDocument addData(List<?> data)
    {
        getFirstSheet().addData(data);

        return this;
    }

    /**
     * Adds a set of data to the Excel
     *
     * @param data       the data to insert
     * @param properties the properties to export (supports object graph navigation)
     *
     * @return the updated sheet
     */
    public ExcelDocument addData(List<?> data, List<String> properties)
    {

        getFirstSheet().addData(data, properties);

        return this;
    }

    /**
     * Adds a set of data to the Excel
     *
     * @param data       the data to insert
     * @param properties the properties description map to export (supports object graph navigation). The key is the
     *                   title, the
     *                   value is the property path.
     *
     * @return the updated sheet
     */
    public ExcelDocument addData(List<?> data, Map<String, String> properties)
    {

        getFirstSheet().addData(data, properties);

        return this;
    }

    /**
     * Adds a set of data to the Excel
     *
     * @param data the data to insert
     *
     * @return the updated sheet
     */
    public ExcelDocument addData(Map<?, ?> data)
    {
        getFirstSheet().addData(data);

        return this;
    }

    /**
     * Read the Excel Document from file.
     *
     * @param path
     *            the file path
     */
    /*
     * public void readFromFile(String path) { InputStream inputStream; HSSFWorkbook workBook; try { inputStream = new
     * FileInputStream(path); workBook = new HSSFWorkbook(inputStream); for (int sheetNumber = 0; sheetNumber <
     * workBook.getNumberOfSheets(); sheetNumber++) { HSSFSheet sheet = workBook.getSheetAt(sheetNumber); ExcelSheet
     * excelSheet = this.addSheet(workBook.getSheetName(sheetNumber)); Iterator<Row> rowIterator = sheet.rowIterator();
     * for (int rowNumber = 0; rowNumber < sheet.getPhysicalNumberOfRows(); rowNumber++) { HSSFRow row =
     * sheet.getRow(rowNumber); ExcelRow excelRow = excelSheet.addRow(); if (row != null) { for(int cellNumber = 0;
     * cellNumber < row.getPhysicalNumberOfCells() ; cellNumber ++) { HSSFCell cell = row.getCell(cellNumber); ExcelCell
     * excelCell = excelRow.addCell(null); if (cell != null)
     * excelCell.setValue(cell.getRichStringCellValue().toString()); } } } } } catch (FileNotFoundException e) {
     * e.printStackTrace(); } catch (IOException e) { e.printStackTrace(); } }
     */

    /**
     * Adds a set of data to the Excel
     *
     * @param obj the obj to insert
     *
     * @return the updated document
     */
    public ExcelDocument addObject(Object obj)
    {
        getFirstSheet().addObject(obj);

        return this;
    }

    /**
     * Adds a set of data to the Excel
     *
     * @param obj        the obj to insert
     * @param properties the properties to export (supports object graph navigation)
     *
     * @return the updated sheet
     */
    public ExcelDocument addObject(Object obj, List<String> properties)
    {

        getFirstSheet().addObject(obj, properties);

        return this;
    }

    /**
     * Adds a set of data to the Excel
     *
     * @param title a prefix text to place in a column at the left of this data
     * @param obj   the obj to insert
     *
     * @return the updated sheet
     */
    public ExcelDocument addObject(String title, Object obj)
    {
        getFirstSheet().addObject(title, obj);

        return this;
    }

    /**
     * Adds a new sheet to the given workbook
     *
     * @param name the name of the new sheet to create
     *
     * @return the created sheet
     */
    public ExcelSheet addSheet(String name)
    {
        ExcelSheet sheet = new ExcelSheet(name, this);
        sheets.put(name, sheet);
        sheetOrder.add(name);

        return sheet;
    }

    /**
     * Creates a {@link HSSFWorkbook} entry for the current Sheets
     *
     * @return the {@link HSSFWorkbook} instance
     */
    protected HSSFWorkbook buildWorkBook()
    {
        HSSFWorkbook workbook = new HSSFWorkbook();
        // XSSFWorkbook teste = new XSSFWorkbook();

        for (ExcelSheet sheet : this.getSheets())
        {
            HSSFSheet hssfSheet = workbook.createSheet(sheet.getName());

            // iterates over the rows
            for (Entry<Integer, ExcelRow> row : sheet.getRows().entrySet())
            {
                HSSFRow hssfRow = hssfSheet.createRow(row.getKey() - 1);
                // iterates over the cells
                for (Entry<Integer, ExcelCell> cell : row.getValue().getCells().entrySet())
                {
                    HSSFCell hssfCell = hssfRow.createCell(new Integer(cell.getKey() - 1));

                    if (NumberUtils.isNumber(cell.getValue().getValue()))
                    {
                        Number number = NumberUtils.createNumber(cell.getValue().getValue());
                        HSSFCellStyle cellStyle = getNumberStyle(workbook, cell.getValue().getValue());

                        if (cellStyle != null)
                            hssfCell.setCellStyle(cellStyle);

                        hssfCell.setCellValue(number.doubleValue());
                    }
                    else
                    {
                        HSSFRichTextString value = new HSSFRichTextString(cell.getValue().getValue());
                        hssfCell.setCellValue(value);
                    }
                }
            }
        }

        return workbook;
    }

    /**
     * @see pt.digitalis.utils.documents.IDocument#exportAsOutputStream()
     */
    public ByteArrayOutputStream exportAsOutputStream()
    {

        ByteArrayOutputStream output = null;

        // Write to the output
        try
        {
            output = new ByteArrayOutputStream();
            getWorkbook().write(output);
            output.close();
        }
        catch (IOException e)
        {
            e.printStackTrace();
        }

        return output;
    }

    /**
     * @see pt.digitalis.utils.documents.IDocument#getBytes()
     */
    public byte[] getBytes()
    {
        return this.getWorkbook().getBytes();
    }

    /**
     * @return the current document first sheet
     */
    private ExcelSheet getFirstSheet()
    {
        if (sheets.isEmpty())
            addSheet("Sheet1");

        return sheets.get(sheetOrder.get(0));
    }

    /**
     * @see pt.digitalis.utils.documents.IDocument#getName()
     */
    public String getName()
    {
        return name;
    }

    /**
     * @see pt.digitalis.utils.documents.IDocument#setName(java.lang.String)
     */
    public void setName(String name)
    {
        this.name = name;
    }

    /**
     * Gets the number style for numbers according to the presence of decimal part or not.
     *
     * @param workbookParam
     * @param value         the value
     *
     * @return the number style
     */
    private HSSFCellStyle getNumberStyle(HSSFWorkbook workbookParam, String value)
    {
        String format = "";
        if (value.contains(".") || value.contains(","))
            format = value.replaceAll("[0-9]", "#");

        if (StringUtils.isNotBlank(format))
        {
            HSSFCellStyle cellStyle = this.styles.get(format);

            if (cellStyle == null)
            {
                cellStyle = workbookParam.createCellStyle();
                cellStyle.setDataFormat(workbookParam.getCreationHelper().createDataFormat().getFormat(format));

                this.styles.put(format, cellStyle);
            }

            return cellStyle;
        }
        else
            return null;
    }

    /**
     * Gets a specific sheet
     *
     * @param name the name of the sheet
     *
     * @return the sheet
     */
    public ExcelSheet getSheet(String name)
    {
        return sheets.get(name);
    }

    /**
     * @return the sheets
     */
    public List<ExcelSheet> getSheets()
    {
        List<ExcelSheet> results = new ArrayList<ExcelSheet>();

        for (String id : sheetOrder)
        {
            results.add(sheets.get(id));
        }

        return results;
    }

    /**
     * @return the usePlainText
     */
    public Boolean getUsePlainText()
    {
        return usePlainText;
    }

    /**
     * @param usePlainText the usePlainText to set
     */
    public void setUsePlainText(Boolean usePlainText)
    {
        this.usePlainText = usePlainText;
    }

    /**
     * Inspector for the 'workbook' attribute.
     *
     * @return the workbook value
     */
    public HSSFWorkbook getWorkbook()
    {
        if (this.workbook == null)
            this.workbook = buildWorkBook();

        return this.workbook;
    }

    /**
     * @see pt.digitalis.utils.documents.IDocument#saveToFile(java.lang.String)
     */
    public void saveToFile(String fileName) throws IOException
    {
        FileOutputStream fout = new FileOutputStream(fileName);
        fout.write(exportAsOutputStream().toByteArray());
        fout.flush();
        fout.close();
    }
}
