package pt.digitalis.utils.documents.excel;

import org.apache.commons.lang.StringEscapeUtils;
import org.jsoup.Jsoup;
import pt.digitalis.utils.common.BeanInspector;
import pt.digitalis.utils.common.IBeanAttributes;
import pt.digitalis.utils.common.IBeanPropertyInspector;
import pt.digitalis.utils.common.StringUtils;

import java.sql.Date;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;

/**
 * A workbook sheet object
 *
 * @author Antnio Silva <a href="mailto:asilva@digitalis.pt">asilva@digitalis.pt</a><br/>
 * @author Pedro Viegas <a href="mailto:pviegas@digitalis.pt">pviegas@digitalis.pt</a><br/>
 * @created 2008/08/01
 */
public class ExcelSheet
{

    /** List of unitary classes to convert using toString */
    @SuppressWarnings("rawtypes")
    final static private List<Class> unitaryClasses = new ArrayList<Class>()
    {

        /**  */
        private static final long serialVersionUID = 3607610035754690014L;

        {
            add(Integer.class);
            add(Long.class);
            add(Boolean.class);
            add(Character.class);
            add(String.class);
            add(Short.class);
            add(Date.class);
            add(Double.class);
        }
    };

    /**
     * All sheet rows
     */
    private final Map<Integer, ExcelRow> rows = new HashMap<Integer, ExcelRow>();

    /** the maximum cell inserted */
    Integer maxRowIndex = 0;

    /** {@link ExcelDocument} */
    private ExcelDocument excelDocument;

    /**
     * the name of the sheet
     */
    private String name;

    /**
     * Default constructor
     *
     * @param name
     */
    public ExcelSheet(String name)
    {
        this.name = name;
    }

    /**
     * Default constructor
     *
     * @param name
     * @param excelDocument the {@link ExcelDocument}
     */
    public ExcelSheet(String name, ExcelDocument excelDocument)
    {
        this.name = name;
        this.excelDocument = excelDocument;
    }

    /**
     * Adds a set of data to the Excel
     *
     * @param data the data to insert
     *
     * @return the updated sheet
     */
    public ExcelSheet addData(List<?> data)
    {

        for (Object obj : data)
        {
            addObject(obj);
        }

        return this;
    }

    /**
     * Adds a set of data to the Excel
     *
     * @param data       the data to insert
     * @param properties the properties to export (supports object graph navigation)
     *
     * @return the updated sheet
     */
    public ExcelSheet addData(List<?> data, List<String> properties)
    {

        for (Object obj : data)
        {
            addObject(obj, properties);
        }

        return this;
    }

    /**
     * Adds a set of data to the Excel
     *
     * @param data       the data to insert
     * @param properties the properties description map to export (supports object graph navigation). The key is the
     *                   title, the
     *                   value is the property path.
     *
     * @return the updated sheet
     */
    public ExcelSheet addData(List<?> data, Map<String, String> properties)
    {

        // Add the title row
        ExcelRow row = addRow();
        for (String title : properties.keySet())
        {
            row.addCell(title);
        }

        // Get the property path and call addData to add all data values
        List<String> propertiesPath = new ArrayList<String>();
        propertiesPath.addAll(properties.values());

        addData(data, propertiesPath);

        return this;
    }

    /**
     * Adds a set of data to the Excel
     *
     * @param data the data to insert
     *
     * @return the updated sheet
     */
    public ExcelSheet addData(Map<?, ?> data)
    {

        for (Entry<?, ?> entry : data.entrySet())
        {
            addObject(entry.getKey().toString(), entry.getValue());
        }

        return this;
    }

    /**
     * Adds a set of data to the Excel
     *
     * @param data       the data to insert
     * @param properties the properties to export (supports object graph navigation)
     *
     * @return the updated sheet
     */
    public ExcelSheet addDataFromBeans(List<IBeanAttributes> data, List<String> properties)
    {

        for (IBeanAttributes obj : data)
        {
            addObject(obj, properties);
        }

        return this;
    }

    /**
     * Adds a set of data to the Excel
     *
     * @param bean  the bean to insert
     * @param attrs the attributes list
     *
     * @return the updated sheet
     */
    public ExcelSheet addObject(IBeanAttributes bean, List<String> attrs)
    {

        if (!attrs.isEmpty() && bean != null)
        {
            ExcelRow row = this.addRow();

            for (String attrName : attrs)
            {
                row.addCell(bean.getAttributeAsString(attrName));
            }
        }

        return this;
    }

    /**
     * Adds a set of data to the Excel
     *
     * @param obj the obj to insert
     *
     * @return the updated sheet
     */
    public ExcelSheet addObject(Object obj)
    {

        Collection<String> values = BeanInspector.getAttributeValues(obj).values();

        if (!values.isEmpty())
        {
            ExcelRow row = this.addRow();

            for (String value : values)
            {
                row.addCell(value);
            }
        }

        return this;
    }

    /**
     * Adds a set of data to the Excel
     *
     * @param obj        the obj to insert
     * @param properties the properties to export (supports object graph navigation)
     *
     * @return the updated sheet
     */
    public ExcelSheet addObject(Object obj, List<String> properties)
    {
        if (obj != null)
        {
            if (properties != null && !properties.isEmpty())
            {
                ExcelRow row = this.addRow();

                for (String prop : properties)
                {
                    String value = this.getObjectPropertyValue(obj, prop);

                    if (value == null)
                        value = this.getObjectPropertyValue(obj, prop.replaceAll("\\.", "\\_"));
                    else if (excelDocument.getUsePlainText())
                    {
                        String plainText = StringEscapeUtils.unescapeHtml(value);
                        value = Jsoup.parse(plainText).text();
                    }

                    row.addCell(value);
                }
            }
        }

        return this;
    }

    /**
     * Adds a set of data to the Excel
     *
     * @param title a prefix text to place in a column at the left of this data
     * @param obj   the obj to insert
     *
     * @return the updated sheet
     */
    public ExcelSheet addObject(String title, Object obj)
    {

        if (unitaryClasses.contains(obj.getClass()))
        {
            ExcelRow row = this.addRow();
            row.addCell(title);
            row.addCell(obj.toString());
        }
        else
        {
            Collection<String> values = BeanInspector.getAttributeValues(obj).values();

            if (!values.isEmpty())
            {
                ExcelRow row = this.addRow();
                row.addCell(title);

                for (String value : values)
                {
                    row.addCell(value);
                }
            }
        }

        return this;
    }

    /**
     * Adds a new row to the sheet
     *
     * @return the created row
     */
    public ExcelRow addRow()
    {
        return addRow(maxRowIndex + 1);
    }

    /**
     * Adds a new row to the sheet
     *
     * @param index the index of the new row
     *
     * @return the created row
     */
    public ExcelRow addRow(Integer index)
    {
        if (index > maxRowIndex)
            maxRowIndex = index;

        ExcelRow row = new ExcelRow();
        rows.put(index, row);

        return row;
    }

    /**
     * @return the excelDocument
     */
    public ExcelDocument getExcelDocument()
    {
        return excelDocument;
    }

    /**
     * @return the name
     */
    public String getName()
    {
        return name;
    }

    /**
     * @param name the name to set
     */
    public void setName(String name)
    {
        this.name = name;
    }

    /**
     * Returns the inner object property by the given name.<br/>
     * Will support view fields name translations (i.e. someProp.innerProp -> someProp_innerProp)
     *
     * @param obj          the object to extract the property from
     * @param propertyName the property name
     *
     * @return the property value
     */
    @SuppressWarnings("rawtypes")
    private String getObjectPropertyValue(Object obj, String propertyName)
    {
        boolean implementsPropertyInspector = obj instanceof IBeanPropertyInspector;
        boolean isMap = obj instanceof Map;
        IBeanPropertyInspector beanInspector = implementsPropertyInspector ? (IBeanPropertyInspector) obj : null;

        if (implementsPropertyInspector)
        {
            if (beanInspector != null)
                return beanInspector.getAttributeAsString(propertyName);
            else
                return null;
        }
        else if (isMap)
            return StringUtils.toStringOrNull(((Map) obj).get(propertyName));
        else
            return BeanInspector.getValueAsString(obj, propertyName);
    }

    /**
     * @param index the row index to return
     *
     * @return the row
     */
    public ExcelRow getRow(Integer index)
    {
        return rows.get(index);
    }

    /**
     * @return the rows
     */
    public Map<Integer, ExcelRow> getRows()
    {
        return rows;
    }
}
