/**
 * 2008, Digitalis Informatica. All rights reserved. Distribuicao e Gestao de Informatica, Lda. Estrada de Paco de Arcos
 * num.9 - Piso -1 2780-666 Paco de Arcos Telefone: (351) 21 4408990 Fax: (351) 21 4408999 http://www.digitalis.pt
 */

package pt.digitalis.utils.reporting.impl.jasperreports;

import net.sf.jasperreports.engine.JRBand;
import net.sf.jasperreports.engine.JRDataSource;
import net.sf.jasperreports.engine.JREmptyDataSource;
import net.sf.jasperreports.engine.JRException;
import net.sf.jasperreports.engine.JRExporterParameter;
import net.sf.jasperreports.engine.JRGroup;
import net.sf.jasperreports.engine.JRQuery;
import net.sf.jasperreports.engine.JRQueryChunk;
import net.sf.jasperreports.engine.JasperCompileManager;
import net.sf.jasperreports.engine.JasperExportManager;
import net.sf.jasperreports.engine.JasperFillManager;
import net.sf.jasperreports.engine.JasperPrint;
import net.sf.jasperreports.engine.JasperReport;
import net.sf.jasperreports.engine.base.JRBaseSubreport;
import net.sf.jasperreports.engine.export.JRHtmlExporter;
import net.sf.jasperreports.engine.export.JRHtmlExporterParameter;
import net.sf.jasperreports.engine.export.JRXlsExporter;
import net.sf.jasperreports.engine.export.JRXlsExporterParameter;
import net.sf.jasperreports.engine.util.JRLoader;
import net.sf.jasperreports.engine.util.JRProperties;
import net.sf.jasperreports.engine.util.JRStyledText;
import pt.digitalis.utils.common.StringUtils;
import pt.digitalis.utils.reporting.AbstractDIFReport;
import pt.digitalis.utils.reporting.IReportDataSource;
import pt.digitalis.utils.reporting.ReportExportFormat;
import pt.digitalis.utils.reporting.exception.ReportingException;

import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.URL;
import java.sql.Connection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * DIFReport implementation for JasperReports.
 *
 * @author Rodrigo Gonalves <a href="mailto:rgoncalves@digitalis.pt">rgoncalves@digitalis.pt</a><br/>
 * @created 2008/12/10
 */
public class ReportJasperImpl extends AbstractDIFReport
{

    /** The JasperPrint object. */
    private JasperPrint jasperPrint;

    /** The JasperReport object. */
    private JasperReport jasperReport;

    /**
     * Fills the report with data. Synchronized operation to prevent concurrent access causing multi-threaded errors.
     *
     * @param report         the report to fill
     * @param jdbcConnection the JDBC connection to get the data from
     * @param parameters     the parameter to pass to the report
     *
     * @return the report filled with the data
     *
     * @exception JRException if the report can't be filled
     */
    static private synchronized JasperPrint fillReportFromConnection(JasperReport report, Connection jdbcConnection,
            Map<String, Object> parameters) throws JRException
    {
        return JasperFillManager.fillReport(report, parameters, jdbcConnection);
    }

    /**
     * Fills the report with data. Synchronized operation to prevent concurrent access causing multi-threaded errors.
     *
     * @param report     the report to fill
     * @param dataSource the data source to get the data from
     * @param parameters the parameter to pass to the report
     *
     * @return the report filled with the data
     *
     * @exception JRException if the report can't be filled
     */
    static private synchronized JasperPrint fillReportFromDataSource(JasperReport report, JRDataSource dataSource,
            Map<String, Object> parameters) throws JRException
    {
        return JasperFillManager.fillReport(report, parameters, dataSource);
    }

    /**
     * @see pt.digitalis.utils.reporting.IReport#compileReport()
     */
    public void compileReport() throws ReportingException
    {
        try
        {
            if (getJasperReport() == null)
            {
                InputStream templateInputStream = null;
                String baseDir = null;

                if (StringUtils.isNotBlank(getTemplatePath()))
                {
                    baseDir = StringUtils
                            .substringBeforeLast(StringUtils.substringBeforeLast(getTemplatePath(), "\\"), "/");
                    templateInputStream =
                            Thread.currentThread().getContextClassLoader().getResourceAsStream(getTemplatePath());
                }
                else if (this.getTemplateInputStream() != null)
                {
                    templateInputStream = this.getTemplateInputStream();
                }

                if (templateInputStream == null)
                    try
                    {
                        templateInputStream = new FileInputStream(getTemplatePath());
                    }
                    catch (FileNotFoundException e)
                    {
                        throw new JRException(e);
                    }

                setJasperReport(JasperCompileManager.compileReport(templateInputStream));
                compileSubreports(getJasperReport(), baseDir);
            }
        }
        catch (JRException exception)
        {
            /* Always reset the report */
            setJasperReport(null);
            throw new ReportingException(ReportingException.REPORT_CANT_BE_COMPILED_MESSAGE + getTemplatePath(),
                    exception);
        }
    }

    /**
     * Compile all the sub-reports for one section
     *
     * @param baseDir the base directory to find the filed on disk. For files not in the class path via getResources
     *                that we
     *                must get via FileInputStrem with full relative path
     * @param list
     *
     * @exception JRException
     */
    private void compileSectionSubreports(String baseDir, List<?> list) throws JRException
    {
        for (Object obj : list)
        {
            if (obj instanceof JRBaseSubreport)
            {
                String jasperPath = ((JRBaseSubreport) obj).getExpression().getText();
                jasperPath = StringUtils.removeEnd(jasperPath, "\"");
                jasperPath = jasperPath.substring(StringUtils.lastIndexOf(jasperPath, '\"') + 1);
                String jrxmlPath = jasperPath.replace(".jasper", ".jrxml");
                String jrxmlPathWithBaseDir = "";
                String jasperPathWithBaseDir = "";

                String jrxmlPathBaseDir =
                        StringUtils.substringBeforeLast(StringUtils.substringBeforeLast(jrxmlPath, "\\"), "/");

                if (!StringUtils.isBlank(baseDir) &&
                    (StringUtils.isBlank(jrxmlPathBaseDir) || !jrxmlPathBaseDir.equals(baseDir)))
                {
                    jrxmlPathWithBaseDir = baseDir + "/" + jrxmlPath;
                    jasperPathWithBaseDir = baseDir + "/" + jasperPath;
                }

                boolean found = false;

                // Try the approach via getResource
                // (gets files in the JAR/WAR/EAR and classpath directories, via classloader)
                URL jasperFile = null;

                if (!"".equals(jasperPathWithBaseDir))
                    jasperFile = Thread.currentThread().getContextClassLoader().getResource(jasperPathWithBaseDir);

                /* The report compilation only happens when needed. The xxx.jasper file doesn't exist. */
                if (jasperFile == null)
                {
                    URL jrxmlFile = null;

                    if (!"".equals(jrxmlPathWithBaseDir))
                        jrxmlFile = Thread.currentThread().getContextClassLoader().getResource(jrxmlPathWithBaseDir);

                    // Has report template (jrxml) but not the compiled one (jasper), compile it!
                    if (jrxmlFile != null)
                    {
                        JasperCompileManager.compileReportToFile(jrxmlFile.getFile());
                        found = true;
                    }
                }
                else
                    found = true;

                if (!found)
                    jasperFile = Thread.currentThread().getContextClassLoader().getResource(jasperPath);

                if (!found && jasperFile == null)
                {
                    URL jrxmlFile = Thread.currentThread().getContextClassLoader().getResource(jrxmlPath);

                    if (jrxmlFile != null)
                    {
                        String subReportPath = JasperCompileManager.compileReportToFile(jrxmlFile.getFile());

                        JasperReport subReport = (JasperReport) JRLoader.loadObjectFromFile(subReportPath);
                        this.compileSubreports(subReport, baseDir);

                        found = true;
                    }
                }
                else
                    found = true;

                // Did not find the file, try to get it via direct file system
                // (current dir relative path, or absolute path, via java.io)
                if (!found)
                {
                    try
                    {
                        FileOutputStream jasperStream = new FileOutputStream(
                                !"".equals(jasperPathWithBaseDir) ? jasperPathWithBaseDir : jasperPath);
                        FileInputStream jrxmlStream = new FileInputStream(
                                !"".equals(jrxmlPathWithBaseDir) ? jrxmlPathWithBaseDir : jrxmlPath);

                        JasperCompileManager.compileReportToStream(jrxmlStream, jasperStream);
                    }
                    catch (FileNotFoundException e)
                    {
                        throw new JRException(e);
                    }
                }
            }
        }
    }

    /**
     * Compile all the sub-reports.
     *
     * @param jr      the jasper report
     * @param baseDir the base directory to find the filed on disk. For files not in the class path vua getResources
     *                that we
     *                must get via FileInputStrem with full relative path
     *
     * @exception JRException the JR exception
     */
    private void compileSubreports(JasperReport jr, String baseDir) throws JRException
    {
        if (jr != null)
        {
            if (jr.getGroups() != null)
            {
                for (JRGroup object : jr.getGroups())
                {
                    if (object.getGroupHeaderSection().getBands() != null)
                    {
                        for (JRBand band : object.getGroupHeaderSection().getBands())
                        {
                            compileSectionSubreports(baseDir, band.getChildren());
                        }
                    }
                    if (object.getGroupFooterSection().getBands() != null)
                    {
                        for (JRBand band : object.getGroupFooterSection().getBands())
                        {
                            compileSectionSubreports(baseDir, band.getChildren());
                        }
                    }
                }
            }
            if (jr.getColumnHeader() != null)
            {
                compileSectionSubreports(baseDir, jr.getColumnHeader().getChildren());
            }

            if (jr.getColumnFooter() != null)
            {
                compileSectionSubreports(baseDir, jr.getColumnFooter().getChildren());
            }

            if (jr.getDetailSection() != null)
            {
                for (JRBand band : jr.getDetailSection().getBands())
                {
                    compileSectionSubreports(baseDir, band.getChildren());
                }
            }

            if (jr.getSummary() != null)
            {
                compileSectionSubreports(baseDir, jr.getSummary().getChildren());
            }

            if (jr.getBackground() != null)
            {
                compileSectionSubreports(baseDir, jr.getBackground().getChildren());
            }

            if (jr.getLastPageFooter() != null)
            {
                compileSectionSubreports(baseDir, jr.getLastPageFooter().getChildren());
            }

            if (jr.getNoData() != null)
            {
                compileSectionSubreports(baseDir, jr.getNoData().getChildren());
            }
        }
    }

    /**
     * @see pt.digitalis.utils.reporting.IReport#exportToFile(java.lang.String)
     */
    public void exportToFile(String outputFileName) throws ReportingException
    {
        try
        {
            String outFile = outputFileName + super.getExportFormat().getFileExtension();

            if (super.getExportFormat().equals(ReportExportFormat.PDF))
                JasperExportManager.exportReportToPdfFile(getJasperPrint(), outFile);

            else if (super.getExportFormat().equals(ReportExportFormat.XLS))
            {
                JRXlsExporter exporter = new JRXlsExporter();

                exporter.setParameter(JRXlsExporterParameter.IS_DETECT_CELL_TYPE, Boolean.TRUE);
                exporter.setParameter(JRXlsExporterParameter.IS_WHITE_PAGE_BACKGROUND, Boolean.FALSE);
                exporter.setParameter(JRXlsExporterParameter.IS_REMOVE_EMPTY_SPACE_BETWEEN_COLUMNS, Boolean.TRUE);
                exporter.setParameter(JRXlsExporterParameter.IS_REMOVE_EMPTY_SPACE_BETWEEN_ROWS, Boolean.TRUE);

                exporter.setParameter(JRExporterParameter.JASPER_PRINT, getJasperPrint());
                exporter.setParameter(JRExporterParameter.OUTPUT_FILE_NAME, outFile);

                exporter.exportReport();
            }

            else if (super.getExportFormat().equals(ReportExportFormat.HTML))
            {
                JRHtmlExporter exporter = new JRHtmlExporter();

                exporter.setParameter(JRHtmlExporterParameter.IS_USING_IMAGES_TO_ALIGN, Boolean.FALSE);

                exporter.setParameter(JRExporterParameter.JASPER_PRINT, getJasperPrint());
                exporter.setParameter(JRExporterParameter.OUTPUT_FILE_NAME, outFile);

                exporter.exportReport();
            }

            else if (super.getExportFormat().equals(ReportExportFormat.XML))
                JasperExportManager.exportReportToXmlFile(getJasperPrint(), outFile, true);
        }
        catch (JRException exception)
        {
            throw new ReportingException(ReportingException.REPORT_CANT_BE_EXPORTED_MESSAGE + outputFileName +
                                         super.getExportFormat().getFileExtension(), exception);
        }
    }

    /**
     * @see pt.digitalis.utils.reporting.IReport#exportToStream(java.io.OutputStream)
     */
    public void exportToStream(OutputStream outputStream) throws ReportingException
    {
        try
        {
            if (super.getExportFormat().equals(ReportExportFormat.PDF))
                JasperExportManager.exportReportToPdfStream(getJasperPrint(), outputStream);

            else if (super.getExportFormat().equals(ReportExportFormat.XML))
                JasperExportManager.exportReportToXmlStream(getJasperPrint(), outputStream);

            else if (super.getExportFormat().equals(ReportExportFormat.XLS))
            {
                JRXlsExporter exporter = new JRXlsExporter();

                exporter.setParameter(JRXlsExporterParameter.IS_DETECT_CELL_TYPE, Boolean.TRUE);
                exporter.setParameter(JRXlsExporterParameter.IS_WHITE_PAGE_BACKGROUND, Boolean.FALSE);
                exporter.setParameter(JRXlsExporterParameter.IS_REMOVE_EMPTY_SPACE_BETWEEN_COLUMNS, Boolean.TRUE);
                exporter.setParameter(JRXlsExporterParameter.IS_REMOVE_EMPTY_SPACE_BETWEEN_ROWS, Boolean.TRUE);

                exporter.setParameter(JRExporterParameter.JASPER_PRINT, getJasperPrint());
                exporter.setParameter(JRExporterParameter.OUTPUT_STREAM, outputStream);

                exporter.exportReport();
            }

            else if (super.getExportFormat().equals(ReportExportFormat.HTML))
            {
                JRHtmlExporter exporter = new JRHtmlExporter();

                exporter.setParameter(JRHtmlExporterParameter.IS_USING_IMAGES_TO_ALIGN, Boolean.FALSE);

                exporter.setParameter(JRExporterParameter.JASPER_PRINT, getJasperPrint());
                exporter.setParameter(JRExporterParameter.OUTPUT_STREAM, outputStream);

                exporter.exportReport();
            }
        }
        catch (JRException exception)
        {
            throw new ReportingException(ReportingException.REPORT_CANT_BE_EXPORTED_MESSAGE + outputStream.toString(),
                    exception);
        }
    }

    /**
     * @see pt.digitalis.utils.reporting.IReport#fillReport()
     */
    public void fillReport() throws ReportingException
    {
        boolean ignoreMissingFont = JRProperties.getBooleanProperty(JRStyledText.PROPERTY_AWT_IGNORE_MISSING_FONT);
        try
        {
            if (getJasperPrint() == null)
            {

                if (getParameters() == null)
                    setParameters(new HashMap<String, Object>());

                JRProperties.setProperty(JRStyledText.PROPERTY_AWT_IGNORE_MISSING_FONT, true);

                setJasperPrint(
                        JasperFillManager.fillReport(getJasperReport(), getParameters(), new JREmptyDataSource()));
            }
        }
        catch (JRException exception)
        {
            throw new ReportingException(ReportingException.REPORT_CANT_BE_FILLED_MESSAGE, exception);
        }
        JRProperties.setProperty(JRStyledText.PROPERTY_AWT_IGNORE_MISSING_FONT, ignoreMissingFont);
    }

    /**
     * @see pt.digitalis.utils.reporting.IReport#fillReportFromConnection(java.sql.Connection)
     */
    public void fillReportFromConnection(Connection jdbcConnection) throws ReportingException
    {
        boolean ignoreMissingFont = JRProperties.getBooleanProperty(JRStyledText.PROPERTY_AWT_IGNORE_MISSING_FONT);

        try
        {
            if (getJasperPrint() == null)
            {

                if (jdbcConnection == null)
                    throw new ReportingException(ReportingException.INVALID_CONNECTION);

                if (getParameters() == null)
                    setParameters(new HashMap<String, Object>());

                JRProperties.setProperty(JRStyledText.PROPERTY_AWT_IGNORE_MISSING_FONT, true);

                setJasperPrint(
                        ReportJasperImpl.fillReportFromConnection(getJasperReport(), jdbcConnection, getParameters()));
            }
        }
        catch (JRException exception)
        {
            throw new ReportingException(ReportingException.REPORT_CANT_BE_FILLED_MESSAGE, exception);
        }

        JRProperties.setProperty(JRStyledText.PROPERTY_AWT_IGNORE_MISSING_FONT, ignoreMissingFont);
    }

    /**
     * @see pt.digitalis.utils.reporting.IReport#fillReportFromDataSource(pt.digitalis.utils.reporting.IReportDataSource)
     */
    public void fillReportFromDataSource(IReportDataSource dataSource) throws ReportingException
    {
        boolean ignoreMissingFont = JRProperties.getBooleanProperty(JRStyledText.PROPERTY_AWT_IGNORE_MISSING_FONT);
        try
        {
            if (getJasperPrint() == null)
            {

                if (dataSource == null)
                    throw new ReportingException(ReportingException.NULL_DATA_SOURCE_MESSAGE);

                if (getParameters() == null)
                    setParameters(new HashMap<String, Object>());

                JRProperties.setProperty(JRStyledText.PROPERTY_AWT_IGNORE_MISSING_FONT, true);

                setJasperPrint(ReportJasperImpl
                        .fillReportFromDataSource(getJasperReport(), (JRDataSource) dataSource, getParameters()));
            }
        }
        catch (JRException exception)
        {
            throw new ReportingException(ReportingException.REPORT_CANT_BE_FILLED_MESSAGE, exception);
        }
        JRProperties.setProperty(JRStyledText.PROPERTY_AWT_IGNORE_MISSING_FONT, ignoreMissingFont);
    }

    /**
     * @see pt.digitalis.utils.reporting.IReport#generateReport(java.sql.Connection, java.io.OutputStream)
     */
    public void generateReport(Connection jdbcConnection, OutputStream outputStream) throws ReportingException
    {
        // Compile report if it's compiled yet
        if (getJasperReport() == null && super.getTemplatePath() != null)
            compileReport();

        // Fill report if it's not filled yet
        if (getJasperPrint() == null)
            fillReportFromConnection(jdbcConnection);

        // Export to stream
        exportToStream(outputStream);
    }

    /**
     * @see pt.digitalis.utils.reporting.IReport#generateReport(java.sql.Connection, java.lang.String)
     */
    public void generateReport(Connection jdbcConnection, String outputFileName) throws ReportingException
    {
        // Compile report if it's compiled yet
        if (getJasperReport() == null && super.getTemplatePath() != null)
            compileReport();

        // Fill report if it's not filled yet
        if (getJasperPrint() == null)
            fillReportFromConnection(jdbcConnection);

        // Export to file
        exportToFile(outputFileName);
    }

    /**
     * @see pt.digitalis.utils.reporting.IReport#generateReport(pt.digitalis.utils.reporting.IReportDataSource,
     *         java.io.OutputStream)
     */
    public void generateReport(IReportDataSource dataSource, OutputStream outputStream) throws ReportingException
    {
        // Compile report if it's compiled yet
        if (getJasperReport() == null && (super.getTemplatePath() != null || super.getTemplateInputStream() != null))
            compileReport();

        // Fill report if it's not filled yet
        if (getJasperPrint() == null)
            fillReportFromDataSource(dataSource);

        // Export to stream
        exportToStream(outputStream);
    }

    /**
     * @see pt.digitalis.utils.reporting.IReport#generateReport(pt.digitalis.utils.reporting.IReportDataSource,
     *         java.lang.String)
     */
    public void generateReport(IReportDataSource dataSource, String outputFileName) throws ReportingException
    {

        // Compile report if it's compiled yet
        if (getJasperReport() == null && (super.getTemplatePath() != null || super.getTemplateInputStream() != null))
            compileReport();

        // Fill report if it's not filled yet
        if (getJasperPrint() == null)
            fillReportFromDataSource(dataSource);

        // Export to file
        exportToFile(outputFileName);
    }

    /**
     * @see pt.digitalis.utils.reporting.IReport#generateReport(java.lang.String)
     */
    public void generateReport(String outputFileName) throws ReportingException
    {
        // Compile report if it's compiled yet
        if (getJasperReport() == null && (super.getTemplatePath() != null || super.getTemplateInputStream() != null))
            compileReport();

        // Fill report if it's not filled yet
        if (getJasperPrint() == null)
            fillReport();

        // Export to file
        exportToFile(outputFileName);
    }

    /**
     * Returns the JasperPrint object.
     *
     * @return the JasperPrint object
     */
    public JasperPrint getJasperPrint()
    {
        return this.jasperPrint;
    }

    /**
     * Sets the JasperPrint object.
     *
     * @param jasperPrint the object to set
     */
    public void setJasperPrint(JasperPrint jasperPrint)
    {
        this.jasperPrint = jasperPrint;
    }

    /**
     * Returns the JasperReport object.
     *
     * @return the JasperReport object
     */
    public JasperReport getJasperReport()
    {
        return this.jasperReport;
    }

    /**
     * Sets the JasperReport object.
     *
     * @param jasperReport the object to set
     */
    public void setJasperReport(JasperReport jasperReport)
    {
        this.jasperReport = jasperReport;
    }

    /**
     * Parses the base query applying report parameters from the current parameter list
     *
     * @return the SQL query
     */
    public String getQuerySQL()
    {
        JRQuery query = this.getJasperReport().getQuery();
        StringBuffer buffer = new StringBuffer();

        for (JRQueryChunk chunk : query.getChunks())
        {
            switch (chunk.getType())
            {
                case JRQueryChunk.TYPE_PARAMETER:
                    buffer.append(this.getParameters().get(chunk.getText()));
                    break;

                default:
                    buffer.append(chunk.getText());
            }
        }

        return buffer.toString();
    }
}
