/**
 * - Digitalis Internal Framework v2.0 - (C) 2007, Digitalis Informatica. Distribuicao e Gestao de Informatica, Lda.
 * Estrada de Paco de Arcos num.9 - Piso -1 2780-666 Paco de Arcos Telefone: (351) 21 4408990 Fax: (351) 21 4408999
 * http://www.digitalis.pt
 */
package pt.digitalis.utils.ioc;

import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.net.URL;
import java.net.URLDecoder;
import java.util.ArrayList;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.jar.JarFile;

import pt.digitalis.log.Logger;
import pt.digitalis.utils.inspection.ClasspathUtils;
import pt.digitalis.utils.inspection.exception.ResourceNotFoundException;
import pt.digitalis.utils.ioc.modules.IIoCModule;
import pt.digitalis.utils.ioc.modules.IoCBindingManager;

/**
 * This class provides common implementations for some interface methods. It's not possible do use IoC to provide an
 * abstraction of itself, so the Bridge DP was used to decouple the abstraction from it's implementation. The actual
 * registry is created with the proper implementation according to the default DiF core configuration.
 * 
 * @author Pedro Viegas <a href="mailto:pviegas@digitalis.pt">pviegas@digitalis.pt</a>
 * @author Rodrigo Gonalves <a href="mailto:rgoncalves@digitalis.pt">rgoncalves@digitalis.pt</a>
 * @created Oct 2, 2007
 */
abstract public class AbstractIoCRegistryImpl implements IIoCRegistry {

    /** Class path file loading tag. */
    final static private String CLASS_PATH_STAMP = ":CP:";

    /** The name of the individual bundle configurations file. */
    final static private String CONFIG_FILENAME = "modules.properties";

    /** The URL's "file" scheme. */
    final static private String FILE_SCHEME = "file:";

    /** 'File-within-file' indicator character. */
    final static private String INSIDE_FILE_STAMP = "!/";

    /** The IoC registry instance. */
    static protected IIoCRegistry instance;

    /** The 'modules' property name. */
    final static private String MODULES_PROPERTY = "modules";

    /** The 'modules' property separator. */
    final static private String MODULES_SEPARATOR = ",";

    /** Prefix for oneJar bundled applications. */
    final static private String ONEJAR_PREFIX = "onejar:";

    /** Resource file loading tag. */
    final static private String RESOURCES_STAMP = ":RC:";

    /**
     * Finds and returns the declared Modules.
     * 
     * @param searchMode
     *            module parsing mode
     * @throws ResourceNotFoundException
     *             if the configuration files can't be found
     */
    @SuppressWarnings("unchecked")
    public void discoverModules(ModuleParser searchMode) throws ResourceNotFoundException
    {

        long startTime = System.currentTimeMillis();
        Map<String, Class<IIoCModule>> moduleClasses = new HashMap<String, Class<IIoCModule>>();

        try
        {
            List<String> configFiles = new ArrayList<String>();

            /*
             * IMPLEMENTATION NOTE: Maven testing does not place classes/resources in the classpath, but adds some
             * classLoader that get's them from outside the classPath. In eclipse, getSystemResources does not pick up
             * resource files inside the class directories in src/java. So, it is handy to have more than one strategy
             * available for finding resources.
             */

            if (searchMode == ModuleParser.CRAWLER || searchMode == ModuleParser.PARANOID)
            {
                List<String> configResources = ClasspathUtils.getClasspathFileNamesWithExtension(CONFIG_FILENAME);

                for (String resource: configResources)
                    configFiles.add(AbstractIoCRegistryImpl.CLASS_PATH_STAMP + resource);
            }

            if (searchMode == ModuleParser.FAST || searchMode == ModuleParser.PARANOID)
            {
                // Get the modules available
                Enumeration<URL> configResources = this.getClass().getClassLoader().getResources(CONFIG_FILENAME);

                while (configResources.hasMoreElements())
                    configFiles.add(AbstractIoCRegistryImpl.RESOURCES_STAMP
                            + URLDecoder.decode(configResources.nextElement().toString(), "UTF-8"));
            }

            // Log the config files found if in debug
            if ((Logger.getLogger().isDebugEnabled() || Logger.getLogger().isTraceEnabled())
                    && (configFiles.size() > 0))
            {
                String message = "Starting Module Discovery...\n   \\\n";

                for (String file: configFiles)
                    message += "   | Config file \"" + file.substring(4) + "\" discovered\n";

                Logger.getLogger().debug(message);
            }

            for (String file: configFiles)
            {
                try
                {
                    Properties props = new Properties();

                    if (file.startsWith(AbstractIoCRegistryImpl.CLASS_PATH_STAMP))
                        props.load(System.class.getResourceAsStream(
                                "/" + file.substring(AbstractIoCRegistryImpl.CLASS_PATH_STAMP.length())));
                    else
                    {
                        /*
                         * Cleaning up the file from protocol debris
                         */
                        file = file.substring(AbstractIoCRegistryImpl.RESOURCES_STAMP.length());

                        if (file.contains(AbstractIoCRegistryImpl.FILE_SCHEME))
                        {

                            file = file.substring(file.indexOf(AbstractIoCRegistryImpl.FILE_SCHEME)
                                    + AbstractIoCRegistryImpl.FILE_SCHEME.length());
                        }
                        /*
                         * IMPLEMENTATION NOTE: In some environments the file begins with the character "/"
                         */
                        // if (file.startsWith(BaseIoCRegistryImpl.SLASH)) {
                        // file = file.substring(BaseIoCRegistryImpl.SLASH.length());
                        // }
                        if (file.contains(AbstractIoCRegistryImpl.INSIDE_FILE_STAMP))
                        {
                            /*
                             * IMPLEMENTATION NOTE: Every path that contains the string :CP:/:RC: must be read in this
                             * special way, because the file that we wish to retrieve resides inside another physical
                             * file on disk. The correct approach is to open the physical file and retrieve the property
                             * file entry.
                             */
                            JarFile jarFile = new JarFile(
                                    file.substring(0, file.indexOf(AbstractIoCRegistryImpl.INSIDE_FILE_STAMP)));
                            props.load(jarFile.getInputStream(jarFile
                                    .getEntry(file.substring(file.indexOf(AbstractIoCRegistryImpl.INSIDE_FILE_STAMP)
                                            + AbstractIoCRegistryImpl.INSIDE_FILE_STAMP.length()))));

                        }
                        else if (file.startsWith(AbstractIoCRegistryImpl.ONEJAR_PREFIX))
                        {
                            InputStream stream = new URL(file).openStream();
                            props.load(stream);
                        }
                        else
                        {
                            props.load(new FileInputStream(file));
                        }
                    }

                    /*
                     * IMPLEMENTATION NOTE: If the modulesProperty has only one value, the split operation fails. This
                     * validation turns the comma separator optional.
                     */
                    String modulesProperty = props.getProperty(AbstractIoCRegistryImpl.MODULES_PROPERTY);
                    String[] modules = null;
                    Logger.getLogger().debug(modulesProperty);
                    if (modulesProperty != null)
                    {
                        modules = modulesProperty.split(AbstractIoCRegistryImpl.MODULES_SEPARATOR);
                        if (modules.length == 0 && !"".equals(modulesProperty))
                        {
                            modules[0] = modulesProperty;
                        }
                    }

                    if (modules != null)
                    {
                        for (String moduleClassName: modules)
                        {
                            Class<?> clazz;

                            try
                            {
                                clazz = Class.forName(moduleClassName.trim());
                                moduleClasses.put(clazz.getName(), (Class<IIoCModule>) clazz);

                            }
                            catch (ClassNotFoundException e)
                            {
                                Logger.getLogger().error("Module class: \"" + moduleClassName.trim() + "\" not found!");
                            }
                        }
                    }

                }
                catch (IOException e)
                {
                    Logger.getLogger().error("Can't load configuration file: " + file);
                    e.printStackTrace();
                }
            }

        }
        catch (IOException e)
        {
            throw new ResourceNotFoundException("Can't find configuration files. No modules will be added!");
        }
        Logger.getLogger().info("Discovered " + moduleClasses.size() + " module(s) in "
                + (System.currentTimeMillis() - startTime) + "ms.");

        // Invoke configure() on IIoCModules to register modules on the manager
        for (Class<IIoCModule> moduleElement: moduleClasses.values())
        {
            try
            {
                IoCBindingManager.getInstance().addModuleBindings(moduleElement.newInstance());
            }
            catch (Exception exception)
            {
                Logger.getLogger().error("Module class: \"" + moduleElement.getName() + "\" couldn't be instantiated!");
            }
        }

        // Process dynamic
        IoCBindingManager.getInstance().processDynamicMudules();
    }

    /**
     * @see pt.digitalis.utils.ioc.IIoCRegistry#getImplementations(java.lang.Class)
     */
    public <T> List<T> getImplementations(Class<T> serviceClass)
    {
        return new ArrayList<T>(getImplementationsMap(serviceClass).values());
    }

    /**
     * @see pt.digitalis.utils.ioc.IIoCRegistry#getImplementationsClasses(java.lang.Class)
     */
    public <T> List<Class<T>> getImplementationsClasses(Class<T> serviceClass)
    {
        return new ArrayList<Class<T>>(getImplementationsClassMap(serviceClass).values());
    }
}
