/**
 *
 */
package pt.digitalis.utils.ioc;

import java.util.List;
import java.util.Map;

/**
 * Defines the expected behavior for the IoC services and contributions registry. All accesses to the registry will be
 * made through this interface. Accessing the registry through this interface reduces the impact of changing the used
 * IoC framework and user defined implementations.
 * 
 * @author Pedro Viegas <a href="mailto:pviegas@digitalis.pt">pviegas@digitalis.pt</a>
 * @author Rodrigo Gonalves <a href="mailto:rgoncalves@digitalis.pt">rgoncalves@digitalis.pt</a>
 * @created Oct 2, 2007
 */
public interface IIoCRegistry {

    /**
     * Returns the implementation of a given interface.
     * 
     * @param <T>
     *            the service's class
     * @param serviceInterface
     *            the service's interface type
     * @return the instance for the service's implementation
     */
    public <T> T getImplementation(Class<T> serviceInterface);

    /**
     * Returns the implementation of a given interface with a given identification.
     * 
     * @param <T>
     *            the service's class
     * @param serviceInterface
     *            the service's interface type
     * @param id
     *            the implementation's id
     * @return the instance for the service's implementation
     */
    public <T> T getImplementation(Class<T> serviceInterface, String id);

    /**
     * Returns the list of implementations for the given interface.
     * 
     * @param <T>
     *            the service's class
     * @param serviceInterface
     *            the service's interface type
     * @return the list of instances for a service's implementations
     */
    public <T> List<T> getImplementations(Class<T> serviceInterface);

    /**
     * Returns the list of implementations for the given interface.<br/>
     * This method is less performant that the getImplementations since in order to get the classes we need to use deep
     * object inspection. Use with care.
     * 
     * @param <T>
     *            the service's class
     * @param serviceInterface
     *            the service's interface type
     * @return the list of implementation classes for a service
     */
    public <T> List<Class<T>> getImplementationsClasses(Class<T> serviceInterface);

    /**
     * Returns the list of implementations classes for the interface with the given ID.<br/>
     * This method is less performant that the getImplementationsMap since in order to get the classes we need to use
     * deep object inspection. Use with care.
     * 
     * @param <T>
     *            the service's class
     * @param serviceInterface
     *            the service's interface type
     * @return a Map with all instance classes for each implementation of the service as V and the corresponding ID as K
     */
    public <T> Map<String, Class<T>> getImplementationsClassMap(Class<T> serviceInterface);

    /**
     * Returns the list of implementations for the interface with the given ID.
     * 
     * @param <T>
     *            the service's class
     * @param serviceInterface
     *            the service's interface type
     * @return a Map with all instances for each implementation of the service as V and the corresponding ID as K
     */
    public <T> Map<String, T> getImplementationsMap(Class<T> serviceInterface);

    /**
     * Injects dependencies into an object. Used to inject an implementation into an object not created through the IoC
     * framework. Requires the object to be annotated with com.google.inject.Inject.
     * 
     * @param obj
     *            the object to inject dependencies into
     * @see com.google.inject.Inject
     */
    public void injectDependencies(Object obj);
}
