/**
 * - Digitalis Internal Framework v2.0 - (C) 2007, Digitalis Informatica. Distribuicao e Gestao de Informatica, Lda.
 * Estrada de Paco de Arcos num.9 - Piso -1 2780-666 Paco de Arcos Telefone: (351) 21 4408990 Fax: (351) 21 4408999
 * http://www.digitalis.pt
 */
package pt.digitalis.utils.ioc.guice;

import java.lang.reflect.Field;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import pt.digitalis.utils.inspection.ReflectionUtils;
import pt.digitalis.utils.inspection.exception.AuxiliaryOperationException;
import pt.digitalis.utils.inspection.exception.ResourceNotFoundException;
import pt.digitalis.utils.ioc.AbstractIoCRegistryImpl;
import pt.digitalis.utils.ioc.IIoCRegistry;
import pt.digitalis.utils.ioc.ModuleParser;
import pt.digitalis.utils.ioc.exception.IoCException;

import com.google.inject.Binding;
import com.google.inject.Guice;
import com.google.inject.Injector;
import com.google.inject.Key;
import com.google.inject.Module;
import com.google.inject.Provider;
import com.google.inject.TypeLiteral;

/**
 * The registry for IoC services and contributions. This is the central piece that manages all contributions.
 * 
 * @author Pedro Viegas <a href="mailto:pviegas@digitalis.pt">pviegas@digitalis.pt</a>
 * @author Rodrigo Gonalves <a href="mailto:rgoncalves@digitalis.pt">rgoncalves@digitalis.pt</a>
 * @created Oct 2, 2007
 */
public class IoCRegistryGuiceImpl extends AbstractIoCRegistryImpl {

    /**
     * Getter for the instance of {@link IIoCRegistry}. Uses the FAST module parsing method by default.
     * 
     * @return the registry instance
     * @throws IoCException
     *             if the registry object can't be instantiated
     */
    static public IIoCRegistry getRegistry() throws IoCException
    {
        return getRegistry(ModuleParser.FAST);
    }

    /**
     * Getter for the instance of {@link IIoCRegistry}. Uses the FAST module parsing method by default. passed.
     * 
     * @param modules
     *            the IoC modules classes
     * @return the registry instance
     * @throws IoCException
     *             if the registry object can't be accessed
     */
    static public IIoCRegistry getRegistry(Collection<Class<? extends Module>> modules) throws IoCException
    {
        if (instance == null)
            instance = new IoCRegistryGuiceImpl(modules);

        return instance;
    }

    /**
     * Getter for the instance of {@link IIoCRegistry}
     * 
     * @param method
     *            the method for module discovery
     * @return the registry instance
     * @throws IoCException
     *             if the registry object can't be instantiated
     */
    static public IIoCRegistry getRegistry(ModuleParser method) throws IoCException
    {
        if (instance == null)
            instance = new IoCRegistryGuiceImpl(method);

        return instance;
    }

    /** Guice IoC injector. */
    private Injector injector;

    /**
     * Constructor. Private to conform with the Singleton DP.
     * 
     * @param moduleClasses
     *            the IoC modules classes
     * @throws IoCException
     *             if the injector can't be initialized
     */
    private IoCRegistryGuiceImpl(Collection<Class<? extends Module>> moduleClasses) throws IoCException
    {
        initializeInjector(moduleClasses);
    }

    /**
     * Constructor. Private to conform with the Singleton DP.
     * 
     * @param method
     *            the method for module discovery
     * @throws IoCException
     *             if any needed resource is not found
     */
    private IoCRegistryGuiceImpl(ModuleParser method) throws IoCException
    {
        try
        {
            discoverModules(method);
            initializeInjector(new ArrayList<Class<? extends Module>>() {

                /**  */
                private static final long serialVersionUID = -1738885420408319516L;

                {
                    add((new ModuleAggregator()).getClass());
                }
            });
        }
        catch (ResourceNotFoundException resourceNotFoundException)
        {
            throw new IoCException("Resource not found: ", resourceNotFoundException);
        }
    }

    /**
     * @see pt.digitalis.utils.ioc.IIoCRegistry#getImplementation(java.lang.Class)
     */
    public <T> T getImplementation(Class<T> serviceClass)
    {
        return injector.getInstance(serviceClass);
    }

    /**
     * @see pt.digitalis.utils.ioc.IIoCRegistry#getImplementation(java.lang.Class, java.lang.String)
     */
    public <T> T getImplementation(Class<T> serviceClass, String id)
    {
        return getImplementationsMap(serviceClass).get(id);
    }

    /**
     * @see pt.digitalis.utils.ioc.IIoCRegistry#getImplementationsClassMap(java.lang.Class)
     */
    public <T> Map<String, Class<T>> getImplementationsClassMap(Class<T> serviceClass)
    {

        Map<String, Class<T>> impl = new HashMap<String, Class<T>>();
        ID id;

        // Get all implementations bound to the given interface
        List<Binding<T>> bindings = injector.findBindingsByType(TypeLiteral.get(serviceClass));

        for (Binding<T> binding: bindings)
        {
            id = (ID) binding.getKey().getAnnotation();

            // Prevent NPE
            if (id != null)
                impl.put(id.value(), this.getImplementationTypeForBinding(binding));
            else
                impl.put(null, this.getImplementationTypeForBinding(binding));
        }
        return impl;
    }

    /**
     * @see pt.digitalis.utils.ioc.IIoCRegistry#getImplementationsMap(java.lang.Class)
     */
    public <T> Map<String, T> getImplementationsMap(Class<T> serviceClass)
    {

        Map<String, T> impl = new HashMap<String, T>();
        ID id;

        // Get all implementations bound to the given interface
        List<Binding<T>> bindings = injector.findBindingsByType(TypeLiteral.get(serviceClass));

        for (Binding<T> binding: bindings)
        {
            id = (ID) binding.getKey().getAnnotation();

            // Prevent NPE
            if (id != null)
                impl.put(id.value(), binding.getProvider().get());
            else
                impl.put(null, binding.getProvider().get());
        }
        return impl;
    }

    /**
     * @param <T>
     * @param binding
     *            the binding to evaluate
     * @return the provider class type
     */
    @SuppressWarnings("unchecked")
    protected <T> Class<T> getImplementationTypeForBinding(Binding<T> binding)
    {
        Class<T> clazz = null;
        Field field;

        try
        {
            // Cannot get the provider class type from the Guice API.
            // So we use Inspection to parse thought the Guice internal fields to get to it!
            // CAUTION: This may (and must certainly will) change in future versions of Guice and call for refactor!)
            Provider<T> provider = binding.getProvider();

            field = provider.getClass().getDeclaredField("val$factory");
            if (!field.isAccessible())
                field.setAccessible(true);

            Object valFactory = field.get(provider);

            field = valFactory.getClass().getDeclaredField("targetKey");
            if (!field.isAccessible())
                field.setAccessible(true);

            Key<T> key = (Key<T>) field.get(valFactory);

            clazz = (Class<T>) key.getTypeLiteral().getType();
        }
        catch (Exception e)
        {
            e.printStackTrace();
        }

        return clazz;
    }

    /**
     * Creates the Guice injector based on the given IoC Modules classes.
     * 
     * @param moduleClasses
     *            the IoC modules classes
     * @throws IoCException
     *             if a module class object can't be instantiated
     */
    private void initializeInjector(Collection<Class<? extends Module>> moduleClasses) throws IoCException
    {

        // Initialize the Module array
        Module[] modules = new Module[moduleClasses.size()];
        // Module array index
        int moduleIndex = 0;

        // For each Module class on the collection...
        for (Class<? extends Module> clazz: moduleClasses)
            try
            {
                // Instantiate object and set it in the module array
                modules[moduleIndex++] = ReflectionUtils.instantiateObjectFromClass(clazz);
            }
            catch (AuxiliaryOperationException auxiliaryOperationException)
            {
                throw new IoCException("Could not instantiate object from class " + clazz.getCanonicalName() + "!",
                        auxiliaryOperationException);
            }

        // Get Guice injector for the passed modules
        injector = Guice.createInjector(modules);
    }

    /**
     * @see pt.digitalis.utils.ioc.IIoCRegistry#injectDependencies(java.lang.Object)
     */
    public void injectDependencies(Object obj)
    {
        injector.injectMembers(obj);
    }
}
