/**
 * 2007, Digitalis Informatica. All rights reserved. Distribuicao e Gestao de Informatica, Lda. Estrada de Paco de Arcos
 * num.9 - Piso -1 2780-666 Paco de Arcos Telefone: (351) 21 4408990 Fax: (351) 21 4408999 http://www.digitalis.pt
 */
package pt.digitalis.utils.ioc.modules;

/**
 * Defines a IoC binding. Uses the 'Fluent Interface' idiom for subsequent method invocations context relaying. The
 * setter methods update the binding on the binder.
 * 
 * @author Pedro Viegas <a href="mailto:pviegas@digitalis.pt">pviegas@digitalis.pt</a>
 * @author Rodrigo Gonalves <a href="mailto:rgoncalves@digitalis.pt">rgoncalves@digitalis.pt</a><br/>
 * @created May 2, 2008
 */
public class IoCBinding implements IIoCBinding {

    /** Id for multi-implementation contributions. */
    private String id = null;

    /** Type of the class that implements the contribution. */
    private Class<?> implementationType;

    /** Type of the interface that implements the contribution. */
    private Class<?> interfaceType;

    /** Flag for non-overriddable binding. */
    private boolean isFinal = false;

    /** Flag for singleton binding. */
    private boolean isSingleton = false;

    /** Flag for override binding. */
    private boolean override = false;

    /** The IoC binder associated to this binding. */
    private IoCBinder theBinder;

    /**
     * Constructs a new IoCBinding object.
     * 
     * @param interfaceType
     *            the interface type
     * @param implementationType
     *            implementation type
     * @param isFinal
     *            'final' flag
     * @param override
     *            'override' flag
     * @param isSingleton
     *            'singleton' flag
     * @param id
     *            the binding id
     * @param binder
     *            the IoC binder associated to this binding
     */
    public IoCBinding(Class<?> interfaceType, Class<?> implementationType, boolean isFinal, boolean override,
            boolean isSingleton, String id, IoCBinder binder)
    {
        this(interfaceType, implementationType, binder);
        this.isFinal = isFinal;
        this.isSingleton = isSingleton;
        this.override = override;
        this.id = id;
    }

    /**
     * Constructs a new IoCBinding object. The binding is defined as non-final and non-singleton.
     * 
     * @param interfaceType
     *            the interface type
     * @param implementationType
     *            the implementation type
     * @param binder
     *            the IoC binder associated to this binding
     */
    public IoCBinding(Class<?> interfaceType, Class<?> implementationType, IoCBinder binder)
    {
        this.interfaceType = interfaceType;
        this.implementationType = implementationType;
        this.theBinder = binder;
    }

    /**
     * @see pt.digitalis.utils.ioc.modules.IIoCBinding#asFinal()
     */
    public IIoCBinding asFinal()
    {
        this.isFinal = true;
        this.override = true;

        updateBinding();

        return this;
    }

    /**
     * @see pt.digitalis.utils.ioc.modules.IIoCBinding#asSingleton()
     */
    public IIoCBinding asSingleton()
    {
        this.isSingleton = true;

        updateBinding();

        return this;
    }

    /** Deletes the reference to the binder. */
    protected void deleteBinderReference()
    {
        this.theBinder = null;
    }

    /**
     * Inspector for the 'id' property.
     * 
     * @return the id
     */
    public String getId()
    {
        return this.id;
    }

    /**
     * Inspector for the implementation type property.
     * 
     * @return the implementation type
     */
    public Class<?> getImplementationType()
    {
        return this.implementationType;
    }

    /**
     * Inspector for the interface type property.
     * 
     * @return the interface type
     */
    public Class<?> getInterfaceType()
    {
        return this.interfaceType;
    }

    /**
     * Inspector for the 'final' property.
     * 
     * @return the isFinal
     */
    public boolean isFinal()
    {
        return this.isFinal;
    }

    /**
     * @return T if the current module is a multiple binding (it it has ID defined)
     */
    public boolean isMultiple()
    {
        return id != null;
    }

    /**
     * Inspector for the 'override' attribute.
     * 
     * @return the override value
     */
    public boolean isOverride()
    {
        return this.override;
    }

    /**
     * Inspector for the 'singleton' property.
     * 
     * @return the isSingleton
     */
    public boolean isSingleton()
    {
        return this.isSingleton;
    }

    /**
     * @see pt.digitalis.utils.ioc.modules.IIoCBinding#override()
     */
    public IIoCBinding override()
    {
        this.override = true;

        updateBinding();

        return this;
    }

    /**
     * @see java.lang.Object#toString()
     */
    @Override
    public String toString()
    {
        StringBuffer binding = new StringBuffer();
        binding.append("Implementation " + implementationType.getCanonicalName() + "\n    is bound to interface "
                + interfaceType.getCanonicalName());

        if ((getId() != null) || isFinal || isSingleton)
            binding.append("\n   ");

        if (getId() != null)
            binding.append(" with id: \"" + getId() + "\"");

        if (isFinal())
            binding.append(" as final");

        if (isSingleton())
            binding.append(" as singleton");

        binding.append(".\n");

        return binding.toString();
    }

    /** Updates this binding's reference on the binder. */
    private void updateBinding()
    {
        this.theBinder.updateBinding(this);
    }

    /**
     * @see pt.digitalis.utils.ioc.modules.IIoCBinding#withId(java.lang.String)
     */
    public IIoCBinding withId(String id)
    {
        this.id = id;

        updateBinding();

        return this;
    }
}
