/**
 * 2008, Digitalis Informatica. All rights reserved. Distribuicao e Gestao de Informatica, Lda. Estrada de Paco de Arcos
 * num.9 - Piso -1 2780-666 Paco de Arcos Telefone: (351) 21 4408990 Fax: (351) 21 4408999 http://www.digitalis.pt
 */
package pt.digitalis.utils.ioc.modules;

import pt.digitalis.log.Logger;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * Manages the IoC bindings.
 *
 * @author Pedro Viegas <a href="mailto:pviegas@digitalis.pt">pviegas@digitalis.pt</a>
 * @author Rodrigo Gonalves <a href="mailto:rgoncalves@digitalis.pt">rgoncalves@digitalis.pt</a>
 * @created May 2, 2008
 */
public class IoCBindingManager
{

    /** The binding sign as constant. */
    final static private String BINDING_SIGN = " <-> ";

    /** The singleton instance. */
    static private IoCBindingManager instance = null;

    /** Binding map <K = interface type , V = list binding object>. */
    private Map<Class<?>, List<IoCBinding>> bindings = new HashMap<Class<?>, List<IoCBinding>>();

    /** dynamic modules for startup load on end. After initialization will be set to null. */
    private List<IIoCModule> dynamicModules = new ArrayList<IIoCModule>();

    /** Default ctor. Private to enforce the Singleton DP. */
    private IoCBindingManager()
    {
    }

    /**
     * Singleton instance accessor.
     *
     * @return the instance
     */
    static public IoCBindingManager getInstance()
    {

        if (instance == null)
            instance = new IoCBindingManager();

        return instance;
    }

    /**
     * Adds a new binding to the manager. responsible for all logic behind the binding process.
     *
     * @param binding the binding to add
     */
    public void addBinding(IoCBinding binding)
    {

        BindingValitationResultAction validBind = BindingValitationResultAction.INVALID_SKIP;
        List<IoCBinding> existingBindings = new ArrayList<IoCBinding>();

        // If binding does not exist yet, add it (no matter if it is final/override or not)
        if (!bindings.containsKey(binding.getInterfaceType()))
        {
            existingBindings = new ArrayList<IoCBinding>();
            existingBindings.add(binding);
            validBind = BindingValitationResultAction.VALID_ADD;
        }
        else
        {
            // If the interface exists...
            // Get the existing bindings (for easier access)
            existingBindings = bindings.get(binding.getInterfaceType());

            if ((existingBindings == null) || existingBindings.isEmpty())
            {
                // The interface exists but no binds are associated. Add at will, it is as if it does not exist...
                existingBindings = new ArrayList<IoCBinding>();
                existingBindings.add(binding);
                validBind = BindingValitationResultAction.VALID_ADD;
            }
            else
            {
                // Bindings exist...
                if (binding.isMultiple())
                {

                    // Multiple bind (has ID), check for the correct
                    IoCBinding previousBindForID = null;

                    for (IoCBinding current : existingBindings)
                    {
                        // We decided to allow adding of single and multiple bindings to the same interface, for
                        // flexibility (like default contribution from the multiple existing ones)

                        // if (!current.isMultiple())
                        // {
                        // // Found a single contribution. Select this to force the validation failing and reporting
                        // previousBindForID = current;
                        // break;
                        // }

                        if (binding.getId().equals(current.getId()))
                        {
                            // Found the correct ID contribution. Select it
                            previousBindForID = current;
                            break;
                        }
                    }

                    validBind = validateBindingRules(previousBindForID, binding);

                    if (validBind.isValid())
                    {
                        if (previousBindForID != null && validBind == BindingValitationResultAction.VALID_REPLACE)
                            existingBindings.remove(previousBindForID);

                        existingBindings.add(binding);
                    }
                }
                else
                {
                    // Not multiple, validate with the first item (it should be the only one. If more than one they will
                    // be multiple and this will fail as it should)
                    validBind = validateBindingRules(existingBindings.get(0), binding);

                    if (validBind == BindingValitationResultAction.VALID_REPLACE)
                        existingBindings.clear();

                    if (validBind.isValid())
                        existingBindings.add(binding);
                }
            }
        }

        if (validBind.isValid())
            // If the bind is valid update the updated binding list
            bindings.put(binding.getInterfaceType(), existingBindings);
    }

    /**
     * Adds all bindings of the given module.
     *
     * @param module the IoC module
     */
    public void addModuleBindings(IIoCModule module)
    {
        if (module instanceof IIoCDynamicModule)
            dynamicModules.add(module);

        else
        {
            addModuleBindingsToManager(module);
        }
    }

    /**
     * Adds all bindings of the given module.
     *
     * @param module the IoC module
     */
    private void addModuleBindingsToManager(IIoCModule module)
    {
        IoCBinder binder = new IoCBinder();
        module.configure(binder);
        binder.addBindingsToManager();
    }

    /**
     * Returns the first available implementation for the given interface type.
     *
     * @param interfaceType the interface type for which the implementation is requested
     *
     * @return the first available implementation for the given interface type
     */
    public IoCBinding getBinding(Class<?> interfaceType)
    {
        return this.bindings.get(interfaceType).get(0);
    }

    /**
     * Returns the implementation with a given id for the given interface type.
     *
     * @param interfaceType the interface type for which the implementation is requested.
     * @param id            the implementation id
     *
     * @return the implementations with the given id
     */
    public IoCBinding getBinding(Class<?> interfaceType, String id)
    {

        for (IoCBinding binding : bindings.get(interfaceType))
        {
            if (id.equals(binding.getId()))
                return binding;
        }

        return null;
    }

    /**
     * Returns all the available bindings.
     *
     * @return all the available bindings
     */
    public Map<Class<?>, List<IoCBinding>> getBindings()
    {
        return this.bindings;
    }

    /**
     * Returns all the available bindings for a given interface.
     *
     * @param interfaceType the interface type for which the implementations are requested.
     *
     * @return the list of available implementations for the given interface type
     */
    public List<IoCBinding> getBindings(Class<?> interfaceType)
    {
        return this.bindings.get(interfaceType);
    }

    /**
     * After all modules are initialized the dynamic modules should be processed
     */
    public void processDynamicMudules()
    {
        if (dynamicModules != null)
        {
            for (IIoCModule module : dynamicModules)
            {
                addModuleBindingsToManager(module);
            }

            dynamicModules = null;
        }
    }

    /**
     * Clears all existing bindings. Use for test purposes only!
     */
    protected void resetBindings()
    {
        this.bindings.clear();
    }

    /**
     * @see java.lang.Object#toString()
     */
    @Override
    public String toString()
    {
        StringBuffer buffer = new StringBuffer();

        buffer.append("Bindings:\n--------------------");

        for (Class<?> interfaceClass : this.bindings.keySet())
        {
            for (IoCBinding bind : bindings.get(interfaceClass))
            {
                buffer.append("\n    - " + bind.toString());
            }
        }

        return buffer.toString();
    }

    /**
     * Validates final, overriddable, single and multiple binding rules for all Module bindings. Helper method for the
     * add Binding process/method. This method is also responsible for reporting warnings for bad usage to the logger.
     *
     * @param previousBind the existing binding to compare
     * @param newBind      the new bind to validate
     *
     * @return the T if the bind passes the validation process
     */
    private BindingValitationResultAction validateBindingRules(IoCBinding previousBind, IoCBinding newBind)
    {

        if (previousBind == null)
            // No previous bind. Nothing to validate, return true
            return BindingValitationResultAction.VALID_ADD;

        else
        {
            // If existing binding IS a multiple binding and the new is NOT...
            if (previousBind.isMultiple() && !newBind.isMultiple())
            {
                Logger.getLogger().warn("Binding " + newBind.getInterfaceType().getCanonicalName() + BINDING_SIGN +
                                        newBind.getImplementationType().getCanonicalName() +
                                        " appears to be a SINGLE binding (no id!) and is being used on a previous multiple bindings context!");
                return BindingValitationResultAction.VALID_ADD;// false;
            }

            // If existing binding is a NOT multiple binding and the new IS...
            else if (!previousBind.isMultiple() && newBind.isMultiple())
            {
                Logger.getLogger().warn("Binding " + newBind.getInterfaceType().getCanonicalName() + BINDING_SIGN +
                                        newBind.getImplementationType().getCanonicalName() +
                                        " appears to be a MULTIPLE binding and is being used on a single binding context!");
                return BindingValitationResultAction.VALID_ADD;// false;
            }
            else
            {
                // Both are Multiple or single bindings...

                // If existing binding is final, it's not overriddable. Issue warning.
                if (previousBind.isFinal())
                {
                    Logger.getLogger()
                            .warn("Binding " + previousBind.getInterfaceType().getCanonicalName() + BINDING_SIGN +
                                  previousBind.getImplementationType().getCanonicalName() +
                                  " was declared FINAL and thus can't be overridden with the new implementation " +
                                  newBind.getImplementationType().getCanonicalName() + "!");

                    return BindingValitationResultAction.INVALID_SKIP;
                }
                else if (!newBind.isOverride())
                {
                    // If the new is not marked as override
                    Logger.getLogger()
                            .warn("Binding " + previousBind.getInterfaceType().getCanonicalName() + BINDING_SIGN +
                                  previousBind.getImplementationType().getCanonicalName() +
                                  " allready exists and the new implementation " +
                                  newBind.getImplementationType().getCanonicalName() + " is not declared as Override!");

                    return BindingValitationResultAction.INVALID_SKIP;
                }
                else
                {
                    // If old is not final (it's overriddable)
                    // If new is declared as override
                    // All OK, validate the new bind
                    return BindingValitationResultAction.VALID_REPLACE;
                }
            }
        }
    }

    /**
     * Use to inform the BindingManager process of the action to perform for the validated binding
     *
     * @author Pedro Viegas <a href="mailto:pviegas@digitalis.pt">pviegas@digitalis.pt</a><br/>
     * @created 8 de Fev de 2011
     */
    protected enum BindingValitationResultAction
    {
        /** Invalid bind. Skip it! */
        INVALID_SKIP,
        /** Valid bind. Add it to the previous binding list */
        VALID_ADD,
        /** Valid bind. Replace previous binds with this one. */
        VALID_REPLACE;

        /**
         * @return T if the result if VALID_ADD or VALID_REPLACE
         */
        public boolean isValid()
        {
            return (this == VALID_ADD || this == VALID_REPLACE);
        }
    }
}
