/**
 * 2007, Digitalis Informatica. All rights reserved. Distribuicao e Gestao de Informatica, Lda. Estrada de Paco de Arcos
 * num.9 - Piso -1 2780-666 Paco de Arcos Telefone: (351) 21 4408990 Fax: (351) 21 4408999 http://www.digitalis.pt
 */

package pt.digitalis.utils.ldap;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;

import pt.digitalis.utils.common.collections.CaseInsensitiveHashMap;

/**
 * Wrapper type for an LDAP user. <br>
 * TODO: Viegas: Must try to create a common interface that this implementation and the IIdentityManager from DIF use.
 * This would prevent the need for the DIF IdentityManagerImpl to convert a list of LDAPUser objects to a list of
 * IDIFUser objects.
 * 
 * @author Rodrigo Gonalves <a href="mailto:rgoncalves@digitalis.pt">rgoncalves@digitalis.pt</a><br/>
 * @author Luis Pinto <a href="lpinto@digitalis.pt">lpinto@digitalis.pt</a><br/>
 * @created 2008/03/31
 */
public class LDAPUser extends LDAPEntity {

    /** The 'display name' attribute. */
    final static public String DISPLAY_NAME = "displayName";

    /** The 'given name' attribute. */
    final static public String GIVEN_NAME = "givenName";

    /** The user's display name. */
    private String displayName;

    /** The user's e-mail. */
    private String eMail;

    /** The user's given name. */
    private String givenName;

    /** The user's login name. */
    private String loginName;

    /**
     * The parameters map. <K = LDAP attribute to store the parameter, V = the parameter value>. Assumes that an
     * external entity deals with the mapping from attribute name to business parameter name.
     */
    private final CaseInsensitiveHashMap<String> parameters = new CaseInsensitiveHashMap<String>();

    /**
     * The bulk parameters map to remove.
     */
    private final List<String> parametersToRemove = new ArrayList<String>();

    /**
     * The user's password. This attribute is used only to create users, since password is never returned by LDAP
     * servers.
     */
    private String password;

    /** The user's "user name". */
    private String userName;

    /**
     * Inspector for the display name.
     * 
     * @return the displayName
     */
    public String getDisplayName()
    {
        return this.displayName;
    }

    /**
     * Inspector for the e-mail.
     * 
     * @return the user's e-mail.
     */
    public String getEmail()
    {
        return this.eMail;
    }

    /**
     * Inspector for the 'enabled' property.
     * 
     * @return T if the user is enabled, F otherwise
     */
    public boolean getEnabled()
    {
        // TODO: LDAP: Investigate this functionality in LDAP.
        return false;
    }

    /**
     * Inspector for the given name.
     * 
     * @return the givenName
     */
    public String getGivenName()
    {
        return this.givenName;
    }

    /**
     * Inspector for the login name.
     * 
     * @return the name
     */
    public String getLoginName()
    {
        return this.loginName;
    }

    /**
     * Returns a given parameter value, identified by the name of the attribute on which the value is stored.
     * 
     * @param attributeName
     *            the name of the attribute on which the parameter value is stored
     * @return the parameter value or null if the map doesn't contain the supplied key
     */
    public String getParameter(String attributeName)
    {
        return this.parameters.get(attributeName);
    }

    /**
     * Returns the parameters map.
     * 
     * @return the parameters map
     */
    public Map<String, String> getParameters()
    {
        return this.parameters;
    }

    /**
     * Inspector for the 'bulkParametersToRemove' attribute.
     * 
     * @return the bulkParametersToRemove value
     */
    public List<String> getParametersToRemove()
    {
        return parametersToRemove;
    }

    /**
     * Inspector for the password.
     * 
     * @return the password
     */
    public String getPassword()
    {
        return this.password;
    }

    /**
     * Inspector for the user name.
     * 
     * @return the user name
     */
    public String getUserName()
    {
        return this.userName;
    }

    /**
     * Removes a parameter from the parameters map.
     * 
     * @param attributeName
     *            the name of the attribute on which the parameter value is stored
     */
    public void removeParameter(String attributeName)
    {
        this.parameters.remove(attributeName);
        this.parametersToRemove.add(attributeName);
    }

    /**
     * Modifier for the display name.
     * 
     * @param displayName
     *            the displayName to set
     */
    public void setDisplayName(String displayName)
    {
        this.displayName = displayName;
    }

    /**
     * Modifier for the e-mail.
     * 
     * @param eMail
     *            the new e-mail to set
     */
    public void setEmail(String eMail)
    {
        this.eMail = eMail;
    }

    /**
     * Modifier for the 'enabled' property.
     * 
     * @param enabled
     *            the new 'enabled' value
     */
    public void setEnabled(boolean enabled)
    {
        // TODO: LDAP: Investigate this functionality in LDAP.
    }

    /**
     * Modifier for the given name.
     * 
     * @param givenName
     *            the givenName to set
     */
    public void setGivenName(String givenName)
    {
        this.givenName = givenName;
    }

    /**
     * Modifier for the login name.
     * 
     * @param loginName
     *            the name to set
     */
    public void setLoginName(String loginName)
    {
        this.loginName = loginName;
    }

    /**
     * Sets a parameter on the parameters map.
     * 
     * @param attributeName
     *            the name of the LDAP attribute that will store the parameter
     * @param value
     *            the parameter value
     */
    public void setParameter(String attributeName, String value)
    {
        this.parameters.put(attributeName, value);
    }

    /**
     * Modifier for the password.
     * 
     * @param password
     *            the password to set
     */
    public void setPassword(String password)
    {
        this.password = password;
    }

    /**
     * Modifier for the user name.
     * 
     * @param userName
     *            the user name to set
     */
    public void setUserName(String userName)
    {
        this.userName = userName;
    }

    /**
     * @see java.lang.Object#toString()
     */
    @Override
    public String toString()
    {

        // String constants
        final String N_A = "  N/A";
        final String NEW_LINE = "\n";
        final String EQUALS = "=";
        final String DISPLAY_NAME = "displayName=";
        final String GIVEN_NAME = "givenName=";
        final String LOGIN_NAME = "loginName=";
        final String NAME = "name=";
        final String USER_NAME = "userName=";
        final String PARENT_GROUP_DN = "parentGroupDN=";

        StringBuffer ldapUser = new StringBuffer();

        if (this.displayName == null)
            ldapUser.append(DISPLAY_NAME + N_A + NEW_LINE);
        else
            ldapUser.append(DISPLAY_NAME + this.displayName + NEW_LINE);

        if (this.givenName == null)
            ldapUser.append(GIVEN_NAME + N_A + NEW_LINE);
        else
            ldapUser.append(GIVEN_NAME + this.givenName + NEW_LINE);

        if (this.loginName == null)
            ldapUser.append(LOGIN_NAME + N_A + NEW_LINE);
        else
            ldapUser.append(LOGIN_NAME + this.loginName + NEW_LINE);

        if (this.getName() == null)
            ldapUser.append(NAME + N_A + NEW_LINE);
        else
            ldapUser.append(NAME + this.getName() + NEW_LINE);

        if (this.userName == null)
            ldapUser.append(USER_NAME + N_A + NEW_LINE);
        else
            ldapUser.append(USER_NAME + this.userName + NEW_LINE);

        if (this.getParentGroupDN() == null)
            ldapUser.append(PARENT_GROUP_DN + N_A + NEW_LINE);
        else
            ldapUser.append(PARENT_GROUP_DN + this.getParentGroupDN() + NEW_LINE);

        if (this.parameters.size() > 0)
        {
            for (String attributeName: this.parameters.keySet())
            {
                ldapUser.append(attributeName + EQUALS + this.parameters.get(attributeName) + NEW_LINE);
            }
        }

        return super.toString() + ldapUser.toString();
    }
}
