/**
 * 2007, Digitalis Informatica. All rights reserved. Distribuicao e Gestao de Informatica, Lda. Estrada de Paco de Arcos
 * num.9 - Piso -1 2780-666 Paco de Arcos Telefone: (351) 21 4408990 Fax: (351) 21 4408999 http://www.digitalis.pt
 */
package pt.digitalis.utils.ldap.impl;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Hashtable;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;
import java.util.Vector;

import javax.naming.AuthenticationException;
import javax.naming.Context;
import javax.naming.NameNotFoundException;
import javax.naming.NamingEnumeration;
import javax.naming.NamingException;
import javax.naming.directory.Attribute;
import javax.naming.directory.Attributes;
import javax.naming.directory.BasicAttribute;
import javax.naming.directory.BasicAttributes;
import javax.naming.directory.ModificationItem;
import javax.naming.directory.SchemaViolationException;
import javax.naming.directory.SearchControls;
import javax.naming.directory.SearchResult;
import javax.naming.ldap.Control;
import javax.naming.ldap.InitialLdapContext;
import javax.naming.ldap.LdapContext;
import javax.naming.ldap.PagedResultsControl;
import javax.naming.ldap.PagedResultsResponseControl;

import pt.digitalis.log.ILogWrapper;
import pt.digitalis.log.LogLevel;
import pt.digitalis.log.Logger;
import pt.digitalis.utils.common.StringUtils;
import pt.digitalis.utils.common.collections.CaseInsensitiveHashMap;
import pt.digitalis.utils.config.ConfigurationsPreferencesImpl;
import pt.digitalis.utils.ldap.ILDAPUtils;
import pt.digitalis.utils.ldap.LDAPConfigurations;
import pt.digitalis.utils.ldap.LDAPEntity;
import pt.digitalis.utils.ldap.LDAPGroup;
import pt.digitalis.utils.ldap.LDAPUser;
import pt.digitalis.utils.ldap.exception.LDAPOperationException;
import pt.digitalis.utils.ldap.exception.LDAPOperationReadOnlyException;

/**
 * Defines a set operations common to all LDAP implementations.
 * 
 * @author Rodrigo Gon�alves <a href="mailto:rgoncalves@digitalis.pt">rgoncalves@digitalis.pt</a><br/>
 * @author Luis Pinto <a href="lpinto@digitalis.pt">lpinto@digitalis.pt</a><br/>
 * @author F�bio Souto <a href="mailto:fsouto@digitalis.pt">fsouto@digitalis.pt</a><br/>
 * @created Mar 26, 2008
 */
abstract public class AbstractLDAPUtils implements ILDAPUtils {

    /** The "CN=" constant string. */
    final static protected String CN_TAG = "cn=";

    /** the groups mapping already found. For logging purposes */
    static private List<String> groupMappingsFound = new ArrayList<String>();

    /** the API logger. */
    private static ILogWrapper logger = null;

    /** The 'N/A' constant string. */
    final static protected String NON_AVAILABLE = "N/A";

    /**
     * Extracts the common name from the distinguished name.
     * 
     * @param distinguishedName
     *            the distinguished name to process
     * @return the common name extracted from the distinguished name
     */
    final static protected String getCNFromDN(String distinguishedName)
    {
        return (distinguishedName.split(",")[0]).split("=")[1];
    }

    /**
     * Inspector for the 'logger' attribute.
     * 
     * @return the logger value
     */
    protected static ILogWrapper getLogger()
    {

        if (AbstractLDAPUtils.logger == null)
        {
            AbstractLDAPUtils.logger = Logger.getLogger("LDAP Utils ", LogLevel.INFO);
        }
        return AbstractLDAPUtils.logger;
    }

    /**
     * Parses the controls.
     * 
     * @param controls
     *            the controls
     * @return the byte[]
     * @throws NamingException
     *             the naming exception
     */
    public static byte[] parseControls(Control[] controls) throws NamingException
    {

        byte[] cookie = null;

        if (controls != null)
        {

            for (int i = 0; i < controls.length; i++)
            {
                if (controls[i] instanceof PagedResultsResponseControl)
                {
                    PagedResultsResponseControl prrc = (PagedResultsResponseControl) controls[i];
                    cookie = prrc.getCookie();
                }
            }
        }

        return (cookie == null) ? new byte[0] : cookie;
    }

    /** The LDAP configurations. */
    protected LDAPConfigurations ldapConfigurations = null;

    /**
     * Adds an attribute to a given LDAP entity.
     * 
     * @param distinguishedName
     *            the entity's distinguished name
     * @param attributeName
     *            the attribute name
     * @param value
     *            the attribute value
     * @throws LDAPOperationException
     *             if the attribute value can't be changed
     */
    final private void addAttribute(String distinguishedName, String attributeName, String value)
            throws LDAPOperationException
    {
        if (this.isReadOnly())
            throw new LDAPOperationReadOnlyException();

        /* May be a mapped attribute */
        String attributeRealName = getConfigurations().getAttributesMapping().get(attributeName);
        if (attributeRealName == null)
        {
            attributeRealName = attributeName;
        }

        ModificationItem[] mods = new ModificationItem[1];

        mods[0] = new ModificationItem(LdapContext.ADD_ATTRIBUTE, new BasicAttribute(attributeRealName,
                value.toString()));

        modifyAttributes(distinguishedName, mods, false);
    }

    /**
     * @see pt.digitalis.utils.ldap.ILDAPUtils#addGroup(pt.digitalis.utils.ldap.LDAPGroup)
     */
    public void addGroup(LDAPGroup newGroup) throws LDAPOperationException
    {
        if (this.isReadOnly())
            throw new LDAPOperationReadOnlyException();

        String groupDN = null;
        String newEntityDN = null;

        String groupMapping = this.getConfigurations().getGroupMappings().get(newGroup.getCommonName());

        if (groupMapping != null)
        {
            newGroup.setCommonName(groupMapping);
        }

        if (newGroup.getParentGroupDN() != null)
        {
            groupDN = newGroup.getParentGroupDN();
            newEntityDN = calculateDistinguishedName(newGroup.getCommonName(), groupDN);
        }
        else
        {
            if (getConfigurations().getDefaultGroupDN() != null)
            {
                groupDN = getConfigurations().getDefaultGroupDN();
                newEntityDN = calculateDistinguishedName(newGroup.getCommonName(), groupDN);
            }
            else
                // groupDN = this.ldapConfigurations.getBaseSearchDN();
                newEntityDN = AbstractLDAPUtils.CN_TAG + newGroup.getCommonName() + ","
                        + getConfigurations().getBaseSearchDN();
        }

        newGroup.setDistinguishedName(newEntityDN);
        createSubcontext(newGroup.getDistinguishedName(), getAttributesForGroupAddition(newGroup));
    }

    /**
     * @see pt.digitalis.utils.ldap.ILDAPUtils#addGroupAttribute(java.lang.String, java.lang.String, java.lang.Object)
     */
    public void addGroupAttribute(String commonName, String attributeName, Object value) throws LDAPOperationException
    {
        if (this.isReadOnly())
            throw new LDAPOperationReadOnlyException();
        addAttribute(findGroupByCommonName(commonName).getDistinguishedName(), attributeName, value.toString());
    }

    /**
     * @see pt.digitalis.utils.ldap.ILDAPUtils#addUser(pt.digitalis.utils.ldap.LDAPUser)
     */
    public void addUser(LDAPUser newUser) throws LDAPOperationException
    {
        if (this.isReadOnly())
            throw new LDAPOperationReadOnlyException();

        String exclusionCharaters = this.getConfigurations().getExclusionCharaters();

        if (exclusionCharaters != null && !"".equals(exclusionCharaters))
        {
            for (int i = 0; i < exclusionCharaters.length(); i++)
                if (newUser.getLoginName() != null && newUser.getLoginName().indexOf(exclusionCharaters.charAt(i)) > -1)
                    throw new LDAPOperationException("Your login name cannot contain the characters "
                            + exclusionCharaters);
        }
        String userParentGroupDN = newUser.getParentGroupDN();

        if (userParentGroupDN == null)
        {
            if (getConfigurations().getDefaultProfileDN() != null)
                userParentGroupDN = getConfigurations().getDefaultProfileDN();
            else if (getConfigurations().getBaseSearchDN() != null)
                userParentGroupDN = getConfigurations().getBaseSearchDN();

            /* Parent group shouldn't create the user group association. */
            newUser.setParentGroupDN(userParentGroupDN);
        }

        newUser.setDistinguishedName(calculateDistinguishedName(newUser.getLoginName(), userParentGroupDN));

        createSubcontext(newUser.getDistinguishedName(), getAttributesForUserAddition(newUser));
        changePassword(newUser.getLoginName(), newUser.getPassword());
    }

    /**
     * @see pt.digitalis.utils.ldap.ILDAPUtils#addUserAttribute(java.lang.String, java.lang.String, java.lang.Object)
     */
    public void addUserAttribute(String loginName, String attributeName, Object value) throws LDAPOperationException
    {
        if (this.isReadOnly())
            throw new LDAPOperationReadOnlyException();

        addAttribute(findUserByLogin(loginName).getDistinguishedName(), attributeName, value.toString());
    }

    /**
     * @see pt.digitalis.utils.ldap.ILDAPUtils#addUserToGroup(java.lang.String, java.lang.String)
     */
    public void addUserToGroup(String groupCN, String userLogin) throws LDAPOperationException
    {
        if (this.isReadOnly())
            throw new LDAPOperationReadOnlyException();

        this.setGroupAttribute(groupCN, LDAPGroup.MEMBER_NAME, findUserByLogin(userLogin).getDistinguishedName());
    }

    /**
     * Builds the attribute string for LDAP searching.
     * 
     * @param attributes
     *            The attributes to search for
     * @return A string containing the LDAP query
     * @throws LDAPOperationException
     *             the lDAP operation exception
     */
    private String buildAttributesString(Map<String, String> attributes) throws LDAPOperationException
    {
        StringBuffer query = new StringBuffer();
        for (String attributeKey: attributes.keySet())
        {
            boolean isBulkParam = false;
            String attributeValue = "";
            String attributeBulkKey = "";
            // if we are looking for users which belong to a group, we cannot search for the group name only,
            // we must get the group's full DN
            if (this.getUserParentGroupAttributeName().equals(attributeKey))
                attributeValue = findGroupByCommonName(attributes.get(attributeKey)).getDistinguishedName();
            // we're searching for names or something else that is trivial
            else if (getNameAttributeName().equals(attributeKey) || getUserLoginAttributeName().equals(attributeKey)
                    || getMailAttributeName().equals(attributeKey)
                    || getDisplayNameAttributeName().equals(attributeKey))
                attributeValue = attributes.get(attributeKey);
            else if (ldapConfigurations.getAttributesMapping().get(attributeKey.toUpperCase()) != null)
            {
                attributeValue = attributes.get(attributeKey);
                attributeKey = ldapConfigurations.getAttributesMapping().get(attributeKey.toUpperCase());
            }
            else if (ldapConfigurations.getBulkParametersAttributeName() != null)
            {
                isBulkParam = true;
                attributeValue = attributes.get(attributeKey);
                attributeBulkKey = attributeKey;
                attributeKey = ldapConfigurations.getBulkParametersAttributeName();
            }
            else
                attributeValue = attributes.get(attributeKey);
            query.append("(" + attributeKey + "=" + (isBulkParam ? "*" + attributeBulkKey + "=" : "") + attributeValue
                    + (isBulkParam ? ";*" : "") + ")");
        }
        return query.toString();
    }

    /**
     * Calculates an entity's distinguished name. <br>
     * <br>
     * To be implemented for each LDAP technology.
     * 
     * @param commonName
     *            the entity's common name
     * @param mainGroupCommonName
     *            the entity's main group name
     * @return the entity's distinguished name
     * @throws LDAPOperationException
     *             if the distinguished name can't be calculated
     */
    abstract protected String calculateDistinguishedName(String commonName, String mainGroupCommonName)
            throws LDAPOperationException;

    /**
     * @see pt.digitalis.utils.ldap.ILDAPUtils#changePassword(java.lang.String, java.lang.String)
     */
    abstract public void changePassword(String loginName, String newPassword) throws LDAPOperationException;

    /**
     * Converts an Attributes object into an LDAPGroup.
     * 
     * @param attributes
     *            the object to convert
     * @param distinguishedName
     *            the group DN
     * @return the converted object
     */
    protected LDAPGroup convertFromAttributesToLDAPGroup(Attributes attributes, String distinguishedName)
    {

        // Init result
        LDAPGroup ldapGroup = new LDAPGroup();

        // Set DN
        ldapGroup.setDistinguishedName(distinguishedName);

        // Populate fields
        try
        {

            if (attributes.get(LDAPEntity.COMMON_NAME) != null)
            {
                String groupMappingValue = this.ldapConfigurations.getGroupMapping(attributes
                        .get(LDAPEntity.COMMON_NAME).get().toString());
                if (groupMappingValue != null && !"".equals(groupMappingValue))
                    ldapGroup.setCommonName(groupMappingValue);
                else
                    ldapGroup.setCommonName(attributes.get(LDAPEntity.COMMON_NAME).get().toString());
            }
            else
                ldapGroup.setCommonName(NON_AVAILABLE);

            if (attributes.get(getNameAttributeName()) != null)
                ldapGroup.setName(attributes.get(getNameAttributeName()).get().toString());
            else
                ldapGroup.setName(NON_AVAILABLE);

            if (attributes.get(LDAPEntity.DESCRIPTION) != null)
                ldapGroup.setDescription(attributes.get(LDAPEntity.DESCRIPTION).get().toString());
            else
                ldapGroup.setDescription(NON_AVAILABLE);

            if (attributes.get(getGroupParentGroupAttributeName()) != null)
                ldapGroup.setParentGroupDN(attributes.get(getGroupParentGroupAttributeName()).get().toString());
            else
                ldapGroup.setParentGroupDN(NON_AVAILABLE);

        }
        catch (NamingException namingException)
        {
            namingException.printStackTrace();
        }

        // Return result
        return ldapGroup;
    }

    /**
     * Converts an Attributes object into an LDAPUser.
     * 
     * @param attributes
     *            the object to convert
     * @param distinguishedName
     *            the user DN
     * @return the converted object
     */
    protected LDAPUser convertFromAttributesToLDAPUser(Attributes attributes, String distinguishedName)
    {
        return convertFromAttributesToLDAPUser(attributes, distinguishedName, true);
    }

    /**
     * Converts an Attributes object into an LDAPUser.
     * 
     * @param attributes
     *            the object to convert
     * @param distinguishedName
     *            the user DN
     * @param convertAttributes
     *            the convert attributes
     * @return the converted object
     */
    protected LDAPUser convertFromAttributesToLDAPUser(Attributes attributes, String distinguishedName,
            boolean convertAttributes)
    {

        // Init result
        LDAPUser ldapUser = new LDAPUser();

        // Populate fields
        ldapUser.setDistinguishedName(distinguishedName);

        try
        {
            /*
             * Implementation Note: The following idiom is used to prevent NullPointerException on null attribute values
             * when methods are called upon the attribute value. if(attribute != null)
             * transferObject.setValue(<attribute-value>); else transferObject.setValue("N/A"); If the attribute value
             * is null the transfer object member is set to null, and the execution flow is prevented to run through a
             * method call on a null object.
             */

            if (attributes.get(LDAPEntity.COMMON_NAME) != null)
                ldapUser.setCommonName(attributes.get(LDAPEntity.COMMON_NAME).get().toString());
            else
                ldapUser.setCommonName(NON_AVAILABLE);

            if (attributes.get(LDAPEntity.NAME) != null)
                ldapUser.setName(attributes.get(LDAPEntity.NAME).get().toString());
            else
                ldapUser.setName(NON_AVAILABLE);

            if (attributes.get(LDAPUser.DISPLAY_NAME) != null)
                ldapUser.setDisplayName(attributes.get(LDAPUser.DISPLAY_NAME).get().toString());
            else
                ldapUser.setDisplayName(NON_AVAILABLE);

            if (attributes.get(LDAPUser.GIVEN_NAME) != null)
                ldapUser.setGivenName(attributes.get(LDAPUser.GIVEN_NAME).get().toString());
            else
                ldapUser.setGivenName(NON_AVAILABLE);

            if (attributes.get(getUserLoginAttributeName()) != null)
                ldapUser.setLoginName(attributes.get(getUserLoginAttributeName()).get().toString());
            else
                ldapUser.setLoginName(NON_AVAILABLE);

            if (attributes.get(getMailAttributeName()) != null)
                ldapUser.setEmail(attributes.get(getMailAttributeName()).get().toString());
            else
                ldapUser.setEmail(NON_AVAILABLE);

            if (attributes.get(getUserParentGroupAttributeName()) != null)
                ldapUser.setParentGroupDN(attributes.get(getUserParentGroupAttributeName()).get().toString());
            else
                ldapUser.setParentGroupDN(NON_AVAILABLE);

            if (convertAttributes)
            {

                // Bulk parameters
                Map<String, String> bulkParametres = this.getBulkParametersValues(attributes);
                for (Entry<String, String> entry: bulkParametres.entrySet())
                {
                    ldapUser.setParameter(entry.getKey(), entry.getValue());
                }

                /*
                 * Parameters. Implementation note: this snippet captures all the attributes that are not the ones
                 * above. It's not possible to know what attributes were written by the client application, so all the
                 * attributes that are not the ones above are treated as parameters. It makes no difference for
                 * parameter management, since LDAP Utils assumes that the client application knows how to map from
                 * attributes to app parameters.
                 */
                NamingEnumeration<String> attributeNames = attributes.getIDs();
                try
                {
                    while (attributeNames.hasMore())
                    {
                        String attributeName = attributeNames.next();

                        if (!LDAPEntity.COMMON_NAME.equals(attributeName) && !LDAPEntity.NAME.equals(attributeName)
                                && !LDAPUser.DISPLAY_NAME.equals(attributeName)
                                && !LDAPUser.GIVEN_NAME.equals(attributeName)
                                && !getUserLoginAttributeName().equals(attributeName)
                                && !getMailAttributeName().equals(attributeName)
                                && !getPasswordAttributeName().equals(attributeName)
                                && !getUserParentGroupAttributeName().equals(attributeName)
                                && !getConfigurations().getBulkParametersAttributeName().equals(attributeName))

                            if (getConfigurations().getAttributesMapping().containsValue(attributeName))
                            {

                                ldapUser.setParameter(getKey(attributeName), attributes.get(attributeName).get()
                                        .toString());
                            }
                            else
                            {
                                ldapUser.setParameter(attributeName, attributes.get(attributeName).get().toString());
                            }
                    }
                }
                finally
                {
                    attributeNames.close();
                }

            }
            else
            {
                NamingEnumeration<String> attributeNames = attributes.getIDs();
                try
                {
                    while (attributeNames.hasMore())
                    {
                        String attributeName = attributeNames.next();
                        ldapUser.setParameter(attributeName, attributes.get(attributeName).get().toString());
                    }
                }
                finally
                {
                    attributeNames.close();
                }

            }

        }
        catch (NamingException namingException)
        {
            namingException.printStackTrace();
        }

        // Return result
        return ldapUser;
    }

    /**
     * Converts a SearchResult object into an LDAPGroup.
     * 
     * @param searchResult
     *            the object to convert
     * @return the converted object
     */
    protected LDAPGroup convertFromSearchResultToLDAPGroup(SearchResult searchResult)
    {
        // Init result
        LDAPGroup ldapGroup = null;

        if (searchResult != null)
        {
            // Convert from search results attributes to group
            ldapGroup = convertFromAttributesToLDAPGroup(searchResult.getAttributes(),
                    searchResult.getNameInNamespace());
        }

        // Return result
        return ldapGroup;
    }

    /**
     * Converts a SearchResult object into an LDAPUser.
     * 
     * @param searchResult
     *            the object to convert
     * @return the converted object
     */
    protected LDAPUser convertFromSearchResultToLDAPUser(SearchResult searchResult)
    {
        // Init result
        LDAPUser ldapUser = null;

        if (searchResult != null)
        {
            // Convert from search results attributes to user
            ldapUser = convertFromAttributesToLDAPUser(searchResult.getAttributes(), searchResult.getNameInNamespace(),
                    true);
        }

        // Return result
        return ldapUser;
    }

    /**
     * Converts a SearchResult object into an LDAPUser.
     * 
     * @param searchResult
     *            the object to convert
     * @param convertAttributes
     *            the convert attributes
     * @return the converted object
     */
    protected LDAPUser convertFromSearchResultToLDAPUser(SearchResult searchResult, boolean convertAttributes)
    {
        // Init result
        LDAPUser ldapUser = null;

        if (searchResult != null)
        {
            // Convert from search results attributes to user
            ldapUser = convertFromAttributesToLDAPUser(searchResult.getAttributes(), searchResult.getNameInNamespace(),
                    convertAttributes);
        }

        // Return result
        return ldapUser;
    }

    /**
     * Converts a search result into a map.
     * 
     * @param searchResult
     *            the search result object
     * @return the map with the converted values
     * @throws LDAPOperationException
     *             if the conversion can't be performed
     */
    final protected Map<String, Object> convertFromSearchResultToMap(SearchResult searchResult)
            throws LDAPOperationException
    {

        Map<String, Object> attributeMap = new CaseInsensitiveHashMap<Object>();

        if (searchResult != null)
        {
            Attributes attributes = searchResult.getAttributes();
            NamingEnumeration<? extends Attribute> attrs = attributes.getAll();

            try
            {
                while (attrs.hasMore())
                {
                    Attribute attribute = attrs.next();

                    if (attribute.getID().equalsIgnoreCase(getConfigurations().getBulkParametersAttributeName()))
                    {

                        // Bulk parameters
                        Map<String, String> bulkParametres = this.getBulkParametersValues(attributes);
                        for (Entry<String, String> entry: bulkParametres.entrySet())
                        {
                            attributeMap.put(entry.getKey(), entry.getValue());
                        }

                    }
                    else
                    {
                        String commonName = null;
                        if (attribute.getID().equals(getCommonName())
                                && this.getConfigurations().getGroupMapping(attribute.get().toString()) != null)
                        {
                            commonName = this.getConfigurations().getGroupMapping(attribute.get().toString());
                        }

                        if (getConfigurations().getAttributesMapping().containsValue(attribute.getID()))
                        {
                            attributeMap.put(getKey(attribute.getID()),
                                    commonName != null ? commonName : attribute.get());
                        }
                        else
                        {
                            attributeMap.put(attribute.getID(), commonName != null ? commonName : attribute.get());
                        }

                    }
                }
            }
            catch (NamingException namingException)
            {
                throw new LDAPOperationException("Could not fetch attribute!", namingException);
            }
            finally
            {
                try
                {
                    attrs.close();
                }
                catch (NamingException e)
                {
                    throw new LDAPOperationException("Error closing NamingEnumeration!", e);
                }
            }
        }

        return attributeMap;
    }

    /**
     * @see pt.digitalis.utils.ldap.ILDAPUtils#countAllGroups()
     */
    public int countAllGroups() throws NamingException, LDAPOperationException
    {
        return countAllGroups(false);
    }

    /**
     * @see pt.digitalis.utils.ldap.ILDAPUtils#countAllGroups(boolean)
     */
    public int countAllGroups(boolean commonNameDistinct) throws NamingException, LDAPOperationException
    {
        String search = "(" + getObjectClassName() + "=" + getGroupClassName() + ")";
        LdapContext context = getLDAPContext();
        try
        {
            List<SearchResult> returnedGroups = doLDAPCount(context, getConfigurations().getBaseSearchDN(), search);

            // HashSet create to purge duplicate groups. Can't use "countAllGroups" because it returns duplicated values
            if (commonNameDistinct)
            {
                Set<String> groupsCommonName = new HashSet<String>();

                for (SearchResult searchResult: returnedGroups)
                {
                    LDAPGroup group = convertFromSearchResultToLDAPGroup(searchResult);

                    if (searchResult != null && (searchResult.getObject() instanceof Context))
                    {
                        ((Context) searchResult.getObject()).close();
                    }

                    groupsCommonName.add(group.getCommonName());
                }
                return groupsCommonName.size();

            }
            else
            {
                return returnedGroups.size();
            }
        }
        catch (NamingException namingException)
        {
            throw new LDAPOperationException("Could not count all groups!", namingException);
        }
        catch (IOException e)
        {
            throw new LDAPOperationException("Could not count all groups!", e);
        }
        finally
        {
            context.close();
        }
    }

    /**
     * @see pt.digitalis.utils.ldap.ILDAPUtils#countAllGroupsOfUser(java.lang.String)
     */
    public int countAllGroupsOfUser(String loginName) throws NamingException, LDAPOperationException
    {
        Set<String> result = new HashSet<String>();
        LDAPUser user = findUserByLogin(loginName);

        String search = "(&" + "(" + getObjectClassName() + "=" + getGroupClassName() + ")(" + getGroupAttributeName()
                + "=" + user.getDistinguishedName() + "))";
        LdapContext context = getLDAPContext();
        try
        {
            List<SearchResult> returnedGroups = null;

            returnedGroups = doLDAPCount(context, getConfigurations().getBaseSearchDN(), search);

            for (SearchResult searchResult: returnedGroups)
            {
                result.add(searchResult.getNameInNamespace().toLowerCase());

                if (searchResult.getObject() instanceof Context)
                    ((Context) searchResult.getObject()).close();

            }
        }
        catch (NamingException namingException)
        {
            throw new LDAPOperationException("Could not get groups of user " + loginName + "!", namingException);
        }
        catch (IOException e)
        {
            throw new LDAPOperationException("Could not get groups of user " + loginName + "!", e);
        }
        finally
        {
            context.close();
        }

        return result.size();
    }

    /**
     * @see pt.digitalis.utils.ldap.ILDAPUtils#countAllUsers()
     */
    public int countAllUsers() throws NamingException, LDAPOperationException
    {
        String search = "(" + getObjectClassName() + "=" + getUserClassName() + ")";
        LdapContext context = getLDAPContext();
        try
        {

            List<SearchResult> returnedUsers = doLDAPCount(context, getConfigurations().getBaseSearchDN(), search);

            return returnedUsers.size();
        }
        catch (NamingException namingException)
        {
            throw new LDAPOperationException("Could not count all users!", namingException);
        }
        catch (IOException e)
        {
            throw new LDAPOperationException("Could not count all users!", e);
        }
        finally
        {
            context.close();
        }
    }

    /**
     * @see pt.digitalis.utils.ldap.ILDAPUtils#countAllUsers(java.lang.String)
     */
    public int countAllUsers(String groupId) throws LDAPOperationException
    {
        return getGroupMembers(groupId).size();
    }

    /**
     * @see pt.digitalis.utils.ldap.ILDAPUtils#countUsers(java.util.Map)
     */
    public int countUsers(Map<String, String> attributes) throws LDAPOperationException
    {
        StringBuffer query = new StringBuffer("(&" + "(" + getObjectClassName() + "=" + getUserClassName() + ")");

        query.append(this.buildAttributesString(attributes));
        query.append(")");

        LdapContext context = getLDAPContext();
        try
        {
            List<SearchResult> returnedUsers = doLDAPCount(context, getConfigurations().getBaseSearchDN(),
                    query.toString());

            return returnedUsers.size();
        }
        catch (NamingException namingException)
        {
            throw new LDAPOperationException("Could not count all groups!", namingException);
        }
        catch (IOException e)
        {
            throw new LDAPOperationException("Could not count all groups!", e);
        }
        finally
        {
            try
            {
                context.close();
            }
            catch (NamingException e)
            {
                throw new LDAPOperationException("Error closing context!", e);
            }
        }

    }

    /**
     * Creates a new LDAP subcontext with the given distinguished name and attributes.
     * 
     * @param distinguishedName
     *            the subcontext's distinguished name
     * @param attributes
     *            the subcontext's attributes
     * @throws LDAPOperationException
     *             if the subcontext can't be created
     */
    final private void createSubcontext(String distinguishedName, Attributes attributes) throws LDAPOperationException
    {
        LdapContext ctx = getLDAPContext();
        try
        {
            ctx.createSubcontext(distinguishedName, attributes).close();

        }
        catch (NamingException namingException)
        {
            throw new LDAPOperationException("Could not create LDAP subcontext!!", namingException);
        }
        finally
        {
            try
            {
                ctx.close();
            }
            catch (NamingException e)
            {
                throw new LDAPOperationException("Error closing context!", e);
            }
        }
    }

    /**
     * Destroys the existing LDAP subcontext with the given distinguished name.
     * 
     * @param distinguishedName
     *            the subcontext's distinguished name
     * @throws LDAPOperationException
     *             if the subcontext can't be destroyed
     */
    final private void destroySubcontext(String distinguishedName) throws LDAPOperationException
    {
        LdapContext ctx = getLDAPContext();
        try
        {
            ctx.destroySubcontext(distinguishedName);
        }
        catch (NamingException namingException)
        {
            throw new LDAPOperationException("Could not remove LDAP subcontext!!", namingException);
        }
        finally
        {
            try
            {
                ctx.close();
            }
            catch (NamingException e)
            {
                throw new LDAPOperationException("Error closing context!", e);
            }
        }
    }

    /**
     * Performs a lightweight search on LDAP, for counting purposes.
     * 
     * @param context
     *            a valid Ldap context
     * @param baseNode
     *            the base node to start the search
     * @param searchCriteria
     *            the criteria to search
     * @return an enumeration of SearchResults for the objects that satisfy the search criteria.
     * @throws NamingException
     *             if the search cannot be executed
     * @throws LDAPOperationException
     *             if there is an error creating the LDAP context
     * @throws IOException
     *             Signals that an I/O exception has occurred.
     */
    public List<SearchResult> doLDAPCount(LdapContext context, String baseNode, String searchCriteria)
            throws NamingException, LDAPOperationException, IOException
    {
        List<SearchResult> results = new ArrayList<SearchResult>();

        try
        {
            // Define scope of search and return type
            SearchControls searchConfigurations = new SearchControls();
            // Return the entire objects
            searchConfigurations.setReturningObjFlag(false);
            // Search all the subtree below the supplied base node
            searchConfigurations.setSearchScope(SearchControls.SUBTREE_SCOPE);
            getLogger().debug("ldapSearch - baseNode: " + baseNode);
            getLogger().debug("ldapSearch - searchCriteria: " + searchCriteria);

            //
            /* Results per page configuration */

            byte[] cookie = null;
            context.setRequestControls(new Control[] {new PagedResultsControl(getConfigurations()
                    .getQuerysPageSizeLimit(), Control.CRITICAL)});

            /*
             * Cycle For each page result.
             */
            do
            {
                NamingEnumeration<SearchResult> resultsEnumeration = context.search(baseNode, searchCriteria,
                        searchConfigurations);

                results.addAll(Collections.list(resultsEnumeration));

                /* Examine the response controls */
                cookie = parseControls(context.getResponseControls());

                /*
                 * LDAP paging system uses a cookie on the server to save request data , this cookie must be saved into
                 * the connection each time a search is made
                 */
                context.setRequestControls(new Control[] {new PagedResultsControl(getConfigurations()
                        .getQuerysPageSizeLimit(), cookie, Control.CRITICAL)});

                // resultsEnumeration.close();
            }
            while ((cookie != null) && (cookie.length != 0));

        }
        finally
        {
            context.close();
        }

        // Return the search performed on the context
        return results;
    }

    /**
     * Performs an LDAP tree search that returns a single value. If more than one value apply the first one is returned
     * 
     * @param search
     *            the search criteria
     * @return the result of the search
     * @throws LDAPOperationException
     *             if the search can't be executed or if more than one entity exists with the same criteria
     */
    private SearchResult doLDAPSearchFirstReturn(String search) throws LDAPOperationException
    {
        // Initialized result
        SearchResult result = null;
        LdapContext context = getLDAPContext();
        try
        {
            // Perform search
            List<SearchResult> ldapResult = doLDAPSearchMultipleReturns(context, getConfigurations().getBaseSearchDN(),
                    search);

            // Check if there is more than one user with the same login name...
            if (ldapResult.size() > 0)
            {

                // No need to check if there are more than one results, will always return the first one!
                result = ldapResult.get(0);

            }
        }
        catch (NamingException namingException)
        {
            throw new LDAPOperationException(namingException);
        }
        catch (IOException ioException)
        {
            throw new LDAPOperationException(ioException);
        }
        finally
        {
            try
            {
                context.close();
            }
            catch (NamingException e)
            {
                throw new LDAPOperationException("Error closing context!", e);
            }
        }

        // Return result
        return result;
    }

    /**
     * Performs an LDAP tree search that can return multiple values.
     * 
     * @param context
     *            a valid Ldap context
     * @param baseNode
     *            the base node to start the search
     * @param searchCriteria
     *            the criteria to search
     * @return an enumeration of SearchResults for the objects that satisfy the search criteria.
     * @throws NamingException
     *             if the search cannot be executed
     * @throws LDAPOperationException
     *             if there is an error creating the LDAP context
     * @throws IOException
     *             Signals that an I/O exception has occurred.
     */
    public List<SearchResult> doLDAPSearchMultipleReturns(LdapContext context, String baseNode, String searchCriteria)
            throws NamingException, LDAPOperationException, IOException
    {
        return doLDAPSearchMultipleReturnsPaging(context, baseNode, searchCriteria, getConfigurations()
                .getQuerysPageSizeLimit(), null);
    }

    /**
     * Performs an LDAP tree search that can return multiple values.
     * 
     * @param context
     *            a valid Ldap context
     * @param baseNode
     *            the base node to start the search
     * @param searchCriteria
     *            the criteria to search
     * @param rowsPerPage
     *            the request number of results limit
     * @param pageToReturn
     *            the page to return, null if all pages must be returned
     * @return an enumeration of SearchResults for the objects that satisfy the search criteria.
     * @throws NamingException
     *             if the search cannot be executed
     * @throws LDAPOperationException
     *             if there is an error creating the LDAP context
     * @throws IOException
     *             Signals that an I/O exception has occurred.
     */
    public List<SearchResult> doLDAPSearchMultipleReturnsPaging(LdapContext context, String baseNode,
            String searchCriteria, int rowsPerPage, Integer pageToReturn) throws NamingException,
            LDAPOperationException, IOException
    {
        List<SearchResult> results = new ArrayList<SearchResult>();
        try
        {
            // Define scope of search and return type
            SearchControls searchConfigurations = new SearchControls();
            // Return the entire objects
            searchConfigurations.setReturningObjFlag(true);
            // Search all the subtree below the supplied base node
            searchConfigurations.setSearchScope(SearchControls.SUBTREE_SCOPE);
            getLogger().debug("ldapSearch - baseNode: " + baseNode);
            getLogger().debug("ldapSearch - searchCriteria: " + searchCriteria);

            //
            /* Results per page configuration */

            byte[] cookie = null;
            int pageCount = 0;
            context.setRequestControls(new Control[] {new PagedResultsControl(rowsPerPage, Control.CRITICAL)});

            /*
             * Cycle For each page result.
             */
            do
            {
                pageCount++;
                NamingEnumeration<SearchResult> resultsEnumeration = context.search(baseNode, searchCriteria,
                        searchConfigurations);

                if (pageToReturn == null || (pageToReturn == pageCount))
                {
                    results.addAll(Collections.list(resultsEnumeration));
                }
                else
                {
                    Collections.list(resultsEnumeration);
                }

                /* Examine the response controls */
                cookie = parseControls(context.getResponseControls());

                /*
                 * LDAP paging system uses a cookie on the server to save request data , this cookie must be saved into
                 * the connection each time a search is made
                 */
                context.setRequestControls(new Control[] {new PagedResultsControl(rowsPerPage, cookie, Control.CRITICAL)});

                // resultsEnumeration.close();
            }
            while ((cookie != null) && (cookie.length != 0) && (pageToReturn == null || pageCount < pageToReturn));
        }
        finally
        {
            context.close();
        }

        // Return the search performed on the context
        return results;
    }

    /**
     * Performs an LDAP tree search that returns a single value.
     * 
     * @param search
     *            the search criteria
     * @return the result of the search
     * @throws LDAPOperationException
     *             if the search can't be executed or if more than one entity exists with the same criteria
     */
    private SearchResult doLDAPSearchSingleReturn(String search) throws LDAPOperationException
    {
        // Initialized result
        SearchResult result = null;

        LdapContext context = getLDAPContext();

        try
        {
            // Flag to control if there are more than one entity with the same common name
            boolean entityLoaded = false;

            // Perform search
            List<SearchResult> ldapResult = doLDAPSearchMultipleReturns(context, getConfigurations().getBaseSearchDN(),
                    search);

            // Check if there is more than one user with the same login name...
            for (SearchResult searchResult: ldapResult)
            {

                if (!entityLoaded)
                {
                    entityLoaded = true;
                }
                else
                {
                    // ...if so, throw exception as this is not allowed!
                    throw new LDAPOperationException("More than one entity exists with the same searched criteria!");
                }

                result = searchResult;
            }
        }
        catch (NamingException namingException)
        {
            throw new LDAPOperationException(namingException);
        }
        catch (IOException e)
        {
            throw new LDAPOperationException(e);
        }
        finally
        {
            try
            {
                context.close();
            }
            catch (NamingException e)
            {
                throw new LDAPOperationException("Error closing context!", e);
            }
        }

        // Return result
        return result;
    }

    /**
     * @see pt.digitalis.utils.ldap.ILDAPUtils#findAllGroups()
     */
    public Set<LDAPGroup> findAllGroups() throws LDAPOperationException
    {
        Set<LDAPGroup> groups = new HashSet<LDAPGroup>();

        String search = "(" + getObjectClassName() + "=" + getGroupClassName() + ")";

        LdapContext context = getLDAPContext();

        try
        {
            List<SearchResult> returnedGroups = doLDAPSearchMultipleReturns(context, getConfigurations()
                    .getBaseSearchDN(), search);

            for (SearchResult searchResult: returnedGroups)
            {
                LDAPGroup group = convertFromSearchResultToLDAPGroup(searchResult);

                if (searchResult != null && (searchResult.getObject() instanceof Context))
                {
                    ((Context) searchResult.getObject()).close();
                }

                groups.add(group);
            }

        }
        catch (NamingException namingException)
        {
            throw new LDAPOperationException("Could not get all groups!", namingException);
        }
        catch (IOException e)
        {
            throw new LDAPOperationException("Could not get all groups!", e);
        }
        finally
        {
            try
            {
                context.close();
            }
            catch (NamingException e)
            {
                throw new LDAPOperationException("Error closing context!", e);
            }
        }
        return groups;

    }

    /**
     * @see pt.digitalis.utils.ldap.ILDAPUtils#findAllUsers()
     */
    public Set<LDAPUser> findAllUsers() throws LDAPOperationException
    {
        Set<LDAPUser> users = new HashSet<LDAPUser>();

        String search = "(" + getObjectClassName() + "=" + getUserClassName() + ")";

        LdapContext context = getLDAPContext();

        try
        {

            List<SearchResult> returnedUsers = doLDAPSearchMultipleReturns(context, getConfigurations()
                    .getBaseSearchDN(), search);

            for (SearchResult searchResult: returnedUsers)
            {
                LDAPUser user = convertFromSearchResultToLDAPUser(searchResult);

                if (searchResult != null && (searchResult.getObject() instanceof Context))
                {
                    ((Context) searchResult.getObject()).close();
                }

                users.add(user);
            }
        }
        catch (NamingException namingException)
        {
            throw new LDAPOperationException("Could not get all users!", namingException);
        }
        catch (IOException e)
        {
            throw new LDAPOperationException("Could not get all users!", e);
        }
        finally
        {
            try
            {
                context.close();
            }
            catch (NamingException e)
            {
                throw new LDAPOperationException("Error closing context!", e);
            }
        }
        return users;
    }

    /**
     * @see pt.digitalis.utils.ldap.ILDAPUtils#findGroupByCommonName(java.lang.String)
     */
    public LDAPGroup findGroupByCommonName(String cn) throws LDAPOperationException
    {
        SearchResult searchResult = getGroupByCommonName(cn);
        LDAPGroup converted = convertFromSearchResultToLDAPGroup(searchResult);

        if (searchResult != null && (searchResult.getObject() instanceof Context))
        {
            try
            {
                ((Context) searchResult.getObject()).close();
            }
            catch (NamingException e)
            {
                e.printStackTrace();
            }
        }

        return converted;
    }

    /**
     * @see pt.digitalis.utils.ldap.ILDAPUtils#findGroupByDistinguishedName(java.lang.String)
     */
    public LDAPGroup findGroupByDistinguishedName(String dn) throws LDAPOperationException
    {
        return getGroupByDistinguishedName(dn);
    }

    /**
     * @see pt.digitalis.utils.ldap.ILDAPUtils#findGroups(int, int)
     */
    public Set<LDAPGroup> findGroups(int rowsPerPage, int pageToReturn) throws LDAPOperationException
    {
        Set<LDAPGroup> groups = new HashSet<LDAPGroup>();

        String search = "(" + getObjectClassName() + "=" + getGroupClassName() + ")";

        LdapContext context = getLDAPContext();

        try
        {

            List<SearchResult> returnedGroups = doLDAPSearchMultipleReturnsPaging(context, getConfigurations()
                    .getBaseSearchDN(), search, rowsPerPage, pageToReturn);

            for (SearchResult searchResult: returnedGroups)
            {
                LDAPGroup group = convertFromSearchResultToLDAPGroup(searchResult);

                if (searchResult != null && (searchResult.getObject() instanceof Context))
                {
                    ((Context) searchResult.getObject()).close();
                }

                groups.add(group);
            }
        }
        catch (NamingException namingException)
        {
            throw new LDAPOperationException("Could not get all groups!", namingException);
        }
        catch (IOException e)
        {
            throw new LDAPOperationException("Could not get all groups!", e);
        }
        finally
        {
            try
            {
                context.close();
            }
            catch (NamingException e)
            {
                throw new LDAPOperationException("Error closing context!", e);
            }
        }
        return groups;
    }

    /**
     * @see pt.digitalis.utils.ldap.ILDAPUtils#findGroupsOfUser(java.lang.String)
     */
    public Set<LDAPGroup> findGroupsOfUser(String loginName) throws LDAPOperationException
    {
        return this.findGroupsOfUserPagination(loginName, null, null);
    }

    /**
     * @see pt.digitalis.utils.ldap.ILDAPUtils#findGroupsOfUserPagination(java.lang.String, java.lang.Integer,
     *      java.lang.Integer)
     */
    public Set<LDAPGroup> findGroupsOfUserPagination(String loginName, Integer rowsPerPage, Integer pageToReturn)
            throws LDAPOperationException
    {
        HashMap<String, LDAPGroup> groups = new HashMap<String, LDAPGroup>();
        LDAPGroup group = null;
        LDAPUser user = findUserByLogin(loginName);
        boolean hasPagination = pageToReturn != null && rowsPerPage != null;

        String search = "(&" + "(" + getObjectClassName() + "=" + getGroupClassName() + ")(" + getGroupAttributeName()
                + "=" + user.getDistinguishedName() + "))";

        LdapContext context = getLDAPContext();

        try
        {
            List<SearchResult> returnedGroups = null;

            if (hasPagination)
            {
                returnedGroups = doLDAPSearchMultipleReturnsPaging(context, getConfigurations().getBaseSearchDN(),
                        search, rowsPerPage, pageToReturn);
            }
            else
            {
                returnedGroups = doLDAPSearchMultipleReturns(context, getConfigurations().getBaseSearchDN(), search);
            }

            for (SearchResult searchResult: returnedGroups)
            {

                group = convertFromSearchResultToLDAPGroup(searchResult);

                if (searchResult != null && (searchResult.getObject() instanceof Context))
                {
                    ((Context) searchResult.getObject()).close();
                }

                groups.put(group.getCommonName(), group);
            }

        }
        catch (NamingException namingException)
        {
            throw new LDAPOperationException("Could not get groups of user " + loginName + "!", namingException);
        }
        catch (IOException e)
        {
            throw new LDAPOperationException("Could not get groups of user " + loginName + "!", e);
        }
        finally
        {
            try
            {
                context.close();
            }
            catch (NamingException e)
            {
                throw new LDAPOperationException("Error closing context!", e);
            }
        }

        return new HashSet<LDAPGroup>(groups.values());
    }

    /**
     * @see pt.digitalis.utils.ldap.ILDAPUtils#findUserByDistinguishedName(java.lang.String)
     */
    public LDAPUser findUserByDistinguishedName(String dn) throws LDAPOperationException
    {
        return getUserByDistinguishedName(dn);
    }

    /**
     * @see pt.digitalis.utils.ldap.ILDAPUtils#findUserByLogin(java.lang.String)
     */
    public LDAPUser findUserByLogin(String loginName) throws LDAPOperationException
    {
        return findUserByLogin(loginName, true);
    }

    /**
     * Returns the user with a given login name.
     * 
     * @param loginName
     *            the user's login name
     * @param convertAttributes
     *            the convert attributes
     * @return the LDAPUser with the given login name
     * @throws LDAPOperationException
     *             if the operation cannot be executed
     */
    public LDAPUser findUserByLogin(String loginName, boolean convertAttributes) throws LDAPOperationException
    {
        SearchResult result = getByLogin(loginName);
        LDAPUser converted = convertFromSearchResultToLDAPUser(result, convertAttributes);

        if (result != null && (result.getObject() instanceof Context))
        {
            try
            {
                ((Context) result.getObject()).close();
            }
            catch (NamingException e)
            {
                e.printStackTrace();
            }
        }

        return converted;
    }

    /**
     * @see pt.digitalis.utils.ldap.ILDAPUtils#findUsers(int, int)
     */
    public Set<LDAPUser> findUsers(int rowsPerPage, int pageToReturn) throws LDAPOperationException
    {
        Set<LDAPUser> users = new HashSet<LDAPUser>();

        String search = "(" + getObjectClassName() + "=" + getUserClassName() + ")";
        LdapContext context = getLDAPContext();

        try
        {

            List<SearchResult> returnedUsers = doLDAPSearchMultipleReturnsPaging(context, getConfigurations()
                    .getBaseSearchDN(), search, rowsPerPage, pageToReturn);

            for (SearchResult searchResult: returnedUsers)
            {
                LDAPUser user = convertFromSearchResultToLDAPUser(searchResult);

                if (searchResult != null && (searchResult.getObject() instanceof Context))
                {
                    ((Context) searchResult.getObject()).close();
                }

                users.add(user);
            }
        }
        catch (NamingException namingException)
        {
            throw new LDAPOperationException("Could not get all users!", namingException);
        }
        catch (IOException e)
        {
            throw new LDAPOperationException("Could not get all users!", e);
        }
        finally
        {
            try
            {
                context.close();
            }
            catch (NamingException e)
            {
                throw new LDAPOperationException("Error closing context!", e);
            }
        }

        return users;
    }

    /**
     * @see pt.digitalis.utils.ldap.ILDAPUtils#findUsersByAttribute(java.lang.String, java.lang.String)
     */
    public Set<LDAPUser> findUsersByAttribute(String attribute, String value) throws LDAPOperationException
    {
        Map<String, String> attributes = new HashMap<String, String>();
        attributes.put(attribute, value);

        return this.findUsersByAttributes(attributes);
    }

    /**
     * @see pt.digitalis.utils.ldap.ILDAPUtils#findUsersByAttributes(java.util.Map)
     */
    public Set<LDAPUser> findUsersByAttributes(Map<String, String> attributes) throws LDAPOperationException
    {
        return this.findUsersByAttributes(attributes, null, null);
    }

    /**
     * @see pt.digitalis.utils.ldap.ILDAPUtils#findUsersByAttributes(java.util.Map, Integer, Integer)
     */
    public Set<LDAPUser> findUsersByAttributes(Map<String, String> attributes, Integer rowsPerPage, Integer pageToReturn)
            throws LDAPOperationException
    {
        Set<LDAPUser> users = new HashSet<LDAPUser>();

        StringBuffer query = new StringBuffer("(&" + "(" + getObjectClassName() + "=" + getUserClassName() + ")");
        query.append(this.buildAttributesString(attributes));
        query.append(")");
        LdapContext context = getLDAPContext();

        try
        {
            List<SearchResult> returnedUsers = null;
            if (rowsPerPage != null && pageToReturn != null)
                returnedUsers = doLDAPSearchMultipleReturnsPaging(context, getConfigurations().getBaseSearchDN(),
                        query.toString(), rowsPerPage, pageToReturn);
            else
                returnedUsers = doLDAPSearchMultipleReturns(context, getConfigurations().getBaseSearchDN(),
                        query.toString());

            for (SearchResult searchResult: returnedUsers)
            {
                LDAPUser user = convertFromSearchResultToLDAPUser(searchResult);

                if (searchResult != null && (searchResult.getObject() instanceof Context))
                {
                    ((Context) searchResult.getObject()).close();
                }

                users.add(user);

            }
        }
        catch (NamingException namingException)
        {
            throw new LDAPOperationException("Could not get users using query " + query.toString(), namingException);
        }
        catch (IOException e)
        {
            throw new LDAPOperationException("Could not get users using query " + query.toString(), e);
        }
        finally
        {
            try
            {
                context.close();
            }
            catch (NamingException e)
            {
                throw new LDAPOperationException("Error closing context!", e);
            }
        }

        return users;
    }

    /**
     * @see pt.digitalis.utils.ldap.ILDAPUtils#findUsersByEmail(java.lang.String)
     */
    public Set<LDAPUser> findUsersByEmail(String value) throws LDAPOperationException
    {
        return findUsersByAttribute(getMailAttributeName(), value);
    }

    /**
     * @see pt.digitalis.utils.ldap.ILDAPUtils#findUsersInGroup(java.lang.String)
     */
    public Map<String, LDAPUser> findUsersInGroup(String groupCN) throws LDAPOperationException
    {
        // Initialize result
        Map<String, LDAPUser> usersFromGroup = new HashMap<String, LDAPUser>();

        // Get the DNs of the group members
        Vector<String> members = getGroupMembers(groupCN);

        if (members.size() > 0)
        {

            // Variable reuse
            LDAPUser groupUser = null;

            // Fetch users from DNs and put them on the result
            for (String memberDN: members)
            {
                /*
                 * Implementation Note: findUserByDistinguishedName(String) method throws NameNotFoundException if user
                 * can't be found. If a particular user can't be found the method's execution should proceed and the
                 * results returned to caller.
                 */
                try
                {
                    groupUser = findUserByDistinguishedName(memberDN);

                    if (!usersFromGroup.containsKey(groupUser.getCommonName()))
                        usersFromGroup.put(groupUser.getCommonName(), groupUser);
                }
                catch (LDAPOperationException ldapOperationException)
                {
                    if (!ldapOperationException.getCause().getClass().equals(NameNotFoundException.class))
                    {
                        throw new LDAPOperationException("Could not find users in group " + groupCN + "!",
                                ldapOperationException);
                    }
                }
            }
        }

        return usersFromGroup;
    }

    /**
     * Prepares the LDAP attributes for group addition.
     * 
     * @param newGroup
     *            the group to add
     * @return the attribute list
     * @throws LDAPOperationException
     *             if the attribute list can't be prepared
     */
    protected Attributes getAttributesForGroupAddition(LDAPGroup newGroup) throws LDAPOperationException
    {
        Attributes attrs = new BasicAttributes(true);

        attrs.put(getObjectClassName(), getGroupClassName());

        if (newGroup.getName() != null)
            attrs.put(getNameAttributeName(), newGroup.getName());

        if (newGroup.getDescription() != null)
            attrs.put(getDescriptionAttributeName(), newGroup.getDescription());

        if (newGroup.getParentGroupDN() != null)
            attrs.put(getGroupParentGroupAttributeName(), newGroup.getParentGroupDN());

        return attrs;
    }

    /**
     * Prepares the LDAP attributes for user addition.
     * 
     * @param newUser
     *            the user to add
     * @return the attribute list
     * @throws LDAPOperationException
     *             if the attribute list can't be prepared
     */
    protected Attributes getAttributesForUserAddition(LDAPUser newUser) throws LDAPOperationException
    {
        final String N_A = "N/A";

        Attributes attrs = new BasicAttributes(true);
        attrs.put(getObjectClassName(), getUserClassName());
        attrs.put(getUserLoginAttributeName(), newUser.getLoginName());
        attrs.put(getNameAttributeName(), newUser.getUserName());
        attrs.put(getMailAttributeName(), (newUser.getEmail() == null ? " " : newUser.getEmail()));
        attrs.put(getDisplayNameAttributeName(), newUser.getDisplayName());
        attrs.put(getGivenNameAttributeName(), newUser.getGivenName());

        if (N_A.equals(newUser.getParentGroupDN()) || newUser.getParentGroupDN() == null)
            attrs.put(getUserParentGroupAttributeName(), N_A);
        else
            attrs.put(getUserParentGroupAttributeName(), findGroupByDistinguishedName(newUser.getParentGroupDN())
                    .getDistinguishedName());

        // Parameters
        if (newUser.getParameters().size() > 0)
        {
            // Bulk parameters will be stored on a single attribute, so the attribute value must be built from the
            // LDAPUser data.
            StringBuilder attributeValue = new StringBuilder();

            // Each parameter will be stored on an attribute
            for (String attributeName: newUser.getParameters().keySet())
            {

                if (getConfigurations().getAttributesMapping().containsKey(attributeName))
                {
                    attrs.put(getConfigurations().getAttributesMapping().get(attributeName),
                            newUser.getParameter(attributeName));
                }
                else
                {
                    attributeValue.append(attributeName + "=" + newUser.getParameter(attributeName) + ";");
                }
            }
            if (attributeValue.length() > 0)
            {
                // Store the value on the attribute
                attrs.put(getConfigurations().getBulkParametersAttributeName(), attributeValue.toString());
            }

        }

        return attrs;
    }

    /**
     * Gets the bulk parameters values.
     * 
     * @param attributes
     *            the attributes
     * @return {@link HashMap} with bulk parameters
     * @throws NamingException
     *             the naming exception
     */
    private Map<String, String> getBulkParametersValues(Attributes attributes) throws NamingException
    {
        Map<String, String> result = new HashMap<String, String>();
        if (getConfigurations().getBulkParametersAttributeName() != null
                && attributes.get(getConfigurations().getBulkParametersAttributeName()) != null)
        {
            String[] bulkParameters = (attributes.get(getConfigurations().getBulkParametersAttributeName()).get()
                    .toString()).split(";");

            String[] bulkParameter = null;

            for (int i = 0; i < bulkParameters.length; i++)
            {
                // Split "key=value" around '='
                bulkParameter = bulkParameters[i].split("=");
                // bulkParameter[0] -> parameter name
                // bulkParameter[1] -> parameter value
                if (bulkParameter.length == 2)
                    result.put(bulkParameter[0], bulkParameter[1]);
            }
        }
        return result;
    }

    /**
     * Finds an entity by distinguished name. To be overridden on the different LDAP implementations.
     * 
     * @param distinguishedName
     *            the distinguished name
     * @return the result of the search by distinguished name
     * @throws LDAPOperationException
     *             if operation cannot be performed
     */
    protected SearchResult getByDistinguishedName(String distinguishedName) throws LDAPOperationException
    {
        // The search string
        String search = "(" + LDAPEntity.DISTINGUISHED_NAME + "=" + distinguishedName + ")";
        SearchResult result = doLDAPSearchSingleReturn(search);

        return result;
    }

    /**
     * Finds an user by login name. To be overridden on the different LDAP implementations.
     * 
     * @param loginName
     *            the login name of the user to find
     * @return the result of the search by login name
     * @throws LDAPOperationException
     *             if operation cannot be performed
     */
    protected SearchResult getByLogin(String loginName) throws LDAPOperationException
    {
        // The search string
        String search = "(&(" + getUserIdentifierName() + ")(" + getUserLoginAttributeName() + "=" + loginName + "))";
        SearchResult result = doLDAPSearchSingleReturn(search);

        return result;
    }

    /**
     * @see pt.digitalis.utils.ldap.ILDAPUtils#getChildGroupsByCN(java.lang.String)
     */
    public Set<LDAPGroup> getChildGroupsByCN(String commonName) throws LDAPOperationException
    {
        return getChildGroupsByDN(findGroupByCommonName(commonName).getDistinguishedName());
    }

    /**
     * @see pt.digitalis.utils.ldap.ILDAPUtils#getChildGroupsByDN(java.lang.String)
     */
    public Set<LDAPGroup> getChildGroupsByDN(String distinguishedName) throws LDAPOperationException
    {
        LdapContext context = getLDAPContext();
        try
        {
            Set<LDAPGroup> childGroups = new HashSet<LDAPGroup>();

            String search = "(" + getGroupParentGroupAttributeName() + "=" + distinguishedName + ")";
            List<SearchResult> groups = doLDAPSearchMultipleReturns(context, getConfigurations().getBaseSearchDN(),
                    search);

            for (SearchResult searchResult: groups)
            {

                LDAPGroup group = convertFromSearchResultToLDAPGroup(searchResult);
                if (searchResult != null && (searchResult.getObject() instanceof Context))
                {
                    ((LdapContext) searchResult.getObject()).close();
                }

                childGroups.add(group);
            }

            /* Fill's the parent data */
            Set<LDAPGroup> grandSonsGroups = new HashSet<LDAPGroup>();
            for (LDAPGroup group: childGroups)
            {
                if (!group.getDistinguishedName().equalsIgnoreCase(group.getParentGroupDN()))
                {
                    grandSonsGroups.addAll(getChildGroupsByDN(group.getDistinguishedName()));
                }
            }
            childGroups.addAll(grandSonsGroups);
            return childGroups;
        }
        catch (NamingException namingException)
        {
            throw new LDAPOperationException("Could not get child groups of " + distinguishedName + "!",
                    namingException);
        }
        catch (IOException e)
        {
            throw new LDAPOperationException("Could not get child groups of " + distinguishedName + "!", e);
        }
        finally
        {
            try
            {
                context.close();
            }
            catch (NamingException e)
            {
                throw new LDAPOperationException("Error closing context!", e);
            }
        }
    }

    /**
     * Returns the standard LDAP name for the 'cn' attribute.
     * 
     * @return the standard LDAP name for the 'cn' attribute
     */
    final protected String getCommonName()
    {
        return "cn";
    }

    /**
     * @see pt.digitalis.utils.ldap.ILDAPUtils#getConfigurations()
     */
    public LDAPConfigurations getConfigurations()
    {
        if (this.ldapConfigurations == null)
            this.ldapConfigurations = new ConfigurationsPreferencesImpl().readConfiguration(LDAPConfigurations.class);

        return this.ldapConfigurations;
    }

    /**
     * Returns the standard LDAP name for the 'description' attribute. To be overridden on the different LDAP
     * implementations. This method must NOT be qualified with 'final' and/or 'static' to preserve polymorphic behavior.
     * 
     * @return the standard LDAP name for the 'description' attribute
     */
    protected String getDescriptionAttributeName()
    {
        return "description";
    }

    /**
     * Returns the standard LDAP name for the 'displayName' attribute. To be overridden on the different LDAP
     * implementations. This method must NOT be qualified with 'final' and/or 'static' to preserve polymorphic behavior.
     * 
     * @return the standard LDAP name for the 'displayName' attribute
     */
    protected String getDisplayNameAttributeName()
    {
        return "displayName";
    }

    /**
     * Returns the standard LDAP name for the 'givenName' attribute. To be overridden on the different LDAP
     * implementations. This method must NOT be qualified with 'final' and/or 'static' to preserve polymorphic behavior.
     * 
     * @return the standard LDAP name for the 'givenName' attribute
     */
    protected String getGivenNameAttributeName()
    {
        return "givenName";
    }

    /**
     * @see pt.digitalis.utils.ldap.ILDAPUtils#getGroupAttribute(java.lang.String, java.lang.String)
     */
    public Object getGroupAttribute(String attributeName, String commonName) throws LDAPOperationException
    {
        return getGroupAttributes(commonName).get(attributeName);
    }

    /**
     * @see pt.digitalis.utils.ldap.ILDAPUtils#getGroupAttributeName()
     */
    public String getGroupAttributeName()
    {
        return "member";
    }

    /**
     * @see pt.digitalis.utils.ldap.ILDAPUtils#getGroupAttributes(java.lang.String)
     */
    public Map<String, Object> getGroupAttributes(String commonName) throws LDAPOperationException
    {
        SearchResult searchResult = getGroupByCommonName(commonName);
        Map<String, Object> converted = convertFromSearchResultToMap(searchResult);

        if (searchResult != null && (searchResult.getObject() instanceof Context))
        {
            try
            {
                ((Context) searchResult.getObject()).close();
            }
            catch (NamingException e)
            {
                e.printStackTrace();
            }
        }

        return converted;
    }

    /**
     * Finds an entity by common name. To be overridden on the different LDAP implementations.
     * 
     * @param cn
     *            the common name
     * @return the result of the search by common name
     * @throws LDAPOperationException
     *             if operation cannot be performed
     */
    protected SearchResult getGroupByCommonName(String cn) throws LDAPOperationException
    {

        if (this.getConfigurations().getGroupMappings().get(cn) != null
                && !"".equals(this.getConfigurations().getGroupMappings().get(cn)))
        {
            if (!groupMappingsFound.contains(cn))
            {
                getLogger().info(
                        "Group Mapping found: Resolving " + cn + " to "
                                + this.getConfigurations().getGroupMappings().get(cn));
                groupMappingsFound.add(cn);
            }

            cn = this.getConfigurations().getGroupMappings().get(cn);
        }

        // The search string
        String search = "(&" + "(" + getObjectClassName() + "=" + getGroupClassName() + ")(" + getCommonName() + "="
                + cn + "))";

        // Fetch the LDAP context
        SearchResult result = doLDAPSearchFirstReturn(search);

        return result;
    }

    /**
     * Finds a group by distinguished name.
     * 
     * @param distinguishedName
     *            the distinguished name
     * @return the group
     * @throws LDAPOperationException
     *             if operation cannot be performed
     */
    protected LDAPGroup getGroupByDistinguishedName(String distinguishedName) throws LDAPOperationException
    {
        LdapContext ctx = getLDAPContext();
        try
        {

            Attributes attrs = ctx.getAttributes(distinguishedName);

            return convertFromAttributesToLDAPGroup(attrs, distinguishedName);
        }
        catch (NamingException namingException)
        {
            throw new LDAPOperationException("Could not LDAP context!!", namingException);
        }
        finally
        {
            try
            {
                ctx.close();
            }
            catch (NamingException e)
            {
                e.printStackTrace();
            }
        }
    }

    /**
     * Returns the standard LDAP name for the 'group' attribute. To be implemented for the different LDAP technologies.
     * 
     * @return the 'group' attribute name for each implementation
     */
    abstract protected String getGroupClassName();

    /**
     * Returns the group identifier name.
     * 
     * @return the group identifier name
     */
    final protected String getGroupIdentifierName()
    {
        return getObjectClassName() + "=" + getGroupClassName();
    }

    /**
     * Gets the group members.
     * 
     * @param groupCN
     *            the group cn
     * @return the group members
     * @throws LDAPOperationException
     *             the lDAP operation exception
     */
    private Vector<String> getGroupMembers(String groupCN) throws LDAPOperationException
    {
        SearchResult group = null;
        Vector<String> members = new Vector<String>();
        try
        {
            // Find group
            group = getGroupByCommonName(groupCN);
            if (group.getAttributes().get(getGroupAttributeName()) != null)
            {
                // Get the DNs of the group members
                members = processMembers(group.getAttributes().get(getGroupAttributeName()));
            }
        }
        catch (NamingException namingException)
        {
            throw new LDAPOperationException("Could not LDAP context!!", namingException);
        }
        finally
        {
            if (group != null && (group.getObject() instanceof Context))
            {
                try
                {
                    ((Context) group.getObject()).close();
                }
                catch (NamingException e)
                {
                    e.printStackTrace();
                }
            }
        }
        return members;
    }

    /**
     * Returns the implementation dependent name for the LDAP attribute that stores the group's parent group. To be
     * implemented for the different LDAP technologies.
     * 
     * @return the parent group attribute name for each implementation
     */
    abstract public String getGroupParentGroupAttributeName();

    /**
     * Get the attribute key based on a value.
     * 
     * @param attributeValue
     *            the attribute value
     * @return the key
     */
    private String getKey(String attributeValue)
    {
        String key = null;
        Iterator<Entry<String, String>> iterator = getConfigurations().getAttributesMapping().entrySet().iterator();
        while (iterator.hasNext())
        {
            Entry<String, String> entry = iterator.next();

            if (entry.getValue().equalsIgnoreCase(attributeValue))
            {
                key = entry.getKey();
            }
        }
        return key;
    }

    /**
     * Returns the LDAP context to use for search and modification operations.
     * 
     * @return the LDAP context
     * @throws LDAPOperationException
     *             if the context can't be created.
     */
    final protected LdapContext getLDAPContext() throws LDAPOperationException
    {
        try
        {
            return login(getConfigurations().getUserDN(), getConfigurations().getPassword(), false);
        }
        catch (NamingException namingException)
        {
            throw new LDAPOperationException("Could not get LDAP context!", namingException);
        }
    }

    /**
     * Returns the standard LDAP name for the 'e-mail' attribute. To be overridden on the different LDAP
     * implementations. This method must NOT be qualified with 'final' and/or 'static' to preserve polymorphic behavior.
     * 
     * @return the standard LDAP name for the 'e-mail' attribute
     */
    public String getMailAttributeName()
    {
        return "mail";
    }

    /**
     * Returns the standard LDAP name for the 'name' attribute. To be overridden on the different LDAP implementations.
     * This method must NOT be qualified with 'final' and/or 'static' to preserve polymorphic behavior.
     * 
     * @return the standard LDAP name for the 'name' attribute
     */
    public String getNameAttributeName()
    {
        return "name";
    }

    /**
     * @see pt.digitalis.utils.ldap.ILDAPUtils#getNonAvailableValue()
     */
    public String getNonAvailableValue()
    {
        return NON_AVAILABLE;
    }

    /**
     * Returns the 'objectClass' attribute name.
     * 
     * @return the 'objectClass' attribute name
     */
    final protected String getObjectClassName()
    {
        return "objectClass";
    }

    /**
     * Returns the standard LDAP name for the 'password' attribute. To be overridden on the different LDAP
     * implementations. This method must NOT be qualified with 'final' and/or 'static' to preserve polymorphic behavior.
     * 
     * @return the standard LDAP name for the 'member' attribute
     */
    protected String getPasswordAttributeName()
    {
        return "userPassword";
    }

    /**
     * Returns the LDAP context to use for search and modification operations with a secure connection.
     * 
     * @return the LDAP context
     * @throws LDAPOperationException
     *             if the context can't be created.
     */
    final protected LdapContext getSecureLDAPContext() throws LDAPOperationException
    {
        try
        {
            return login(getConfigurations().getUserDN(), getConfigurations().getPassword(), true);
        }
        catch (NamingException namingException)
        {
            throw new LDAPOperationException("Could not get secure connection to LDAP server!", namingException);
        }
    }

    /**
     * Returns the standard LDAP name for the 'surname' attribute. To be overridden on the different LDAP
     * implementations. This method must NOT be qualified with 'final' and/or 'static' to preserve polymorphic behavior.
     * 
     * @return the standard LDAP name for the 'surname' attribute
     */
    protected String getSurnameAttributeName()
    {
        return "sn";
    }

    /**
     * @see pt.digitalis.utils.ldap.ILDAPUtils#getUserAttribute(java.lang.String, java.lang.String)
     */
    public Object getUserAttribute(String attributeName, String loginName) throws LDAPOperationException
    {
        return getUserAttributes(loginName).get(attributeName);
    }

    /**
     * @see pt.digitalis.utils.ldap.ILDAPUtils#getUserAttributes(java.lang.String)
     */
    public Map<String, Object> getUserAttributes(String loginName) throws LDAPOperationException
    {
        SearchResult searchResult = getByLogin(loginName);
        Map<String, Object> converted = convertFromSearchResultToMap(searchResult);

        if (searchResult != null && (searchResult.getObject() instanceof Context))
        {
            try
            {
                ((Context) searchResult.getObject()).close();
            }
            catch (NamingException e)
            {
                e.printStackTrace();
            }
        }

        return converted;
    }

    /**
     * Finds an user by distinguished name.
     * 
     * @param distinguishedName
     *            the distinguished name
     * @return the user
     * @throws LDAPOperationException
     *             if operation cannot be performed
     */
    protected LDAPUser getUserByDistinguishedName(String distinguishedName) throws LDAPOperationException
    {
        try
        {
            LdapContext ctx = getLDAPContext();
            Attributes attrs = ctx.getAttributes(distinguishedName);
            ctx.close();

            return convertFromAttributesToLDAPUser(attrs, distinguishedName);
        }
        catch (NamingException namingException)
        {
            throw new LDAPOperationException("Could not get user with DN=" + distinguishedName + "!", namingException);
        }
    }

    /**
     * Returns the standard LDAP name for the 'user' attribute. To be implemented for the different LDAP technologies.
     * 
     * @return the 'user' attribute name for each implementation
     */
    abstract protected String getUserClassName();

    /**
     * Returns the user identifier name.
     * 
     * @return the user identifier name
     */
    final protected String getUserIdentifierName()
    {
        return getObjectClassName() + "=" + getUserClassName();
    }

    /**
     * Finds a user with a given login name on a given group.
     * 
     * @param userLogin
     *            the user's login name
     * @param groupCN
     *            the group's common name
     * @return the user, if it exists, or null otherwise
     * @throws LDAPOperationException
     *             if the user can't be fetched
     */
    protected SearchResult getUserInGroup(String userLogin, String groupCN) throws LDAPOperationException
    {

        LDAPUser user = findUserByLogin(userLogin);

        if (user != null)
        {
            // The search string
            String search = "(&(" + getCommonName() + "=" + groupCN + ")(" + LDAPGroup.MEMBER_NAME + "="
                    + user.getDistinguishedName() + "))";
            SearchResult result = doLDAPSearchFirstReturn(search);

            return result;
        }
        else
            return null;
    }

    /**
     * @see pt.digitalis.utils.ldap.ILDAPUtils#getUserLoginAttributeName()
     */
    public String getUserLoginAttributeName()
    {
        return "cn";
    }

    /**
     * @see pt.digitalis.utils.ldap.ILDAPUtils#getUserParentGroupAttributeName()
     */
    public String getUserParentGroupAttributeName()
    {
        return "manager";
    }

    /**
     * Group contains attribute.
     * 
     * @param attributeName
     *            the attribute name
     * @param commonName
     *            the common name
     * @return true, if successful
     * @throws LDAPOperationException
     *             the lDAP operation exception
     * @see pt.digitalis.utils.ldap.ILDAPUtils#groupContainsAttribute(java.lang.String, java.lang.String)
     */
    public boolean groupContainsAttribute(String attributeName, String commonName) throws LDAPOperationException
    {
        return getGroupAttributes(commonName).containsKey(attributeName);
    }

    /**
     * @see pt.digitalis.utils.ldap.ILDAPUtils#groupExists(java.lang.String)
     */
    public boolean groupExists(String groupCN) throws LDAPOperationException
    {
        return (findGroupByCommonName(groupCN) != null);
    }

    /**
     * @see pt.digitalis.utils.ldap.ILDAPUtils#isIdentityValid(java.lang.String, java.lang.String)
     */
    public boolean isIdentityValid(String loginName, String suppliedPassword) throws LDAPOperationException
    {
        LdapContext ctx = null;
        try
        {
            ctx = login(findUserByLogin(loginName).getDistinguishedName(), suppliedPassword, false);

        }
        catch (AuthenticationException authenticationException)
        {
            return false;
        } /*
           * If the supplied login name doesn't match a real user's login a NPE will be thrown. Bad login = invalid
           * identity!
           */
        catch (NullPointerException nullPointerException)
        {
            return false;
        }
        catch (NamingException namingException)
        {
            throw new LDAPOperationException(namingException);
        }
        finally
        {
            if (ctx != null)
            {
                try
                {
                    ctx.close();
                }
                catch (NamingException e)
                {
                    throw new LDAPOperationException("Error closing context!", e);
                }
            }
        }

        return true;
    }

    /**
     * @see pt.digitalis.utils.ldap.ILDAPUtils#isReadOnly()
     */
    public boolean isReadOnly()
    {
        return this.getConfigurations().getReadOnly();
    }

    /**
     * @see pt.digitalis.utils.ldap.ILDAPUtils#isUserInGroup(java.lang.String, java.lang.String)
     */
    public boolean isUserInGroup(String groupCN, String userLogin) throws LDAPOperationException
    {
        SearchResult searchResult = getUserInGroup(userLogin, groupCN);
        LDAPUser user = convertFromSearchResultToLDAPUser(searchResult);

        if (searchResult != null && (searchResult.getObject() instanceof Context))
        {
            try
            {
                ((Context) searchResult.getObject()).close();
            }
            catch (NamingException e)
            {
                e.printStackTrace();
            }
        }

        if (user != null)
            return true;

        return false;
    }

    /**
     * Logs an user on the LDAP server. Offers the ability to use a secure connection.
     * 
     * @param loginName
     *            the LDAP user login name
     * @param password
     *            the password associated to the login
     * @param secureConnection
     *            T if the connection is secure, F otherwise
     * @return an LDAP context
     * @throws NamingException
     *             if the context building fails
     */
    private LdapContext login(String loginName, String password, boolean secureConnection) throws NamingException
    {

        // Initialize the environment properties object
        Hashtable<String, String> environmentProperties = new Hashtable<String, String>();

        // Populate the environment properties object
        environmentProperties.put(Context.INITIAL_CONTEXT_FACTORY, "com.sun.jndi.ldap.LdapCtxFactory");
        environmentProperties.put("com.sun.jndi.ldap.connect.pool", "true");
        environmentProperties.put("com.sun.jndi.ldap.connect.pool.maxsize", "100");
        // Workarround for bug: http://www-01.ibm.com/support/docview.wss?uid=swg21516788
        environmentProperties.put("com.sun.jndi.ldap.read.timeout", "0"); // 0 seconds timeout - no wait!
        environmentProperties.put("com.sun.jndi.ldap.connect.timeout", "60000"); // 6 seconds timeout
        environmentProperties.put("java.naming.ldap.version", "3");
        environmentProperties.put(Context.SECURITY_AUTHENTICATION, "simple");
        environmentProperties.put(Context.REFERRAL, "ignore");

        // Set login name
        environmentProperties.put(Context.SECURITY_PRINCIPAL, loginName);
        // Set password
        environmentProperties.put(Context.SECURITY_CREDENTIALS, password);

        // Configure a secure connection if needed
        if (secureConnection)
        {
            environmentProperties.put(Context.PROVIDER_URL, "ldaps://" + getConfigurations().getHostName() + ":"
                    + getConfigurations().getSSLPort());
            environmentProperties.put(Context.SECURITY_PROTOCOL, "ssl");
            environmentProperties.put("java.naming.ldap.factory.socket", "javax.net.ssl.SSLSocketFactory");
            String keystore = System.getProperty("java.home") + "/lib/security/cacerts";
            System.setProperty("javax.net.ssl.trustStore", keystore);
        }
        else
        // Configure a regular connection
        {
            environmentProperties.put(Context.SECURITY_PROTOCOL, "plain");
            environmentProperties.put(Context.PROVIDER_URL, "ldap://" + getConfigurations().getHostName() + ":"
                    + getConfigurations().getPort());
            // Enable connection pooling
        }

        // Return a new LDAP context with the environment properties defined above
        LdapContext ctx = new InitialLdapContext(environmentProperties, null);
        return ctx;
    }

    /**
     * Modifies the passed attributes of an entity with the given DN.
     * 
     * @param distinguishedName
     *            the DN of the entity to modify
     * @param mods
     *            the attributes to modify
     * @param secure
     *            if the connection is secure
     * @throws LDAPOperationException
     *             if the attributes can't be modified
     */
    final protected void modifyAttributes(String distinguishedName, ModificationItem[] mods, boolean secure)
            throws LDAPOperationException
    {
        if (mods.length > 0)
        {
            LdapContext context = null;
            try
            {
                if (secure)
                    context = getSecureLDAPContext();
                else
                    context = getLDAPContext();
                context.modifyAttributes(distinguishedName, mods);

                context.close();
            }
            catch (NamingException namingException)
            {
                throw new LDAPOperationException("Could not modify attributes for LDAP entity with DN: "
                        + distinguishedName + "!", namingException);
            }
            finally
            {
                if (context != null)
                {
                    try
                    {
                        context.close();
                    }
                    catch (NamingException e)
                    {
                        throw new LDAPOperationException("Error closing context!", e);
                    }
                }
            }
        }
    }

    /**
     * Process members.
     * 
     * @param attribute
     *            the member attribute
     * @return a vector with the members DNs
     * @throws NamingException
     *             the naming exception
     */
    private Vector<String> processMembers(Attribute attribute) throws NamingException
    {
        Vector<String> members = new Vector<String>();

        NamingEnumeration<?> nameEnum = attribute.getAll();
        try
        {
            while (nameEnum.hasMore())
            {
                String nameEnumValue = (String) nameEnum.next();

                if (StringUtils.isNotBlank(nameEnumValue))
                {
                    members.add(nameEnumValue);
                }
            }
        }
        finally
        {
            nameEnum.close();
        }
        return members;
    }

    /**
     * Removes a given attribute's on a given LDAP entity.
     * 
     * @param distinguishedName
     *            the entity's distinguished name
     * @param attributeName
     *            the attribute name
     * @throws LDAPOperationException
     *             if the attribute value can't be changed
     */
    final private void removeAttribute(String distinguishedName, String attributeName) throws LDAPOperationException
    {
        ModificationItem[] mods = new ModificationItem[1];

        mods[0] = new ModificationItem(LdapContext.REMOVE_ATTRIBUTE, new BasicAttribute(attributeName));

        modifyAttributes(distinguishedName, mods, false);
    }

    /**
     * Removes a given attribute's on a given LDAP entity.
     * 
     * @param distinguishedName
     *            the entity's distinguished name
     * @param attributeName
     *            the attribute name
     * @param value
     *            the attribute value
     * @throws LDAPOperationException
     *             if the attribute value can't be changed
     */
    final private void removeAttributeWithValue(String distinguishedName, String attributeName, String value)
            throws LDAPOperationException
    {
        ModificationItem[] mods = new ModificationItem[1];

        mods[0] = new ModificationItem(LdapContext.REMOVE_ATTRIBUTE, new BasicAttribute(attributeName, value));

        try
        {
            modifyAttributes(distinguishedName, mods, false);
        }
        catch (LDAPOperationException ldapOperationException)
        {
            // If the attribute is mandatory it must be set to null
            if (ldapOperationException.getCause().getClass().equals(SchemaViolationException.class))
            {
                mods[0] = new ModificationItem(LdapContext.REPLACE_ATTRIBUTE, new BasicAttribute(attributeName, ""));
                modifyAttributes(distinguishedName, mods, false);
            }
            else
                throw new LDAPOperationException("Could not remove attribute " + attributeName
                        + " from entity with DN=" + distinguishedName + "!", ldapOperationException);
        }
    }

    /**
     * @see pt.digitalis.utils.ldap.ILDAPUtils#removeGroup(java.lang.String)
     */
    public void removeGroup(String groupCN) throws LDAPOperationException
    {
        if (this.isReadOnly())
            throw new LDAPOperationReadOnlyException();

        LDAPGroup groupToRemove = findGroupByCommonName(groupCN);

        if (groupToRemove != null)
        {
            destroySubcontext(groupToRemove.getDistinguishedName());
        }
    }

    /**
     * @see pt.digitalis.utils.ldap.ILDAPUtils#removeGroupAttribute(java.lang.String, java.lang.String)
     */
    public void removeGroupAttribute(String commonName, String attributeName) throws LDAPOperationException
    {
        if (this.isReadOnly())
            throw new LDAPOperationReadOnlyException();

        removeAttribute(findGroupByCommonName(commonName).getDistinguishedName(), attributeName);
    }

    /**
     * @see pt.digitalis.utils.ldap.ILDAPUtils#removeUser(java.lang.String)
     */
    public void removeUser(String loginName) throws LDAPOperationException
    {
        if (this.isReadOnly())
            throw new LDAPOperationReadOnlyException();

        LDAPUser userToRemove = findUserByLogin(loginName);

        if (userToRemove != null)
        {
            destroySubcontext(userToRemove.getDistinguishedName());
        }
    }

    /**
     * @see pt.digitalis.utils.ldap.ILDAPUtils#removeUserAttribute(java.lang.String, java.lang.String)
     */
    public void removeUserAttribute(String loginName, String attributeName) throws LDAPOperationException
    {
        if (this.isReadOnly())
            throw new LDAPOperationReadOnlyException();

        LDAPUser user = findUserByLogin(loginName);

        if (user != null)
        {
            user.removeParameter(attributeName);
            this.updateUser(user, user.getLoginName());
        }
    }

    /**
     * @see pt.digitalis.utils.ldap.ILDAPUtils#removeUserFromGroup(java.lang.String, java.lang.String)
     */
    public void removeUserFromGroup(String groupCN, String userLogin) throws LDAPOperationException
    {
        if (this.isReadOnly())
            throw new LDAPOperationReadOnlyException();

        LDAPGroup group = findGroupByCommonName(groupCN);
        LDAPUser user = findUserByLogin(userLogin);

        if (group != null && user != null)
            this.removeAttributeWithValue(group.getDistinguishedName(), LDAPGroup.MEMBER_NAME,
                    user.getDistinguishedName());
    }

    /**
     * Replace a given attribute's value on a given LDAP entity.
     * 
     * @param distinguishedName
     *            the entity's distinguished name
     * @param attributeName
     *            the attribute name
     * @param value
     *            the attribute value
     * @throws LDAPOperationException
     *             if the attribute value can't be changed
     */
    final private void replaceAttribute(String distinguishedName, String attributeName, String value)
            throws LDAPOperationException
    {
        ModificationItem[] mods = new ModificationItem[1];

        mods[0] = new ModificationItem(LdapContext.REPLACE_ATTRIBUTE, new BasicAttribute(attributeName,
                value.toString()));

        modifyAttributes(distinguishedName, mods, false);
    }

    /**
     * @see pt.digitalis.utils.ldap.ILDAPUtils#resetConfigurations()
     */
    public void resetConfigurations()
    {
        this.ldapConfigurations = null;
    }

    /**
     * @see pt.digitalis.utils.ldap.ILDAPUtils#setGroupAttribute(java.lang.String, java.lang.String, java.lang.Object)
     */
    public void setGroupAttribute(String commonName, String attributeName, Object value) throws LDAPOperationException
    {
        if (this.isReadOnly())
            throw new LDAPOperationReadOnlyException();

        LDAPGroup group = findGroupByCommonName(commonName);
        if (group == null)
        {
            throw new LDAPOperationException("The group with the common name \"" + commonName + "\" doesn't exists!");
        }

        /*
         * The Member attribute's must always be added, never replaced. In the other cases the attribute value must be
         * replaced
         */
        if (LDAPGroup.MEMBER_NAME.equals(attributeName))
        {
            this.addAttribute(group.getDistinguishedName(), attributeName, value.toString());
        }
        else
        {
            this.replaceAttribute(group.getDistinguishedName(), attributeName, value.toString());
        }
    }

    /**
     * @see pt.digitalis.utils.ldap.ILDAPUtils#setLogger(pt.digitalis.log.ILogWrapper)
     */
    public void setLogger(ILogWrapper logger)
    {
        AbstractLDAPUtils.logger = logger;
    }

    /**
     * @see pt.digitalis.utils.ldap.ILDAPUtils#setUserAttribute(java.lang.String, java.lang.String, java.lang.Object)
     */
    public void setUserAttribute(String loginName, String attributeName, Object value) throws LDAPOperationException
    {
        if (this.isReadOnly())
            throw new LDAPOperationReadOnlyException();

        LDAPUser user = findUserByLogin(loginName);

        if (user != null)
        {
            user.setParameter(attributeName, value.toString());
            this.updateUser(user, user.getLoginName());
        }
        else
        {
            throw new LDAPOperationException("Could not store parameter on user attribute for user with login "
                    + loginName
                    + "! User was not found on LDAP server...\nCheck user was created before and exists on server.");
        }
    }

    /**
     * @see pt.digitalis.utils.ldap.ILDAPUtils#updateGroup(pt.digitalis.utils.ldap.LDAPGroup, java.lang.String)
     *      Implementation note: the attribute 'name' is not eligible for modification (for AD, check other impls).
     */
    public void updateGroup(LDAPGroup groupToUpdate, String groupCN) throws LDAPOperationException
    {
        if (this.isReadOnly())
            throw new LDAPOperationReadOnlyException();

        // Retrieve old user data to get the old data
        LDAPGroup existingGroup = findGroupByCommonName(groupCN);

        ArrayList<ModificationItem> itens = new ArrayList<ModificationItem>();

        // 'description'
        if (groupToUpdate.getDescription() != null
                && !groupToUpdate.getDescription().equals(existingGroup.getDescription()))
            itens.add(new ModificationItem(LdapContext.REPLACE_ATTRIBUTE, new BasicAttribute(
                    getDescriptionAttributeName(), groupToUpdate.getDescription())));
        // 'parentGroup'
        if (groupToUpdate.getParentGroupDN() != null
                && !groupToUpdate.getParentGroupDN().equals(existingGroup.getParentGroupDN()))
            itens.add(new ModificationItem(LdapContext.REPLACE_ATTRIBUTE, new BasicAttribute(
                    getGroupParentGroupAttributeName(), groupToUpdate.getParentGroupDN())));

        if (itens.size() > 0)
        {
            ModificationItem mods[] = new ModificationItem[itens.size()];
            for (int i = 0; i < itens.size(); i++)
            {
                mods[i] = itens.get(i);
            }

            modifyAttributes(existingGroup.getDistinguishedName(), mods, false);
        }

        /* Group modification must be made after all other attributes since it can cause DN changes. */
        // Main group
        if (ldapConfigurations.getAllowDistinguishedNameModifications()
                && (groupToUpdate.getParentGroupDN() != null && !groupToUpdate.getParentGroupDN().equals(
                        existingGroup.getParentGroupDN())) || groupToUpdate.getCommonName() != null
                && !groupToUpdate.getCommonName().equals(existingGroup.getCommonName()))
        {

            // Get common name from user to change
            String commonName = groupToUpdate.getCommonName();

            // Get the parent group name
            String parentGroup = groupToUpdate.getParentGroupDN();

            // Make sure that new DN can be calculated even if some attributes have no value
            if (commonName == null)
                commonName = existingGroup.getCommonName();

            if (parentGroup == null)
                parentGroup = getGroupByDistinguishedName(existingGroup.getParentGroupDN()).getDistinguishedName();

            String newDN = calculateDistinguishedName(commonName, parentGroup);

            try
            {
                if (!existingGroup.getDistinguishedName().equals(newDN))
                {
                    LdapContext ctx = getLDAPContext();
                    try
                    {
                        ctx.rename(existingGroup.getDistinguishedName(), newDN);
                    }
                    finally
                    {
                        ctx.close();
                    }
                }
            }
            catch (LDAPOperationException ldapOperationException)
            {
                throw ldapOperationException;
            }
            catch (NamingException namingException)
            {
                throw new LDAPOperationException("Could not update group's parent group...", namingException);
            }
        }
    }

    /**
     * @see pt.digitalis.utils.ldap.ILDAPUtils#updateUser(pt.digitalis.utils.ldap.LDAPUser, java.lang.String)
     */
    public void updateUser(LDAPUser userToUpdate, String userLogin) throws LDAPOperationException
    {
        if (this.isReadOnly())
            throw new LDAPOperationReadOnlyException();

        // Retrieve old user data to get the old data
        LDAPUser existingUser = findUserByLogin(userLogin, false);

        if (existingUser != null)
        {
            ArrayList<ModificationItem> items = new ArrayList<ModificationItem>();

            // 'displayName'
            if (userToUpdate.getDisplayName() != null
                    && !userToUpdate.getDisplayName().equalsIgnoreCase(existingUser.getDisplayName()))
            {
                items.add(new ModificationItem(LdapContext.REPLACE_ATTRIBUTE, new BasicAttribute(
                        getDisplayNameAttributeName(), userToUpdate.getDisplayName())));
            }

            // 'e-mail'
            if (userToUpdate.getEmail() != null && !userToUpdate.getEmail().equals(existingUser.getEmail()))
                items.add(new ModificationItem(LdapContext.REPLACE_ATTRIBUTE, new BasicAttribute(
                        getMailAttributeName(), userToUpdate.getEmail())));
            // 'givenName'
            if (userToUpdate.getGivenName() != null && !userToUpdate.getGivenName().equals(existingUser.getGivenName()))
                items.add(new ModificationItem(LdapContext.REPLACE_ATTRIBUTE, new BasicAttribute(
                        getGivenNameAttributeName(), userToUpdate.getGivenName())));

            /*
             * Password change method is implementation dependent so it's not done through ModificationItem. Password
             * does not come from LDAP server, so it's ALWAYS changed!!! Done before login change to assure that the
             * former login name still works.
             */
            if (userToUpdate.getPassword() != null)
                changePassword(userLogin, userToUpdate.getPassword());

            // '"login"'
            if (userToUpdate.getLoginName() != null && !userToUpdate.getLoginName().equals(existingUser.getLoginName()))
                items.add(new ModificationItem(LdapContext.REPLACE_ATTRIBUTE, new BasicAttribute(
                        getUserLoginAttributeName(), userToUpdate.getLoginName())));

            // '"mainGroup"'
            if (userToUpdate.getParentGroupDN() != null
                    && !userToUpdate.getParentGroupDN().equals(existingUser.getParentGroupDN()))
                items.add(new ModificationItem(LdapContext.REPLACE_ATTRIBUTE, new BasicAttribute(
                        getUserParentGroupAttributeName(), userToUpdate.getParentGroupDN())));

            // Parameters
            Map<String, String> bulkParameters = new HashMap<String, String>();
            for (String parameterName: userToUpdate.getParameters().keySet())
            {
                if (getConfigurations().getAttributesMapping().containsKey(parameterName))
                {
                    items.add(new ModificationItem(LdapContext.REPLACE_ATTRIBUTE, new BasicAttribute(
                            getConfigurations().getAttributesMapping().get(parameterName), userToUpdate
                                    .getParameter(parameterName))));
                }
                else
                {
                    if (existingUser.getParameter(parameterName) != null)
                    {
                        if (!getUnchangeableLDAPAttributes().contains(parameterName.toUpperCase())
                                && !userToUpdate.getParameter(parameterName).equalsIgnoreCase(
                                        existingUser.getParameter(parameterName)))
                            items.add(new ModificationItem(LdapContext.REPLACE_ATTRIBUTE, new BasicAttribute(
                                    parameterName, userToUpdate.getParameter(parameterName))));
                    }
                    else
                    {
                        bulkParameters.put(parameterName, userToUpdate.getParameter(parameterName));
                    }
                }
            }

            for (String parameterName: userToUpdate.getParametersToRemove())
            {
                if (getConfigurations().getAttributesMapping().containsKey(parameterName))
                {
                    items.add(new ModificationItem(LdapContext.REMOVE_ATTRIBUTE, new BasicAttribute(getConfigurations()
                            .getAttributesMapping().get(parameterName))));
                }
                else
                {
                    bulkParameters.remove(parameterName);
                }
            }

            // The value of the Bulk parameters to commit
            StringBuilder bulkParameterAttributeValue = new StringBuilder();
            for (Entry<String, String> entry: bulkParameters.entrySet())
            {
                bulkParameterAttributeValue.append(entry.getKey() + "=" + entry.getValue() + ";");
            }

            if (bulkParameterAttributeValue.length() == 0)
            {
                /* LDAP doesn't like empty strings on REPLACE_ATTRIBUTE */
                bulkParameterAttributeValue.append(" ");
            }

            items.add(new ModificationItem(LdapContext.REPLACE_ATTRIBUTE, new BasicAttribute(getConfigurations()
                    .getBulkParametersAttributeName(), bulkParameterAttributeValue.toString())));

            // Persist changes
            if (items.size() > 0)
            {
                ModificationItem mods[] = new ModificationItem[items.size()];
                for (int i = 0; i < items.size(); i++)
                {
                    mods[i] = items.get(i);
                }

                modifyAttributes(existingUser.getDistinguishedName(), mods, false);
            }

            /* Group modification must be made after all other attributes since it can cause DN changes. */
            // Main group
            if (ldapConfigurations.getAllowDistinguishedNameModifications() && userToUpdate.getParentGroupDN() != null
                    && !userToUpdate.getParentGroupDN().equals(existingUser.getParentGroupDN()))
            {

                // Get login name from user to change
                String loginName = userToUpdate.getLoginName();

                // If only the group changes, get login from existing user data
                if (loginName == null)
                    loginName = existingUser.getLoginName();

                String newDN = calculateDistinguishedName(loginName, userToUpdate.getParentGroupDN());
                try
                {
                    if (!existingUser.getDistinguishedName().equals(newDN))
                    {
                        LdapContext ctx = getLDAPContext();
                        try
                        {
                            ctx.rename(existingUser.getDistinguishedName(), newDN);
                        }
                        finally
                        {
                            ctx.close();
                        }
                    }
                }
                catch (LDAPOperationException ldapOperationException)
                {
                    throw ldapOperationException;
                }
                catch (NamingException namingException)
                {
                    throw new LDAPOperationException("Could not update user's main group...", namingException);
                }
            }
        }
    }

    /**
     * @see pt.digitalis.utils.ldap.ILDAPUtils#userContainsAttribute(java.lang.String, java.lang.String)
     */
    public boolean userContainsAttribute(String id, String loginName) throws LDAPOperationException
    {
        return getUserAttributes(loginName).containsKey(id);
    }

    /**
     * @see pt.digitalis.utils.ldap.ILDAPUtils#userExists(java.lang.String)
     */
    public boolean userExists(String loginName) throws LDAPOperationException
    {
        return (findUserByLogin(loginName) != null);
    }

}
