/**
 * 2007, Digitalis Informatica. All rights reserved. Distribuicao e Gestao de Informatica, Lda. Estrada de Paco de Arcos
 * num.9 - Piso -1 2780-666 Paco de Arcos Telefone: (351) 21 4408990 Fax: (351) 21 4408999 http://www.digitalis.pt
 */
package pt.digitalis.utils.ldap;

import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.naming.NamingException;

import pt.digitalis.log.ILogWrapper;
import pt.digitalis.utils.ldap.exception.LDAPOperationException;

/**
 * Defines a set of LDAP operations.
 * 
 * @author Rodrigo Gonalves <a href="mailto:rgoncalves@digitalis.pt">rgoncalves@digitalis.pt</a><br/>
 * @author Luis Pinto <a href="lpinto@digitalis.pt">lpinto@digitalis.pt</a><br/>
 * @author Pedro Viegas <a href="mailto:pviegas@digitalis.pt">pviegas@digitalis.pt</a>
 * @author Fbio Souto <a href="mailto:fsouto@digitalis.pt">fsouto@digitalis.pt</a><br/>
 * @created Mar 26, 2008
 */
public interface ILDAPUtils {

    /**
     * Adds a new group to the LDAP tree.
     * 
     * @param newGroup
     *            the group to add
     * @throws LDAPOperationException
     *             if the group can't be created
     */
    public void addGroup(LDAPGroup newGroup) throws LDAPOperationException;

    /* --- User operations --- */

    /**
     * Adds an attribute to a given group.
     * 
     * @param commonName
     *            the group's common name
     * @param attributeName
     *            the attribute name
     * @param value
     *            the attribute value
     * @throws LDAPOperationException
     */
    public void addGroupAttribute(String commonName, String attributeName, Object value) throws LDAPOperationException;

    /**
     * Adds a new user to the LDAP tree.
     * 
     * @param newUser
     *            the new user to add to the LDAP server
     * @throws LDAPOperationException
     *             if the user can't be created
     */
    public void addUser(LDAPUser newUser) throws LDAPOperationException;

    /**
     * Adds an attribute to a given user.
     * 
     * @param loginName
     *            the user login name
     * @param attributeName
     *            the attribute name
     * @param value
     *            the attribute value
     * @throws LDAPOperationException
     *             if the attribute can't be added
     */
    public void addUserAttribute(String loginName, String attributeName, Object value) throws LDAPOperationException;

    /**
     * Adds a user from a given group.
     * 
     * @param groupCN
     *            the CN of the group
     * @param userLogin
     *            the login name of the user to add
     * @throws LDAPOperationException
     *             if the user groups can't be added
     */
    public void addUserToGroup(String groupCN, String userLogin) throws LDAPOperationException;

    /**
     * Changes the password for a given user.
     * 
     * @param loginName
     *            the login name of the user to change the password
     * @param newPassword
     *            the new password to set
     * @throws LDAPOperationException
     *             if the operation cannot be executed
     */
    public void changePassword(String loginName, String newPassword) throws LDAPOperationException;

    /**
     * Counts all groups in the LDAP directory
     * 
     * @return the number of groups in the LDAP directory
     * @throws NamingException
     * @throws LDAPOperationException
     */
    public int countAllGroups() throws NamingException, LDAPOperationException;

    /**
     * Counts all groups in the LDAP directory
     * 
     * @param commonNameDistinct
     *            if True makes a distinct count by common name
     * @return the number of groups in the LDAP directory
     * @throws NamingException
     * @throws LDAPOperationException
     */
    public int countAllGroups(boolean commonNameDistinct) throws NamingException, LDAPOperationException;

    /**
     * Counts all groups of a user
     * 
     * @param loginName
     *            The user login name
     * @return the number of users in the LDAP directory
     * @throws NamingException
     * @throws LDAPOperationException
     */
    public int countAllGroupsOfUser(String loginName) throws NamingException, LDAPOperationException;

    /**
     * Counts all users in the LDAP directory
     * 
     * @return the number of users in the LDAP directory
     * @throws NamingException
     * @throws LDAPOperationException
     */
    public int countAllUsers() throws NamingException, LDAPOperationException;

    /**
     * Counts all users of a given group.
     * 
     * @param groupId
     *            The group identifier
     * @return The number of users of this group.
     * @throws LDAPOperationException
     */
    public int countAllUsers(String groupId) throws LDAPOperationException;

    /**
     * Count users in the LDAP directory which match the provided attributes
     * 
     * @param attributes
     *            The attributes to be considered in the search
     * @return The number of users that match the provided attributes
     * @throws LDAPOperationException
     *             if the operation can't be executed
     */
    public int countUsers(Map<String, String> attributes) throws LDAPOperationException;

    /**
     * Returns the list of all groups.
     * 
     * @return the list of all the groups
     * @throws LDAPOperationException
     *             if the operation can't be executed
     */
    public Set<LDAPGroup> findAllGroups() throws LDAPOperationException;

    /**
     * Returns the list of all users.
     * 
     * @return the list of all the users
     * @throws LDAPOperationException
     *             if the operation can't be executed
     */
    public Set<LDAPUser> findAllUsers() throws LDAPOperationException;

    /**
     * Returns the group with a given common name.
     * 
     * @param cn
     *            the group's common name
     * @return the LDAPGroup with the given common name
     * @throws LDAPOperationException
     *             if the operation cannot be executed
     */
    public LDAPGroup findGroupByCommonName(String cn) throws LDAPOperationException;

    /**
     * Returns the group with a given distinguished name.
     * 
     * @param dn
     *            the group's distinguished name
     * @return the LDAPGroup with the given distinguished name
     * @throws LDAPOperationException
     *             if the operation cannot be executed
     */
    public LDAPGroup findGroupByDistinguishedName(String dn) throws LDAPOperationException;

    /**
     * Finds a subset of all the groups, according to the parameters
     * 
     * @param rowsPerPage
     *            The number of rows per page
     * @param pageToReturn
     *            The page number to return
     * @return The list containing the specified subset of groups
     * @throws LDAPOperationException
     *             if the operation cannot be executed.
     */
    public Set<LDAPGroup> findGroups(int rowsPerPage, int pageToReturn) throws LDAPOperationException;

    /**
     * Returns the list of a given user's groups.
     * 
     * @param loginName
     *            the user's login name
     * @return the list of all the groups that the user belongs to
     * @throws LDAPOperationException
     *             if the operation can't be executed
     */
    public Set<LDAPGroup> findGroupsOfUser(String loginName) throws LDAPOperationException;

    /**
     * Returns the list of a given user's groups, with pagination (optional). Only includes the user profile if no
     * pagination is not requested. If the user profile already exists in membership form, the profile will be included
     * in the result list is accounted for.
     * 
     * @param loginName
     *            the user's login name
     * @param rowsPerPage
     *            The number of results per page
     * @param pageToReturn
     *            The page number to return
     * @return the list of all the groups that the user belongs to
     * @throws LDAPOperationException
     *             if the operation can't be executed
     */
    public Set<LDAPGroup> findGroupsOfUserPagination(String loginName, Integer rowsPerPage, Integer pageToReturn)
            throws LDAPOperationException;

    /**
     * Returns the user with a given distinguished name.
     * 
     * @param dn
     *            the user's login name
     * @return the LDAPUser with the given login name
     * @throws LDAPOperationException
     *             if the operation cannot be executed
     */
    public LDAPUser findUserByDistinguishedName(String dn) throws LDAPOperationException;

    /* --- Group operations --- */

    /**
     * Returns the user with a given login name.
     * 
     * @param loginName
     *            the user's login name
     * @return the LDAPUser with the given login name
     * @throws LDAPOperationException
     *             if the operation cannot be executed
     */
    public LDAPUser findUserByLogin(String loginName) throws LDAPOperationException;

    /**
     * Finds a subset of all the users, according to the parameters
     * 
     * @param rowsPerPage
     *            The number of rows per page
     * @param pageToReturn
     *            The page number to return
     * @return The list containing the specified subset of groups
     * @throws LDAPOperationException
     *             if the operation cannot be executed
     */
    public Set<LDAPUser> findUsers(int rowsPerPage, int pageToReturn) throws LDAPOperationException;

    /**
     * Finds a set of LDAP users that match any of the specified criteria
     * 
     * @param attributes
     *            A map of (attribute, attribute value) pairs
     * @return A set of LDAP users that match all of the specified criteria
     * @throws LDAPOperationException
     *             if the operation cannot be executed
     */
    public Set<LDAPUser> findUsersByAnyAttribute(Map<String, String> attributes) throws LDAPOperationException;

    /**
     * Returns the list of users with a given attribute value.
     * 
     * @param attribute
     *            the attribute to search
     * @param value
     *            the value for the attribute
     * @return the LDAPUser with the given login name
     * @throws LDAPOperationException
     *             if the operation cannot be executed
     */
    public Set<LDAPUser> findUsersByAttribute(String attribute, String value) throws LDAPOperationException;

    /**
     * Finds a set of LDAP users that match all of the specified criteria
     * 
     * @param attributes
     *            A map of (attribute, attribute value) pairs
     * @return A set of LDAP users that match all of the specified criteria
     * @throws LDAPOperationException
     *             if the operation cannot be executed
     */
    public Set<LDAPUser> findUsersByAttributes(Map<String, String> attributes) throws LDAPOperationException;

    /**
     * Finds a set of LDAP users that match all of the specified criteria
     * 
     * @param attributes
     *            A map of (attribute, attribute value) pairs
     * @param rowsPerPage
     *            The number of results per page
     * @param pageToReturn
     *            The number of the page to return
     * @return A set of LDAP users that match all of the specified criteria
     * @throws LDAPOperationException
     *             if the operation cannot be executed
     */
    public Set<LDAPUser> findUsersByAttributes(Map<String, String> attributes, Integer rowsPerPage, Integer pageToReturn)
            throws LDAPOperationException;

    /**
     * Returns the list of users with a given email attribute value.
     * 
     * @param value
     *            the value for the attribute email
     * @return the LDAPUser with the given login name
     * @throws LDAPOperationException
     *             if the operation cannot be executed
     */
    public Set<LDAPUser> findUsersByEmail(String value) throws LDAPOperationException;

    /**
     * Checks if a given user belongs to a given group.
     * 
     * @param groupCN
     *            the group's CN
     * @return a map of the group's users (<K=login name, V=user>)
     * @throws LDAPOperationException
     *             if the operation can't be executed
     */
    public Map<String, LDAPUser> findUsersInGroup(String groupCN) throws LDAPOperationException;

    /**
     * Returns the list of groups that belong to a parent group given its common name.
     * 
     * @param commonName
     *            the parent group's common name
     * @return the list of all the groups that the user belongs to the parent group
     * @throws LDAPOperationException
     *             if the operation can't be executed
     */
    public Set<LDAPGroup> getChildGroupsByCN(String commonName) throws LDAPOperationException;

    /**
     * Returns the list of groups that belong to a parent group given its distinguished name.
     * 
     * @param distinguishedName
     *            the parent group's distinguished name
     * @return the list of all the groups that the user belongs to the parent group
     * @throws LDAPOperationException
     *             if the operation can't be executed
     */
    public Set<LDAPGroup> getChildGroupsByDN(String distinguishedName) throws LDAPOperationException;

    /**
     * Inspector for the LDAP configurations object.
     * 
     * @return the LDAP configurations object
     */
    public LDAPConfigurations getConfigurations();

    /**
     * Returns a given attribute or null of it doesn't exists.
     * 
     * @param attributeName
     *            the attribute name
     * @param commonName
     *            the group common name
     * @return the attribute with the given id
     * @throws LDAPOperationException
     *             if the user attribute can't be fetched
     */
    public Object getGroupAttribute(String attributeName, String commonName) throws LDAPOperationException;

    /**
     * Gets the name attribute name
     * 
     * @return The string containing the attribute name.
     */
    public String getGroupAttributeName();

    /**
     * Returns the attribute map.
     * 
     * @param commonName
     *            the group common name
     * @return the attribute map
     * @throws LDAPOperationException
     *             if the user attributes can't be fetched
     */
    public Map<String, Object> getGroupAttributes(String commonName) throws LDAPOperationException;

    /**
     * Gets the mail attribute name
     * 
     * @return The string containing the attribute name.
     */
    public String getMailAttributeName();

    /**
     * Gets the name attribute name
     * 
     * @return The string containing the attribute name.
     */
    public String getNameAttributeName();

    /* --- Attribute operations --- */

    /**
     * Get the Non Available Message.
     * 
     * @return the Non Available message.
     */
    public String getNonAvailableValue();

    /**
     * Get the list of the LDAP unchangeable attributes
     * 
     * @return the list of attributes
     */
    public List<String> getUnchangeableLDAPAttributes();

    /**
     * Returns a given attribute or null of it doesn't exists.
     * 
     * @param attributeName
     *            the attribute name
     * @param loginName
     *            the user login name
     * @return the attribute with the given id
     * @throws LDAPOperationException
     *             if the user attribute can't be fetched
     */
    public Object getUserAttribute(String attributeName, String loginName) throws LDAPOperationException;

    /**
     * Returns the attribute map.
     * 
     * @param loginName
     *            the user login name
     * @return the attribute map
     * @throws LDAPOperationException
     *             if the user attributes can't be fetched
     */
    public Map<String, Object> getUserAttributes(String loginName) throws LDAPOperationException;

    /**
     * Gets the user login attribute name
     * 
     * @return The string containing the attribute name.
     */
    public String getUserLoginAttributeName();

    /**
     * Returns the parent group attribute name.
     * 
     * @return A string containing the parent group attribute name
     */
    public String getUserParentGroupAttributeName();

    /**
     * Checks if a given attribute belongs to the entity.
     * 
     * @param attributeName
     *            the attribute name
     * @param commonName
     *            the group common name
     * @return T if the entity has the attribute, F otherwise
     * @throws LDAPOperationException
     *             if the user attribute can't be fetched
     */
    public boolean groupContainsAttribute(String attributeName, String commonName) throws LDAPOperationException;

    /**
     * Checks if a given group exists on the LDAP tree.
     * 
     * @param groupCN
     *            the group's CN
     * @return T if the group is found on the LDAP tree, F otherwise
     * @throws LDAPOperationException
     *             if the operation can't be executed
     */
    public boolean groupExists(String groupCN) throws LDAPOperationException;

    /**
     * Checks if the supplied password matches the one defined for a user with a given ID.
     * 
     * @param loginName
     *            the user ID
     * @param suppliedPassword
     *            the supplied user password
     * @return T if user if its identity is valid, F otherwise
     * @throws LDAPOperationException
     *             if the operation cannot be executed
     */
    public boolean isIdentityValid(String loginName, String suppliedPassword) throws LDAPOperationException;

    /**
     * Get the ldap readOnly configuration value.
     * 
     * @return true or false
     */
    public boolean isReadOnly();

    /**
     * Checks if a given user belongs to a given group.
     * 
     * @param groupCN
     *            the group's CN
     * @param userLogin
     *            the user's login name
     * @return T if the user belongs to the given group, F otherwise
     * @throws LDAPOperationException
     *             if the operation can't be executed
     */
    public boolean isUserInGroup(String groupCN, String userLogin) throws LDAPOperationException;

    /**
     * Removes an existing group from the LDAP tree.
     * 
     * @param groupCN
     *            the group to add
     * @throws LDAPOperationException
     *             if the group can't be removed
     */
    public void removeGroup(String groupCN) throws LDAPOperationException;

    /**
     * Removes an attribute from a given group.
     * 
     * @param commonName
     *            the group's common name
     * @param attributeName
     *            the attribute name
     * @throws LDAPOperationException
     */
    public void removeGroupAttribute(String commonName, String attributeName) throws LDAPOperationException;

    /**
     * Removes an existing user from the LDAP tree.
     * 
     * @param loginName
     *            the id of the user to remove
     * @throws LDAPOperationException
     *             if the user can't be removed
     */
    public void removeUser(String loginName) throws LDAPOperationException;

    /**
     * Removes an attribute from a given user.
     * 
     * @param loginName
     *            the user login name
     * @param attributeName
     *            the attribute name
     * @throws LDAPOperationException
     *             if the attribute can't be added
     */
    public void removeUserAttribute(String loginName, String attributeName) throws LDAPOperationException;

    /**
     * Removes a user from a given group.
     * 
     * @param groupCN
     *            the CN of the group
     * @param userLogin
     *            the login name of the user to remove
     * @throws LDAPOperationException
     *             if the user groups can't be removed
     */
    public void removeUserFromGroup(String groupCN, String userLogin) throws LDAPOperationException;

    /**
     * Resets the LDAP configurations. Forces the configurations to be re-read. VALIDATE: Viegas: This method is helpful
     * to change the configurations in runtime. For example, a management interface user might choose to change the root
     * username or password. The changes must be reflected immediately so the user can keep it's privileges. The
     * management interface should call this method to reinitialize the configurations.
     */
    public void resetConfigurations();

    /**
     * Sets a given attribute value.
     * 
     * @param loginName
     *            the user login name
     * @param attributeName
     *            the attribute name
     * @param value
     *            the attribute value
     * @throws LDAPOperationException
     *             if the attribute can't be set
     */
    public void setGroupAttribute(String loginName, String attributeName, Object value) throws LDAPOperationException;

    /**
     * Set the API logger
     * 
     * @param logger
     *            the new logger value to set
     */
    public void setLogger(ILogWrapper logger);

    /**
     * Sets the a given attribute value.
     * 
     * @param loginName
     *            the user login name
     * @param attributeName
     *            the attribute name
     * @param value
     *            the attribute value
     * @throws LDAPOperationException
     *             if the attribute can't be set
     */
    public void setUserAttribute(String loginName, String attributeName, Object value) throws LDAPOperationException;

    /**
     * Updates an existing group on the LDAP tree. Does nothing if the user doesn't exist.
     * 
     * @param groupToUpdate
     *            the user to update
     * @param groupID
     *            the group ID
     * @throws LDAPOperationException
     *             if the group cannot be updated
     */
    public void updateGroup(LDAPGroup groupToUpdate, String groupID) throws LDAPOperationException;

    /**
     * Updates an existing user on the LDAP tree. Does nothing if the user doesn't exist.
     * 
     * @param userToUpdate
     *            the user to update
     * @param userLogin
     *            the user's original login
     * @throws LDAPOperationException
     *             if the operation cannot be executed
     */
    public void updateUser(LDAPUser userToUpdate, String userLogin) throws LDAPOperationException;

    /**
     * Checks if a given attribute belongs to the entity.
     * 
     * @param attributeName
     *            the attribute name
     * @param loginName
     *            the user login name
     * @return T if the entity has the attribute, F otherwise
     * @throws LDAPOperationException
     *             if the user attribute can't be fetched
     */
    public boolean userContainsAttribute(String attributeName, String loginName) throws LDAPOperationException;

    /**
     * Checks if a user exists on the LDAP tree.
     * 
     * @param loginName
     *            the ID of the user
     * @return T if the user exists, F otherwise
     * @throws LDAPOperationException
     *             if the operation cannot be executed
     */
    public boolean userExists(String loginName) throws LDAPOperationException;

}
