/**
 * 2007, Digitalis Informatica. All rights reserved. Distribuicao e Gestao de Informatica, Lda. Estrada de Paco de Arcos
 * num.9 - Piso -1 2780-666 Paco de Arcos Telefone: (351) 21 4408990 Fax: (351) 21 4408999 http://www.digitalis.pt
 */
package pt.digitalis.utils.ldap;

import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Properties;

import pt.digitalis.utils.common.collections.CaseInsensitiveHashMap;
import pt.digitalis.utils.config.ConfigurationsPreferencesImpl;
import pt.digitalis.utils.config.annotations.ConfigDefault;
import pt.digitalis.utils.config.annotations.ConfigID;
import pt.digitalis.utils.config.annotations.ConfigIgnore;
import pt.digitalis.utils.config.annotations.ConfigSectionID;

/**
 * LDAP configurations for LDAP server access.
 * 
 * @author Rodrigo Gonalves <a href="mailto:rgoncalves@digitalis.pt">rgoncalves@digitalis.pt</a>
 * @author Luis Pinto <a href="lpinto@digitalis.pt">lpinto@digitalis.pt</a><br/>
 * @author Pedro Viegas <a href="mailto:pviegas@digitalis.pt">pviegas@digitalis.pt</a>
 * @created Mar 26, 2008
 */
@ConfigID("dif2")
@ConfigSectionID("LDAPConfigurations")
public class LDAPConfigurations {

    /** The LDAP configurations preferences path. */
    private static String LDAP_CONFIGURATIONS_PATH = "dif2/LDAPConfigurations";

    /** Revert Group Mappings. */
    private static Map<String, String> revertGroupMappings;

    /** The allow Distinguishedd name modifications. */
    private boolean allowDistinguishedNameModifications;

    /**
     * Maps the business-to-LDAP attributes.<K=business, V=LDAP>
     */
    private CaseInsensitiveHashMap<String> attributesMappings;

    /** The base DN for user search. */
    private String baseSearchDN;

    /** The base DN for user creation. */
    private String baseUserDN;

    /** The name of the attribute used to store bulk parameters. */
    private String bulkParametersAttributeName;

    /** The default group DN. */
    private String defaultGroupDN;

    /** The default profile DN. */
    private String defaultProfileDN;

    /** Exclusion parameters for the ldap. */
    private String exclusionCharaters;

    /** Force the connection to ldap to be secure */
    private Boolean forceSecureConnection;

    /** Maps between business defined groups and ldap repository. */
    private Map<String, String> groupMappings;

    /** The host name. */
    private String hostName;

    /** The loginname attribute on ldap */
    private String loginAttribute;

    /** The password. */
    private String password;

    /** The port. */
    private String port;

    /** the page size limit to be used in LDAP query's. */
    private int querysPageSizeLimit;

    /** The Read Only value. */
    private boolean readOnly;

    /** The SSL port. */
    private String sslPort;

    /** The logged user DN. */
    private String userDN;

    /**
     * Inspector for the 'allowDistinguishedNameModifications' attribute.
     * 
     * @return the allowDistinguishedNameModifications value
     */
    @ConfigDefault("false")
    public boolean getAllowDistinguishedNameModifications()
    {
        return allowDistinguishedNameModifications;
    }

    /**
     * Get the attributes names.
     * 
     * @return {@link List} with the attributes names
     */
    @ConfigIgnore
    public Map<String, String> getAttributesMapping()
    {
        if (this.attributesMappings == null)
        {
            this.attributesMappings = new CaseInsensitiveHashMap<String>();
            Properties properties = new ConfigurationsPreferencesImpl().readConfiguration(LDAP_CONFIGURATIONS_PATH,
                    "Mappings");

            Iterator<Object> iteratorProps = properties.keySet().iterator();

            while (iteratorProps.hasNext())
            {
                Object object = iteratorProps.next();
                this.attributesMappings.put(object.toString(), properties.getProperty(object.toString()));
            }
        }

        return this.attributesMappings;

    }

    /**
     * Inspector for the 'baseSearchDN' attribute.
     * 
     * @return the baseSearchDN value
     */
    @ConfigDefault("DC=ldap-server,DC=com")
    public String getBaseSearchDN()
    {
        return this.baseSearchDN;
    }

    /**
     * Getter for the 'baseUserDN' property. NOTE: Used for OID LDAP implementation only!!!
     * 
     * @return the baseUserDN
     */
    @ConfigDefault("CN=Users,DC=ldap-server,DC=com")
    public String getBaseUserDN()
    {
        return baseUserDN;
    }

    /**
     * Returns the name of the attribute used to store bulk parameters.
     * 
     * @return the name of the attribute used to store bulk parameters
     */
    @ConfigDefault("carLicense")
    public String getBulkParametersAttributeName()
    {
        return this.bulkParametersAttributeName;
    }

    /**
     * Getter for the 'defaultGroupDN' property.
     * 
     * @return the baseGroupName
     */
    @ConfigDefault("CN=Groups,DC=ldap-server,DC=com")
    public String getDefaultGroupDN()
    {
        return defaultGroupDN;
    }

    /**
     * Inspector for the 'defaultProfileDN' attribute.
     * 
     * @return the defaultProfileDN value
     */
    @ConfigDefault("CN=Users,DC=ldap-server,DC=com")
    public String getDefaultProfileDN()
    {
        return this.defaultProfileDN;
    }

    /**
     * Inspector for the 'exclusionParameters' attribute.
     * 
     * @return the exclusionParameters value
     */
    @ConfigDefault(",=")
    public String getExclusionCharaters()
    {
        return exclusionCharaters;
    }

    /**
     * Inspector for the 'forceSecureConnection' attribute.
     * 
     * @return the forceSecureConnection value
     */
    @ConfigDefault("false")
    public Boolean getForceSecureConnection()
    {
        return forceSecureConnection;
    }

    /**
     * Get the value for a specific group key.
     * 
     * @param groupName
     *            the group name
     * @return the group mapped value
     */
    @ConfigIgnore
    public String getGroupMapping(String groupName)
    {

        if (revertGroupMappings == null)
        {
            revertGroupMappings = new HashMap<String, String>();
            for (Entry<String, String> entrie: this.getGroupMappings().entrySet())
            {
                revertGroupMappings.put(entrie.getValue(), entrie.getKey());
            }
        }
        return revertGroupMappings.get(groupName);
    }

    /**
     * Get Group Mappings for dif2.
     * 
     * @return {@link List} with the group mappings
     */
    @ConfigIgnore
    public Map<String, String> getGroupMappings()
    {

        if (this.groupMappings == null)
        {
            this.groupMappings = new HashMap<String, String>();

            this.groupMappings = new ConfigurationsPreferencesImpl().readConfigurationAsMap(LDAP_CONFIGURATIONS_PATH,
                    "GroupMappings");
        }
        return groupMappings;
    }

    /**
     * Getter for the 'hostName' property.
     * 
     * @return the hostName
     */
    @ConfigDefault("ldap-server.com")
    public String getHostName()
    {
        return hostName;
    }

    /**
     * Inspector for the 'loginAttribute' attribute.
     * 
     * @return the loginAttribute value
     */
    @ConfigDefault("cn")
    public String getLoginAttribute()
    {
        return loginAttribute;
    }

    /**
     * Inspector for the password property.
     * 
     * @return the password
     */
    @ConfigDefault("password")
    public String getPassword()
    {
        return this.password;
    }

    /**
     * Inspector for the port property.
     * 
     * @return the port
     */
    @ConfigDefault("389")
    public String getPort()
    {
        return port;
    }

    /**
     * Inspector for the 'querysPageSizeLimit' attribute.
     * 
     * @return the querysPageSizeLimit value
     */
    @ConfigDefault("500")
    public int getQuerysPageSizeLimit()
    {
        return querysPageSizeLimit;
    }

    /**
     * Inspector for the 'readOnly' attribute.
     * 
     * @return the readOnly value
     */
    @ConfigDefault("false")
    public boolean getReadOnly()
    {
        return readOnly;
    }

    /**
     * Inspector for the SSL port property.
     * 
     * @return the SSL port
     */
    @ConfigDefault("636")
    public String getSSLPort()
    {
        return sslPort;
    }

    /**
     * Inspector for the logged user DN.
     * 
     * @return the userDN
     */
    @ConfigDefault("CN=Admin,CN=Users,DC=ldap-server,DC=com")
    public String getUserDN()
    {
        return userDN;
    }

    /**
     * Modifier for the 'allowDistinguishedNameModifications' attribute.
     * 
     * @param allowDistinguishedNameModifications
     *            the new allowDistinguishedNameModifications value to set
     */
    public void setAllowDistinguishedNameModifications(boolean allowDistinguishedNameModifications)
    {
        this.allowDistinguishedNameModifications = allowDistinguishedNameModifications;
    }

    /**
     * Modifier for the 'attributeMappings' attribute.
     * 
     * @param attributeMappings
     *            the new attributeMappings value to set
     */
    @ConfigIgnore
    public void setAttributeMappings(Map<String, String> attributeMappings)
    {
        if (this.attributesMappings == null)
            this.attributesMappings = new CaseInsensitiveHashMap<String>();
        else
        {
            this.attributesMappings.clear();
            this.attributesMappings.putAll(attributeMappings);
        }
    }

    /**
     * Modifier for the 'baseSearchDN' attribute.
     * 
     * @param baseSearchDN
     *            the new baseSearchDN value to set
     */
    public void setBaseSearchDN(String baseSearchDN)
    {
        this.baseSearchDN = baseSearchDN;
    }

    /**
     * Setter for the 'baseUserDN' property. NOTE: Used for OID LDAP implementation only!!!
     * 
     * @param baseUserDN
     *            the baseUserDN to set
     */
    public void setBaseUserDN(String baseUserDN)
    {
        this.baseUserDN = baseUserDN;
    }

    /**
     * Sets the name of the attribute used to store bulk parameters.
     * 
     * @param newBulkParametersAttributeName
     *            the new bulk parameters attribute name
     */
    public void setBulkParametersAttributeName(String newBulkParametersAttributeName)
    {
        this.bulkParametersAttributeName = newBulkParametersAttributeName;
    }

    /**
     * Setter for the 'defaultGroupDN' property.
     * 
     * @param newDefaultGroupDN
     *            the baseGroupName to set
     */
    public void setDefaultGroupDN(String newDefaultGroupDN)
    {
        this.defaultGroupDN = newDefaultGroupDN;
    }

    /**
     * Modifier for the 'defaultProfileDN' attribute.
     * 
     * @param defaultProfileDN
     *            the new defaultProfileDN value to set
     */
    public void setDefaultProfileDN(String defaultProfileDN)
    {
        this.defaultProfileDN = defaultProfileDN;
    }

    /**
     * Modifier for the 'exclusionParameters' attribute.
     * 
     * @param exclusionCharaters
     *            the excluded charaters
     */
    public void setExclusionCharaters(String exclusionCharaters)
    {
        this.exclusionCharaters = exclusionCharaters;
    }

    /**
     * Modifier for the 'forceSecureConnection' attribute.
     * 
     * @param forceSecureConnection
     *            the new forceSecureConnection value to set
     */
    public void setForceSecureConnection(Boolean forceSecureConnection)
    {
        this.forceSecureConnection = forceSecureConnection;
    }

    /**
     * Modifier for the 'groupMappings' attribute.
     * 
     * @param groupMappings
     *            the new groupMappings value to set
     */
    @ConfigIgnore
    public void setGroupMappings(Map<String, String> groupMappings)
    {
        this.groupMappings = groupMappings;

        Properties properties = new Properties();
        properties.putAll(this.groupMappings);

        new ConfigurationsPreferencesImpl().writeConfiguration(LDAP_CONFIGURATIONS_PATH, "GroupMappings", properties);
    }

    /**
     * Setter for the 'baseGroupDN' property.
     * 
     * @param hostName
     *            the hostName to set
     */
    public void setHostName(String hostName)
    {
        this.hostName = hostName;
    }

    /**
     * Modifier for the 'loginAttribute' attribute.
     * 
     * @param loginAttribute
     *            the new loginAttribute value to set
     */
    public void setLoginAttribute(String loginAttribute)
    {
        this.loginAttribute = loginAttribute;
    }

    /**
     * Inspector for the password property.
     * 
     * @param password
     *            the new password value to set
     */
    public void setPassword(String password)
    {
        this.password = password;
    }

    /**
     * Modifier for the port property.
     * 
     * @param port
     *            the new port value to set
     */
    public void setPort(String port)
    {
        this.port = port;
    }

    /**
     * Modifier for the 'querysPageSizeLimit' attribute.
     * 
     * @param querysPageSizeLimit
     *            the new querysPageSizeLimit value to set
     */
    public void setQuerysPageSizeLimit(int querysPageSizeLimit)
    {
        this.querysPageSizeLimit = querysPageSizeLimit;
    }

    /**
     * Modifier for the 'readOnly' attribute.
     * 
     * @param readOnly
     *            the new readOnly value to set
     */
    public void setReadOnly(boolean readOnly)
    {
        this.readOnly = readOnly;
    }

    /**
     * Modifier for the SSL port property.
     * 
     * @param sslPort
     *            the new SSL port value to set
     */
    public void setSSLPort(String sslPort)
    {
        this.sslPort = sslPort;
    }

    /**
     * Modifier for the logged user DN.
     * 
     * @param userDN
     *            the userDN to set
     */
    public void setUserDN(String userDN)
    {
        this.userDN = userDN;
    }

}
