/**
 * 2007, Digitalis Informatica. All rights reserved. Distribuicao e Gestao de Informatica, Lda. Estrada de Paco de Arcos
 * num.9 - Piso -1 2780-666 Paco de Arcos Telefone: (351) 21 4408990 Fax: (351) 21 4408999 http://www.digitalis.pt
 */
package pt.digitalis.utils.ldap.impl.openldap;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;

import javax.naming.NamingException;
import javax.naming.directory.Attributes;
import javax.naming.directory.BasicAttribute;
import javax.naming.directory.BasicAttributes;
import javax.naming.directory.ModificationItem;
import javax.naming.ldap.LdapContext;

import pt.digitalis.utils.common.StringUtils;
import pt.digitalis.utils.ldap.LDAPGroup;
import pt.digitalis.utils.ldap.LDAPUser;
import pt.digitalis.utils.ldap.exception.LDAPOperationException;
import pt.digitalis.utils.ldap.exception.LDAPOperationReadOnlyException;
import pt.digitalis.utils.ldap.impl.AbstractLDAPUtils;

/**
 * LDAP Utils implementation for OpenLDAP).
 * 
 * @author Rodrigo Gonalves <a href="mailto:rgoncalves@digitalis.pt">rgoncalves@digitalis.pt</a><br/>
 * @created Jul 02, 2008
 */
public class LDAPUtilsOpenLDAPImpl extends AbstractLDAPUtils {

    /** the unchangeable attributes */
    final static List<String> unchangeableAttributes = new ArrayList<String>();

    /**
     * For the AD implementation an entity's DN is composed of the common name and the parent group's DN, excluding the
     * groups CN.<br>
     * <br>
     * Example:<br>
     * <br>
     * <br>
     * - User login name: 'user1' <br>
     * <br>
     * - Main group's DN: cn=group1,ou=group1,dc=dc1,dc=dc2 <br>
     * <br>
     * - User's DN: cn=user1,ou=group1,dc=dc1,dc=dc2 <br>
     * 
     * @see pt.digitalis.utils.ldap.impl.AbstractLDAPUtils#calculateDistinguishedName(java.lang.String,
     *      java.lang.String)
     */
    @Override
    protected String calculateDistinguishedName(String commonName, String mainGroupDN) throws LDAPOperationException
    {

        if (commonName == null)
            throw new LDAPOperationException(
                    "The supplied CN was null! Cannot calculate the entity's DN without a valid CN...");
        if (mainGroupDN == null)
        {
            throw new LDAPOperationException(
                    "The supplied parent group name was null!! Cannot calculate the entity's DN without a valid parent group name...");
        }

        // Create user CN part
        StringBuffer newDistinguishedName = new StringBuffer(AbstractLDAPUtils.CN_TAG + commonName);

        // Get group DN
        StringBuffer groupDistinguishedName = new StringBuffer(mainGroupDN);

        // Discard group's CN
        groupDistinguishedName.trimToSize();
        groupDistinguishedName.replace(0, groupDistinguishedName.capacity(),
                groupDistinguishedName.substring(groupDistinguishedName.indexOf(",")));

        // Append the group's DN to user's DN
        newDistinguishedName.append(groupDistinguishedName);

        // Return result
        return newDistinguishedName.toString();
    }

    /**
     * @see pt.digitalis.utils.ldap.ILDAPUtils#changePassword(java.lang.String, java.lang.String)
     */
    @Override
    public void changePassword(String loginName, String newPassword) throws LDAPOperationException
    {
        if (this.isReadOnly())
            throw new LDAPOperationReadOnlyException();

        LDAPUser user = findUserByLogin(loginName);

        if (user != null)
        {

            ArrayList<ModificationItem> itens = new ArrayList<ModificationItem>();
            itens.add(new ModificationItem(LdapContext.REPLACE_ATTRIBUTE, new BasicAttribute(
                    getPasswordAttributeName(), newPassword)));

            if (itens.size() > 0)
            {
                ModificationItem mods[] = new ModificationItem[itens.size()];
                for (int i = 0; i < itens.size(); i++)
                {
                    mods[i] = itens.get(i);
                }

                LdapContext ctx = getLDAPContext();
                try
                {
                    ctx.modifyAttributes(user.getDistinguishedName(), mods);
                }
                catch (NamingException namingException)
                {
                    throw new LDAPOperationException("Could not change password for user with DN="
                            + user.getDistinguishedName() + "!", namingException);
                }
                finally
                {
                    try
                    {
                        ctx.close();
                    }
                    catch (NamingException e)
                    {
                        throw new LDAPOperationException("Error closing NamingEnumeration!", e);
                    }
                }
            }
        }
        else
            throw new LDAPOperationException("User with login=" + loginName
                    + " was not found on the LDAP server! Can't change the password on an nonexistent user...");
    }

    /**
     * @see pt.digitalis.utils.ldap.impl.AbstractLDAPUtils#getAttributesForGroupAddition(pt.digitalis.utils.ldap.LDAPGroup)
     */
    @Override
    protected Attributes getAttributesForGroupAddition(LDAPGroup newGroup) throws LDAPOperationException
    {
        Attributes attrs = new BasicAttributes(true);

        attrs.put(getObjectClassName(), getGroupClassName());

        // Populate 'member' attribute -> must exist but will be empty upon creation
        attrs.put(getGroupAttributeName(), "");

        if (newGroup.getDescription() != null)
            attrs.put(getDescriptionAttributeName(), newGroup.getDescription());

        if (newGroup.getParentGroupDN() != null)
        {
            attrs.put(getGroupParentGroupAttributeName(), newGroup.getParentGroupDN());
        }

        return attrs;
    }

    /**
     * @see pt.digitalis.utils.ldap.impl.AbstractLDAPUtils#getAttributesForUserAddition(pt.digitalis.utils.ldap.LDAPUser)
     */
    @Override
    protected Attributes getAttributesForUserAddition(LDAPUser newUser) throws LDAPOperationException
    {

        Attributes attributes = super.getAttributesForUserAddition(newUser);
        attributes.put(getSurnameAttributeName(), newUser.getGivenName());
        attributes.remove(getNameAttributeName());

        return attributes;
    }

    /**
     * @see pt.digitalis.utils.ldap.impl.AbstractLDAPUtils#getGroupClassName()
     */
    @Override
    protected String getGroupClassName()
    {
        return "groupOfNames";
    }

    /**
     * The LDAP attribute 'secretary' was chosen to store the group's parent group information since there is no
     * attribute on the so-called LDAP standard for the this information, and such information is needed for this
     * implementation.
     * 
     * @see pt.digitalis.utils.ldap.impl.AbstractLDAPUtils#getGroupParentGroupAttributeName()
     */
    @Override
    public String getGroupParentGroupAttributeName()
    {
        return "seeAlso";
    }

    /**
     * @see pt.digitalis.utils.ldap.impl.AbstractLDAPUtils#getNameAttributeName()
     */
    @Override
    public String getNameAttributeName()
    {
        return "cn";
    }

    /**
     * @see pt.digitalis.utils.ldap.ILDAPUtils#getUnchangeableLDAPAttributes()
     */
    public List<String> getUnchangeableLDAPAttributes()
    {
        return unchangeableAttributes;
    }

    /**
     * @see pt.digitalis.utils.ldap.impl.AbstractLDAPUtils#getUserClassName()
     */
    @Override
    protected String getUserClassName()
    {
        return "inetOrgPerson";
    }

    /**
     * @see pt.digitalis.utils.ldap.impl.AbstractLDAPUtils#getUserLoginAttributeName()
     */
    @Override
    public String getUserLoginAttributeName()
    {
        return StringUtils.isEmpty(getConfigurations().getLoginAttribute()) ? "cn" : ldapConfigurations
                .getLoginAttribute();
    }

    /**
     * @see pt.digitalis.utils.ldap.ILDAPUtils#updateUser(pt.digitalis.utils.ldap.LDAPUser, java.lang.String)
     */
    @Override
    public void updateUser(LDAPUser userToUpdate, String userLogin) throws LDAPOperationException
    {

        if (this.isReadOnly())
            throw new LDAPOperationReadOnlyException();

        // Retrieve old user data to get the old data
        LDAPUser existingUser = findUserByLogin(userLogin, false);

        if (existingUser != null)
        {
            ArrayList<ModificationItem> items = new ArrayList<ModificationItem>();

            // 'displayName'
            if (userToUpdate.getDisplayName() != null
                    && !userToUpdate.getDisplayName().equalsIgnoreCase(existingUser.getDisplayName()))
            {
                items.add(new ModificationItem(LdapContext.REPLACE_ATTRIBUTE, new BasicAttribute(
                        getDisplayNameAttributeName(), userToUpdate.getDisplayName())));
            }

            // 'e-mail'
            if (userToUpdate.getEmail() != null && !userToUpdate.getEmail().equals(existingUser.getEmail()))
                items.add(new ModificationItem(LdapContext.REPLACE_ATTRIBUTE, new BasicAttribute(
                        getMailAttributeName(), userToUpdate.getEmail())));
            // 'givenName'
            if (userToUpdate.getGivenName() != null && !userToUpdate.getGivenName().equals(existingUser.getGivenName()))
                items.add(new ModificationItem(LdapContext.REPLACE_ATTRIBUTE, new BasicAttribute(
                        getGivenNameAttributeName(), userToUpdate.getGivenName())));

            /*
             * Password change method is implementation dependent so it's not done through ModificationItem. Password
             * does not come from LDAP server, so it's ALWAYS changed!!! Done before login change to assure that the
             * former login name still works.
             */
            if (userToUpdate.getPassword() != null)
                changePassword(userLogin, userToUpdate.getPassword());

            // '"mainGroup"'
            if (userToUpdate.getParentGroupDN() != null
                    && !userToUpdate.getParentGroupDN().equals(existingUser.getParentGroupDN()))
                items.add(new ModificationItem(LdapContext.REPLACE_ATTRIBUTE, new BasicAttribute(
                        getUserParentGroupAttributeName(), userToUpdate.getParentGroupDN())));

            // Parameters
            Map<String, String> bulkParameters = new HashMap<String, String>();
            for (String parameterName: userToUpdate.getParameters().keySet())
            {
                if (getConfigurations().getAttributesMapping().containsKey(parameterName))
                {
                    items.add(new ModificationItem(LdapContext.REPLACE_ATTRIBUTE, new BasicAttribute(
                            getConfigurations().getAttributesMapping().get(parameterName), userToUpdate
                                    .getParameter(parameterName))));
                }
                else
                {
                    if (existingUser.getParameter(parameterName) != null)
                    {
                        if (!getUnchangeableLDAPAttributes().contains(parameterName.toUpperCase())
                                && !userToUpdate.getParameter(parameterName).equalsIgnoreCase(
                                        existingUser.getParameter(parameterName)))
                            items.add(new ModificationItem(LdapContext.REPLACE_ATTRIBUTE, new BasicAttribute(
                                    parameterName, userToUpdate.getParameter(parameterName))));
                    }
                    else
                    {
                        bulkParameters.put(parameterName, userToUpdate.getParameter(parameterName));
                    }
                }
            }

            for (String parameterName: userToUpdate.getParametersToRemove())
            {
                if (getConfigurations().getAttributesMapping().containsKey(parameterName))
                {
                    items.add(new ModificationItem(LdapContext.REMOVE_ATTRIBUTE, new BasicAttribute(getConfigurations()
                            .getAttributesMapping().get(parameterName))));
                }
                else
                {
                    bulkParameters.remove(parameterName);
                }
            }

            // The value of the Bulk parameters to commit
            StringBuilder bulkParameterAttributeValue = new StringBuilder();
            for (Entry<String, String> entry: bulkParameters.entrySet())
            {
                bulkParameterAttributeValue.append(entry.getKey() + "=" + entry.getValue() + ";");
            }

            if (bulkParameterAttributeValue.length() == 0)
            {
                /* LDAP doesn't like empty strings on REPLACE_ATTRIBUTE */
                bulkParameterAttributeValue.append(" ");
            }

            items.add(new ModificationItem(LdapContext.REPLACE_ATTRIBUTE, new BasicAttribute(getConfigurations()
                    .getBulkParametersAttributeName(), bulkParameterAttributeValue.toString())));

            // Persist changes
            if (items.size() > 0)
            {
                ModificationItem mods[] = new ModificationItem[items.size()];
                for (int i = 0; i < items.size(); i++)
                {
                    mods[i] = items.get(i);
                }

                modifyAttributes(existingUser.getDistinguishedName(), mods, false);
            }

            /* Group && CN/login modification must be made after all other attributes since it can cause DN changes. */

            if (ldapConfigurations.getAllowDistinguishedNameModifications()
                    && userToUpdate.getParentGroupDN() != null
                    && !userToUpdate.getParentGroupDN().equals(existingUser.getParentGroupDN())
                    || ((userToUpdate.getCommonName() != null && !userToUpdate.getCommonName().equals(
                            existingUser.getCommonName())) || ((userToUpdate.getLoginName() != null && !userToUpdate
                            .getLoginName().equals(existingUser.getLoginName())))))
            {

                // Get login name from user to change
                String loginName = userToUpdate.getLoginName();

                // If only the group changes, get login from existing user data
                if (loginName == null || NON_AVAILABLE.equals(loginName))
                    loginName = existingUser.getLoginName();

                // Get parent group DN from user to change
                String parentGroupDN = userToUpdate.getParentGroupDN();

                // If the group is not defined, get it from existing user
                if (parentGroupDN == null || NON_AVAILABLE.equals(parentGroupDN))
                    parentGroupDN = existingUser.getParentGroupDN();

                String newDN = calculateDistinguishedName(loginName, parentGroupDN);

                try
                {
                    if (!existingUser.getDistinguishedName().equals(newDN))
                    {
                        LdapContext ctx = getLDAPContext();
                        try
                        {
                            ctx.rename(existingUser.getDistinguishedName(), newDN);
                        }
                        finally
                        {
                            ctx.close();
                        }
                    }
                }
                catch (LDAPOperationException ldapOperationException)
                {
                    throw ldapOperationException;
                }
                catch (NamingException namingException)
                {
                    throw new LDAPOperationException("Could not update user's main group...", namingException);
                }
            }
        }
    }
}
