/**
 * 2007, Digitalis Informatica. All rights reserved. Distribuicao e Gestao de Informatica, Lda. Estrada de Paco de Arcos
 * num.9 - Piso -1 2780-666 Paco de Arcos Telefone: (351) 21 4408990 Fax: (351) 21 4408999 http://www.digitalis.pt
 */
package pt.digitalis.utils.ldap.impl.ad;

import java.io.UnsupportedEncodingException;
import java.util.ArrayList;
import java.util.List;

import javax.naming.directory.Attributes;
import javax.naming.directory.BasicAttribute;
import javax.naming.directory.DirContext;
import javax.naming.directory.ModificationItem;

import pt.digitalis.utils.config.ConfigurationException;
import pt.digitalis.utils.ldap.LDAPUser;
import pt.digitalis.utils.ldap.exception.LDAPOperationException;
import pt.digitalis.utils.ldap.exception.LDAPOperationReadOnlyException;
import pt.digitalis.utils.ldap.impl.AbstractLDAPUtils;

/**
 * LDAP Utils implementation for Active Directory (AD).
 *
 * @author Rodrigo Gonalves <a href="mailto:rgoncalves@digitalis.pt">rgoncalves@digitalis.pt</a><br/>
 * @author Luis Pinto <a href="lpinto@digitalis.pt">lpinto@digitalis.pt</a><br/>
 * @created Apr 01, 2008
 */
@SuppressWarnings("unused")
public class LDAPUtilsActiveDirectoryImpl extends AbstractLDAPUtils {

    /** AD's "account disabled" account option. */
    final static private int UF_ACCOUNTDISABLE = 0x0002;

    /** AD's "account enabled" account option. */
    final static private int UF_ACCOUNTENABLE = 0x0001;

    /** AD's "password don't expire" account option. */
    final static private int UF_DONT_EXPIRE_PASSWD = 0x10000;

    /** AD's "normal account" account option. */
    final static private int UF_NORMAL_ACCOUNT = 0x0200;

    /** AD's "password can't change" account option. */
    final static private int UF_PASSWD_CANT_CHANGE = 0x0040;

    /** AD's "password not required" account option. */
    final static private int UF_PASSWD_NOTREQD = 0x0020;

    /** AD's "password expired" account option. */
    final static private int UF_PASSWORD_EXPIRED = 0x800000;

    /** the unchangeable attributes */
    final static List<String> unchangeableAttributes = new ArrayList<String>();

    /** AD's "user control value" attribute value. */
    final static private String USER_CONTROL_VALUE = Integer
            .toString(UF_NORMAL_ACCOUNT + UF_PASSWD_NOTREQD + UF_DONT_EXPIRE_PASSWD + UF_ACCOUNTENABLE);

    static
    {
        unchangeableAttributes.add("USNCHANGED");
        unchangeableAttributes.add("WHENCHANGED");

    }

    /**
     * For the AD implementation an entity's DN is composed of the common name and the parent group's DN, excluding the
     * groups CN.<br>
     * <br>
     * Example:<br>
     * <br>
     * <br>
     * - User login name: 'user1' <br>
     * <br>
     * - Main group's DN: cn=group1,ou=group1,dc=dc1,dc=dc2 <br>
     * <br>
     * - User's DN: cn=user1,ou=group1,dc=dc1,dc=dc2 <br>
     *
     * @see pt.digitalis.utils.ldap.impl.AbstractLDAPUtils#calculateDistinguishedName(java.lang.String,
     *      java.lang.String)
     */
    @Override
    protected String calculateDistinguishedName(String commonName, String mainGroupDN) throws LDAPOperationException
    {

        if (commonName == null)
            throw new LDAPOperationException(
                    "The supplied CN was null! Cannot calculate the entity's DN without a valid CN...");
        if (mainGroupDN == null)
        {
            throw new LDAPOperationException(
                    "The supplied parent group name was null!! Cannot calculate the entity's DN without a valid parent group name...");
        }

        // Create user CN part
        StringBuffer newDistinguishedName = new StringBuffer(AbstractLDAPUtils.CN_TAG + commonName);

        // Get group DN
        StringBuffer groupDistinguishedName = new StringBuffer(mainGroupDN);

        // Discard group's CN
        groupDistinguishedName.trimToSize();
        groupDistinguishedName.replace(0, groupDistinguishedName.capacity(),
                groupDistinguishedName.substring(groupDistinguishedName.indexOf(",")));

        // Append the group's DN to user's DN
        newDistinguishedName.append(groupDistinguishedName);

        // Return result
        return newDistinguishedName.toString();
    }

    /**
     * @see pt.digitalis.utils.ldap.ILDAPUtils#changePassword(java.lang.String, java.lang.String)
     */
    @Override
    public void changePassword(String loginName, String newPassword) throws LDAPOperationException
    {
        if (this.isReadOnly())
            throw new LDAPOperationReadOnlyException();

        // Define encoding to use for password
        final String ENCODING = "UTF-16LE";

        // Setting the password for the user
        try
        {
            // Quote the password
            String newQuotedPassword = "\"" + newPassword + "\"";
            // Set the encoding
            byte[] newUnicodePassword = newQuotedPassword.getBytes(ENCODING);

            // Define the attribute to change
            // ModificationItem[] mods = new ModificationItem[2];
            ModificationItem[] mods = new ModificationItem[1];
            mods[0] = new ModificationItem(DirContext.REPLACE_ATTRIBUTE,
                    new BasicAttribute("unicodePwd", newUnicodePassword));
            // mods[1] = new ModificationItem(DirContext.REPLACE_ATTRIBUTE, new BasicAttribute("userAccountControl",
            // Integer.toString(UF_NORMAL_ACCOUNT + UF_PASSWORD_EXPIRED)));

            super.modifyAttributes(findUserByLogin(loginName).getDistinguishedName(), mods, true);

        }
        catch (UnsupportedEncodingException unsupportedEncodingException)
        {
            throw new LDAPOperationException(ENCODING + " encoding not suppported!", unsupportedEncodingException);
        }
    }

    /**
     * @see pt.digitalis.utils.ldap.impl.AbstractLDAPUtils#getAttributesForUserAddition(pt.digitalis.utils.ldap.LDAPUser)
     */
    @Override
    protected Attributes getAttributesForUserAddition(LDAPUser newUser) throws LDAPOperationException
    {

        Attributes attrs = super.getAttributesForUserAddition(newUser);
        attrs.put(getUserControlAttributeName(), LDAPUtilsActiveDirectoryImpl.USER_CONTROL_VALUE);

        String result = "";
        String baseDn;

        try
        {
            baseDn = getConfigurations().getBaseSearchDN();
        }
        catch (ConfigurationException e)
        {
            throw new LDAPOperationException("Could not read configuration!", e);
        }

        String[] baseDnArray = baseDn.split(",");

        for (String content: baseDnArray)
        {
            if (content.toUpperCase().startsWith("DC"))
            {
                if ("".equals(result))
                {
                    result = content.split("=")[1];
                }
                else
                {
                    result += "." + content.split("=")[1];
                }
            }
        }

        if ("".equals(result))
        {
            result = newUser.getLoginName();
        }
        else
        {
            result = newUser.getLoginName() + "@" + result;
        }

        /*
         * This attribute contains the UPN that is an Internet-style login name for a user based on the Internet
         * standard RFC 822. The UPN is shorter than the distinguished name and easier to remember. By convention, this
         * should map to the user e-mail name. The value set for this attribute is equal to the length of the user's ID
         * and the domain name. Source: http://msdn.microsoft.com/en-us/library/ms680857%28VS.85%29.aspx
         */
        attrs.put("userPrincipalName", result);

        return attrs;
    }

    /**
     * @see pt.digitalis.utils.ldap.impl.AbstractLDAPUtils#getGroupClassName()
     */
    @Override
    protected String getGroupClassName()
    {
        return "group";
    }

    /**
     * The LDAP attribute 'desktopProfile' was chosen to store the group's parent group information since there is no
     * attribute on the so-called LDAP standard for the this information, and such information is needed for this
     * implementation. VALIDATE: Luis: is this the appropriate attribute?
     *
     * @see pt.digitalis.utils.ldap.impl.AbstractLDAPUtils#getGroupParentGroupAttributeName()
     */
    @Override
    public String getGroupParentGroupAttributeName()
    {
        return "desktopProfile";
    }

    /**
     * @see pt.digitalis.utils.ldap.impl.AbstractLDAPUtils#getMailAttributeName()
     */
    @Override
    public String getMailAttributeName()
    {
        return "mail";
    }

    /**
     * @see pt.digitalis.utils.ldap.impl.AbstractLDAPUtils#getNameAttributeName()
     */
    @Override
    public String getNameAttributeName()
    {
        return "sAMAccountName";
    }

    /**
     * @see pt.digitalis.utils.ldap.ILDAPUtils#getUnchangeableLDAPAttributes()
     */
    public List<String> getUnchangeableLDAPAttributes()
    {
        return unchangeableAttributes;
    }

    /**
     * @see pt.digitalis.utils.ldap.impl.AbstractLDAPUtils#getUserClassName()
     */
    @Override
    protected String getUserClassName()
    {
        return "user";
    }

    /**
     * Returns the standard LDAP name for the 'user account control' attribute. To be overridden on the different LDAP
     * implementations. This method must NOT be qualified with 'final' and/or 'static' to preserve polymorphic behavior.
     *
     * @return the standard LDAP name for the 'user account control' attribute
     */
    protected String getUserControlAttributeName()
    {
        return "userAccountControl";
    }

    /**
     * @see pt.digitalis.utils.ldap.impl.AbstractLDAPUtils#getUserLoginAttributeName()
     */
    @Override
    public String getUserLoginAttributeName()
    {
        return "sAMAccountName";
    }
}
