/**
 * 2013, Digitalis Informatica. All rights reserved. Distribuicao e Gestao de Informatica, Lda. Estrada de Paco de Arcos
 * num.9 - Piso -1 2780-666 Paco de Arcos Telefone: (351) 21 4408990 Fax: (351) 21 4408999 http://www.digitalis.pt
 */

package pt.digitalis.utils.checksum;

import pt.digitalis.utils.common.StringUtils;

import java.io.IOException;
import java.io.InputStream;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;

/**
 * @author Pedro Viegas <a href="mailto:pviegas@digitalis.pt">pviegas@digitalis.pt</a><br/>
 * @created 23 de Mai de 2013
 */
public class SHACheckSumGenerator implements ICheckSumGenerator
{

    /**
     *
     */
    private HashFunction hashFunction;

    /**
     * Default constructor
     *
     * @param hashFunction
     */
    public SHACheckSumGenerator(HashFunction hashFunction)
    {
        this.hashFunction = hashFunction;
    }

    /**
     * @see pt.digitalis.utils.checksum.ICheckSumGenerator#getCheckSumHash(byte[])
     */
    public String getCheckSumHash(byte[] contentToEncrypt) throws CheckSumException
    {
        try
        {
            return new String(this.getDigest().digest(contentToEncrypt));
        }
        catch (NoSuchAlgorithmException e)
        {
            throw new CheckSumException(e);
        }
    }

    /**
     * @see pt.digitalis.utils.checksum.ICheckSumGenerator#getCheckSumHash(java.io.InputStream)
     */
    public String getCheckSumHash(InputStream contentToEncrypt) throws CheckSumException
    {
        try
        {
            MessageDigest md = this.getDigest();

            byte[] data = new byte[1024];
            int read = 0;
            while ((read = contentToEncrypt.read(data)) != -1)
            {
                md.update(data, 0, read);
            }

            return StringUtils.convertToHEXRepresentation(md.digest());
        }
        catch (IOException e)
        {
            throw new CheckSumException(e);
        }
        catch (NoSuchAlgorithmException e)
        {
            throw new CheckSumException(e);
        }
    }

    /**
     * @see pt.digitalis.utils.checksum.ICheckSumGenerator#getCheckSumHash(java.lang.String)
     */
    public String getCheckSumHash(String contentToEncrypt) throws CheckSumException
    {
        return this.getCheckSumHash(contentToEncrypt.getBytes());
    }

    /**
     * @return the {@link MessageDigest} instance for the current generator
     *
     * @exception NoSuchAlgorithmException
     */
    private MessageDigest getDigest() throws NoSuchAlgorithmException
    {
        return MessageDigest.getInstance(this.hashFunction.getIdentifierString());
    }

    /**
     * The hash function to use for hash string creation
     *
     * @author Pedro Viegas <a href="mailto:pviegas@digitalis.pt">pviegas@digitalis.pt</a><br/>
     * @created 23 de Mai de 2013
     */
    public enum HashFunction
    {
        /**
         *
         */
        MD5,
        /**
         *
         */
        SHA1,
        /**
         *
         */
        SHA256,
        /**
         *
         */
        SHA384,
        /**
         *
         */
        SHA512;

        /**
         * @return the string that represents the keyword do use in the Java CheckSum API
         */
        public String getIdentifierString()
        {
            if (this == MD5)
                return "MD5";
            else if (this == SHA256)
                return "SHA-256";
            else if (this == SHA384)
                return "SHA-384";
            else if (this == SHA512)
                return "SHA-512";
            else
                return "SHA-1";
        }
    }
}
