package pt.digitalis.web.ancillaries;

import static org.easymock.EasyMock.createMock;
import static org.easymock.EasyMock.createNiceMock;
import static org.easymock.EasyMock.expect;
import static org.easymock.EasyMock.replay;

import java.util.Locale;
import java.util.Map;
import java.util.Vector;

import javax.servlet.RequestDispatcher;
import javax.servlet.ServletConfig;
import javax.servlet.ServletContext;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.servlet.http.HttpSession;

/**
 * Defines constants and helper methods for the Web unit tests. TODO: Move this to a new project inside digi-utils.
 * Testing-utils. All needing projects will declare this necessity and the dif-core obviously will not need to. We
 * should create more usable functions. Like, execute a servlet/jsp or taglib... and it knows how to do it. If not we
 * will end up with a lot of copy-paste code in the tests.
 * 
 * @author Rodrigo Gonalves <a href="mailto:rgoncalves@digitalis.pt">rgoncalves@digitalis.pt</a><br/>
 * @created 2007/12/21
 */
final public class WebAncillaries {

    /** Accepted languages header. */
    final static public String ACCEPT_LANGUAGE_HEADER = "en-gb,en;q=0.5";

    /** Accessible Web UI Mode. */
    public static final String ACCESSIBLE_WEB_UI_MODE = "access";

    /** A locale */
    final static public Locale EN_LOCALE = new Locale(WebConstants.LANG_EN_GB, WebConstants.EMPTY_STRING);

    /** The F boolean value as a String */
    final static public String FALSE = Boolean.FALSE.toString().toLowerCase();

    /** Firefox main version. */
    final static public String FIREFOX_MAIN_VERSION = "2";

    /** Firefox minor version. */
    final static public String FIREFOX_MINOR_VERSION = "0.0.11";

    /** A Firefox user agent header info. */
    final static public String FIREFOX_USER_AGENT_HEADER = "Mozilla/5.0 Mac Gecko/20071127 Firefox/"
            + FIREFOX_MAIN_VERSION + "." + FIREFOX_MINOR_VERSION;

    /** The IP forward header. */
    final static public String FORWARDED_IP = "127.0.0.1";

    /** The id of the stage to execute. */
    final static public String HOMEPAGE_STAGE_ID = "homestage";

    /** The false id of a non-existent stage to execute. */
    final static public String INEXISTENT_MOCKUP_STAGE_ID = "inexistentmockupteststage";

    /** Accepted languages header for locale evaluation */
    final static public String LOCALE_EVALUATION_ACCEPT_LANGUAGE_HEADER = WebConstants.LANG_EN_GB;

    /** Mac OS tag. */
    final static public String MAC_OS = "Mac - Mac_PowerPC ";

    /** Unknown main version. */
    final static public String MAIN_VERSION_UNKNOWN = "666";

    /** The HTTP 'GET' method. */
    final static public String METHOD_GET = "GET";

    /** Unknown minor version. */
    final static public String MINOR_VERSION_UNKNOWN = "666";

    /** Mozilla main version. */
    final static public String MOZILLA_MAIN_VERSION = "5";

    /** Mozilla minor version. */
    final static public String MOZILLA_MINOR_VERSION = "0";

    /** A Mozilla user agent header info. */
    final static public String MOZILLA_USER_AGENT_HEADER = "Mozilla/" + MOZILLA_MAIN_VERSION + "."
            + MOZILLA_MINOR_VERSION + " " + MAC_OS + " Gecko/20071127 Firefox/2.0.0.11";

    /** MSIE main version. */
    final static public String MSIE_MAIN_VERSION = "7";

    /** MSIE minor version. */
    final static public String MSIE_MINOR_VERSION = "0b";

    /** A MSIE user agent header info. */
    final static public String MSIE_USER_AGENT_HEADER = "Mozilla/" + MSIE_MAIN_VERSION + "." + MSIE_MINOR_VERSION
            + " (compatible; MSIE " + MSIE_MAIN_VERSION + "." + MSIE_MINOR_VERSION + "; " + WebConstants.WIN98_NAME
            + ")";

    /** Opera main version. */
    final static public String OPERA_MAIN_VERSION = "9";

    /** Opera minor version. */
    final static public String OPERA_MINOR_VERSION = "5";

    /** An Opera user agent header info. */
    final static public String OPERA_USER_AGENT_HEADER = "Opera/" + OPERA_MAIN_VERSION + "." + OPERA_MINOR_VERSION
            + " (compatible; Opera " + OPERA_MAIN_VERSION + "." + OPERA_MINOR_VERSION + "; " + WebConstants.WINNT_NAME
            + ")";

    /** A session id. */
    final static public String SESSION_ID = "testSession";

    /** The T boolean value as a String */
    public static final String TRUE = Boolean.TRUE.toString().toLowerCase();

    /** Unknown Mac OS tag. */
    final static public String UNKNOWN_MAC_OS = "Other Mac";

    /** An unknown user agent header info running on an unknown Mac platform. */
    final static public String UNKNOWN_MAC_USER_AGENT_HEADER = "Unknown/" + MAIN_VERSION_UNKNOWN + "."
            + MINOR_VERSION_UNKNOWN + " (compatible; Unknown; " + UNKNOWN_MAC_OS + ")";

    /** An unknown user agent header info running on an unknown win16 platform. */
    final static public String UNKNOWN_MACINTOSH_USER_AGENT_HEADER = "Unknown/" + MAIN_VERSION_UNKNOWN + "."
            + MINOR_VERSION_UNKNOWN + " (compatible; Unknown; " + WebConstants.MAC_NAME + ")";

    /** Unknown OS tag. */
    final static public String UNKNOWN_OS = "unknown";

    /** An unknown user agent header info. */
    final static public String UNKNOWN_USER_AGENT_HEADER = "Unknown/" + MAIN_VERSION_UNKNOWN + "."
            + MINOR_VERSION_UNKNOWN + " (compatible; Unknown; " + UNKNOWN_OS + ")";

    /** An unknown user agent header info running on an unknown win16 platform. */
    final static public String UNKNOWN_WIN16_USER_AGENT_HEADER = "Unknown/" + MAIN_VERSION_UNKNOWN + "."
            + MINOR_VERSION_UNKNOWN + " (compatible; Unknown; " + WebConstants.WIN16_NAME + ")";

    /** An unknown user agent header info running on an unknown win16 platform. */
    final static public String UNKNOWN_WIN95_USER_AGENT_HEADER = "Unknown/" + MAIN_VERSION_UNKNOWN + "."
            + MINOR_VERSION_UNKNOWN + " (compatible; Unknown; " + WebConstants.WIN95_NAME + ")";

    /**
     * Creates an HttpServletRequest mock-up object.
     * 
     * @param parameters
     *            the parameter map
     * @param attributes
     *            the attribute map
     * @return the mock-up HttpServletRequest object
     */
    static public HttpServletRequest createHttpServletRequestMock(Map<String, String> parameters,
            Map<String, Object> attributes)
    {
        return createHttpServletRequestMock(parameters, attributes, null);
    }

    /**
     * Creates an HttpServletRequest mock-up object.
     * 
     * @param parameters
     *            the parameter map
     * @param attributes
     *            the attribute map
     * @param sessionAttributes
     *            the session attribute map
     * @return the mock-up HttpServletRequest object
     */
    static public HttpServletRequest createHttpServletRequestMock(Map<String, String> parameters,
            Map<String, Object> attributes, Map<String, Object> sessionAttributes)
    {
        HttpServletRequest request = createNiceMock(HttpServletRequest.class);

        // Add parameters
        if (parameters != null && parameters.size() > 0)
        {
            Vector<String> parameterNames = new Vector<String>();

            for (String parameterName: parameters.keySet())
            {
                expect(request.getParameter(parameterName)).andReturn(parameters.get(parameterName)).anyTimes();
                parameterNames.add(parameterName);
            }

            expect(request.getParameterNames()).andReturn(parameterNames.elements());
        }

        // Add attributes
        if (attributes != null && attributes.size() > 0)
        {
            Vector<String> attributeNames = new Vector<String>();

            for (String attributeName: attributes.keySet())
            {
                expect(request.getAttribute(attributeName)).andReturn(attributes.get(attributeName)).anyTimes();
                attributeNames.add(attributeName);
            }

            expect(request.getAttributeNames()).andReturn(attributeNames.elements());
        }

        // Create session
        expect(request.getSession()).andReturn(createHttpSessionMock(sessionAttributes)).anyTimes();

        // Set method
        expect(request.getMethod()).andReturn(METHOD_GET).anyTimes();

        // Set headers
        expect(request.getHeader(WebConstants.USER_AGENT_HEADER)).andReturn(WebAncillaries.MOZILLA_USER_AGENT_HEADER)
                .anyTimes();
        expect(request.getHeader(WebConstants.ACCEPT_LANGUAGE_HEADER)).andReturn(WebAncillaries.ACCEPT_LANGUAGE_HEADER)
                .anyTimes();

        replay(request);

        return request;
    }

    /**
     * Creates an HttpServletResponse mock-up object.
     * 
     * @return a mockup HttpServletResponse object
     */
    static public HttpServletResponse createHttpServletResponseMock()
    {
        HttpServletResponse response = createMock(HttpServletResponse.class);

        replay(response);

        return response;
    }

    /**
     * Creates an HttpSession mock-up object.
     * 
     * @param sessionAttributes
     *            the session attributes map
     * @return the mockup HttpSession object
     */
    static public HttpSession createHttpSessionMock(Map<String, Object> sessionAttributes)
    {
        HttpSession session = createMock(HttpSession.class);

        expect(session.getId()).andReturn(WebAncillaries.SESSION_ID);

        // Add attributes
        if (sessionAttributes != null && sessionAttributes.size() > 0)
        {
            Vector<String> attributeNames = new Vector<String>();

            for (String attributeName: sessionAttributes.keySet())
            {
                expect(session.getAttribute(attributeName)).andReturn(sessionAttributes.get(attributeName)).anyTimes();
                session.removeAttribute(attributeName);
                attributeNames.add(attributeName);
            }

            expect(session.getAttributeNames()).andReturn(attributeNames.elements());
        }

        replay(session);

        return session;
    }

    /**
     * Creates an RequestDispatcher mock-up object.
     * 
     * @return a mockup request dispatcher object
     */
    static public RequestDispatcher createRequestDispatcherMock()
    {
        RequestDispatcher dispatcher = createNiceMock(RequestDispatcher.class);
        replay(dispatcher);

        return dispatcher;
    }

    /**
     * Creates an ServletConfig mock-up object.
     * 
     * @param viewPath
     *            the view path
     * @return a mockup servlet config object
     */
    static public ServletConfig createServletConfigMock(String viewPath)
    {
        ServletConfig servletConfig = createNiceMock(ServletConfig.class);

        expect(servletConfig.getServletContext()).andReturn(createServletContextMock(viewPath)).anyTimes();

        replay(servletConfig);

        return servletConfig;
    }

    /**
     * Creates an ServletContext mock-up object.
     * 
     * @param viewPath
     *            the view path
     * @return a mockup servlet context object
     */
    static public ServletContext createServletContextMock(String viewPath)
    {
        ServletContext servletContext = createNiceMock(ServletContext.class);

        expect(servletContext.getRequestDispatcher(viewPath)).andReturn(createRequestDispatcherMock()).anyTimes();

        replay(servletContext);

        return servletContext;
    }
}
